// =============================================================================
// MonkAPI - Test Setup
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { INestApplication, ValidationPipe } from '@nestjs/common';
import { PrismaService } from '../src/shared/database/prisma.service';

// =============================================================================
// MOCK PRISMA SERVICE
// =============================================================================

export const mockPrismaService = {
  user: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
    count: jest.fn(),
  },
  driver: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
    count: jest.fn(),
  },
  booking: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
    count: jest.fn(),
    aggregate: jest.fn(),
  },
  merchant: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
  },
  vehicleType: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
    delete: jest.fn(),
  },
  promoCode: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
  },
  zone: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
  },
  setting: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    upsert: jest.fn(),
  },
  supportTicket: {
    findUnique: jest.fn(),
    findFirst: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    update: jest.fn(),
    count: jest.fn(),
  },
  notification: {
    findUnique: jest.fn(),
    findMany: jest.fn(),
    create: jest.fn(),
    createMany: jest.fn(),
    update: jest.fn(),
    updateMany: jest.fn(),
  },
  $transaction: jest.fn((callback) => callback(mockPrismaService)),
  $connect: jest.fn(),
  $disconnect: jest.fn(),
};

// =============================================================================
// MOCK CONFIG SERVICE
// =============================================================================

export const mockConfigService = {
  get: jest.fn((key: string) => {
    const config: Record<string, any> = {
      'app.name': 'MonkAPI Test',
      'app.env': 'test',
      'jwt.secret': 'test-secret-key',
      'jwt.expiresIn': '1h',
      'jwt.refreshExpiresIn': '7d',
      'database.url': 'mysql://test:test@localhost:3306/test',
      'redis.host': 'localhost',
      'redis.port': 6379,
    };
    return config[key];
  }),
};

// =============================================================================
// MOCK JWT SERVICE
// =============================================================================

export const mockJwtService = {
  sign: jest.fn(() => 'mock-jwt-token'),
  signAsync: jest.fn(() => Promise.resolve('mock-jwt-token')),
  verify: jest.fn(() => ({ sub: 1, merchantId: 1, type: 'user' })),
  verifyAsync: jest.fn(() => Promise.resolve({ sub: 1, merchantId: 1, type: 'user' })),
  decode: jest.fn(() => ({ sub: 1, merchantId: 1, type: 'user' })),
};

// =============================================================================
// MOCK CACHE SERVICE
// =============================================================================

export const mockCacheService = {
  get: jest.fn(),
  set: jest.fn(),
  del: jest.fn(),
  reset: jest.fn(),
  keys: jest.fn(() => []),
  ttl: jest.fn(() => -1),
};

// =============================================================================
// MOCK QUEUE SERVICE
// =============================================================================

export const mockQueueService = {
  add: jest.fn(),
  addBulk: jest.fn(),
  process: jest.fn(),
  getJob: jest.fn(),
  getJobs: jest.fn(() => []),
};

// =============================================================================
// TEST DATA FACTORIES
// =============================================================================

export const createMockUser = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  first_name: 'John',
  last_name: 'Doe',
  email: 'john.doe@example.com',
  phone: '+22890123456',
  password: '$2b$10$hashedpassword',
  status: 1,
  wallet_balance: 10000,
  rating: 4.5,
  total_rides: 25,
  referral_code: 'JOHN123',
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

export const createMockDriver = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  first_name: 'Driver',
  last_name: 'Test',
  email: 'driver@example.com',
  phone: '+22890654321',
  password: '$2b$10$hashedpassword',
  driver_status: 1, // approved
  is_online: 1,
  free_busy: 2, // free
  latitude: 6.1319,
  longitude: 1.2228,
  wallet_balance: 50000,
  rating: 4.8,
  total_rides: 150,
  commission_rate: 20,
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

export const createMockBooking = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  user_id: 1,
  driver_id: null,
  vehicle_type_id: 1,
  booking_number: 'BK-202601-00001',
  booking_type: 'ride',
  booking_status: 'pending',
  payment_status: 0,
  payment_method: 'cash',
  pickup_latitude: 6.1319,
  pickup_longitude: 1.2228,
  pickup_address: '123 Rue Commerce, Lome',
  drop_latitude: 6.1654,
  drop_longitude: 1.2544,
  drop_address: 'Aeroport de Lome',
  estimate_amount: 5000,
  final_amount: null,
  travel_distance: 8.5,
  travel_time: 20,
  surge_multiplier: 1.0,
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

export const createMockVehicleType = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  name: 'Economique',
  description: 'Voiture economique',
  capacity: 4,
  base_fare: 500,
  per_km_fare: 200,
  per_minute_fare: 25,
  minimum_fare: 1000,
  cancellation_fee: 500,
  waiting_fee: 20,
  is_active: true,
  sort_order: 1,
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

export const createMockPromoCode = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  code: 'TEST20',
  name: 'Test Promo',
  type: 'percentage',
  discount_value: 20,
  max_discount: 2000,
  min_order_value: 1000,
  usage_limit: 100,
  usage_per_user: 1,
  used_count: 0,
  is_active: true,
  valid_from: new Date(),
  valid_until: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000),
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

export const createMockZone = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  name: 'Lome Centre',
  type: 'service_area',
  shape: 'polygon',
  coordinates: JSON.stringify([
    [6.1256, 1.2077],
    [6.1256, 1.2477],
    [6.1656, 1.2477],
    [6.1656, 1.2077],
    [6.1256, 1.2077],
  ]),
  is_active: true,
  surge_multiplier: 1.0,
  extra_fee: 0,
  created_at: new Date(),
  updated_at: new Date(),
  ...overrides,
});

// =============================================================================
// TEST APP SETUP
// =============================================================================

export async function createTestingApp(module: any): Promise<INestApplication> {
  const moduleFixture: TestingModule = await Test.createTestingModule(module).compile();

  const app = moduleFixture.createNestApplication();

  app.useGlobalPipes(
    new ValidationPipe({
      whitelist: true,
      forbidNonWhitelisted: true,
      transform: true,
    }),
  );

  await app.init();
  return app;
}

// =============================================================================
// RESET MOCKS
// =============================================================================

export function resetAllMocks() {
  jest.clearAllMocks();

  // Reset Prisma mocks
  Object.values(mockPrismaService).forEach((model) => {
    if (typeof model === 'object' && model !== null) {
      Object.values(model).forEach((method) => {
        if (typeof method === 'function' && 'mockReset' in method) {
          (method as jest.Mock).mockReset();
        }
      });
    }
  });

  // Reset other mocks
  Object.values(mockConfigService).forEach((method) => {
    if (typeof method === 'function' && 'mockReset' in method) {
      (method as jest.Mock).mockReset();
    }
  });

  Object.values(mockJwtService).forEach((method) => {
    if (typeof method === 'function' && 'mockReset' in method) {
      (method as jest.Mock).mockReset();
    }
  });
}

// =============================================================================
// HELPER FUNCTIONS
// =============================================================================

export function expectValidationError(response: any, field: string) {
  expect(response.statusCode).toBe(400);
  expect(response.body.message).toContain(field);
}

export function expectUnauthorized(response: any) {
  expect(response.statusCode).toBe(401);
}

export function expectForbidden(response: any) {
  expect(response.statusCode).toBe(403);
}

export function expectNotFound(response: any) {
  expect(response.statusCode).toBe(404);
}

export function expectSuccess(response: any) {
  expect(response.statusCode).toBe(200);
  expect(response.body.success).toBe(true);
}

export function expectCreated(response: any) {
  expect(response.statusCode).toBe(201);
}

// =============================================================================
// ADDITIONAL MOCK SERVICES
// =============================================================================

export const mockNotificationService = {
  sendToUser: jest.fn(() => Promise.resolve({ success: true })),
  sendToDriver: jest.fn(() => Promise.resolve({ success: true })),
  sendBulk: jest.fn(() => Promise.resolve({ sent: 10, failed: 0 })),
  saveNotification: jest.fn(),
};

export const mockSmsService = {
  sendSms: jest.fn(() => Promise.resolve({ success: true, messageId: 'SMS123' })),
  sendOtp: jest.fn(() => Promise.resolve({ success: true })),
};

export const mockPusherService = {
  trigger: jest.fn(() => Promise.resolve({ success: true })),
  broadcastNewBookingRequest: jest.fn(() => Promise.resolve({ success: true })),
  broadcastBookingAccepted: jest.fn(() => Promise.resolve({ success: true })),
  broadcastDriverLocation: jest.fn(() => Promise.resolve({ success: true })),
  broadcastBookingStatus: jest.fn(() => Promise.resolve({ success: true })),
  getUserChannel: jest.fn((userId: number) => `private-user-${userId}`),
  getDriverChannel: jest.fn((driverId: number) => `private-driver-${driverId}`),
};

export const mockOneSignalService = {
  sendToPlayers: jest.fn(() => Promise.resolve({ success: true, notificationId: 'OS123' })),
  sendToExternalUserId: jest.fn(() => Promise.resolve({ success: true })),
  sendToSegment: jest.fn(() => Promise.resolve({ success: true })),
  notifyNewBookingRequest: jest.fn(() => Promise.resolve({ success: true })),
  notifyBookingAccepted: jest.fn(() => Promise.resolve({ success: true })),
};

export const mockFirebaseService = {
  sendPushNotification: jest.fn(() => Promise.resolve(true)),
  sendMulticast: jest.fn(() => Promise.resolve({ successCount: 10, failureCount: 0 })),
};

export const mockWebsocketService = {
  emitToUser: jest.fn(),
  emitToDriver: jest.fn(),
  emitToBooking: jest.fn(),
  updateDriverLocation: jest.fn(),
};

export const mockSurgePricingService = {
  calculateSurgeMultiplier: jest.fn(() => Promise.resolve({
    multiplier: 1.0,
    reason: 'Normal pricing',
    isActive: false,
  })),
  applyMultiplier: jest.fn((fare: number, multiplier: number) => Math.round(fare * multiplier)),
  clearCache: jest.fn(),
};

export const mockStripe = {
  paymentIntents: {
    create: jest.fn(() => Promise.resolve({ id: 'pi_123', status: 'succeeded' })),
    retrieve: jest.fn(() => Promise.resolve({ id: 'pi_123', status: 'succeeded' })),
  },
  refunds: {
    create: jest.fn(() => Promise.resolve({ id: 're_123', status: 'succeeded' })),
  },
  customers: {
    create: jest.fn(() => Promise.resolve({ id: 'cus_123' })),
    retrieve: jest.fn(() => Promise.resolve({ id: 'cus_123' })),
  },
  paymentMethods: {
    attach: jest.fn(() => Promise.resolve({ id: 'pm_123' })),
    list: jest.fn(() => Promise.resolve({ data: [] })),
  },
};

// =============================================================================
// ADDITIONAL MOCK DATA FACTORIES
// =============================================================================

export const createMockPaymentMethod = (overrides = {}) => ({
  id: 1,
  user_id: 1,
  type: 'card',
  provider: 'stripe',
  card_last_four: '4242',
  card_brand: 'visa',
  is_default: 1,
  stripe_payment_method_id: 'pm_123',
  created_at: new Date(),
  ...overrides,
});

export const createMockTransaction = (overrides = {}) => ({
  id: 1,
  booking_id: 1,
  user_id: 1,
  merchant_id: 1,
  payment_method: 'cash',
  amount: 5000,
  currency: 'XOF',
  status: 'completed',
  transaction_id: 'TXN-123',
  created_at: new Date(),
  ...overrides,
});

export const createMockWalletTransaction = (overrides = {}) => ({
  id: 1,
  user_id: 1,
  amount: 5000,
  type: 'credit',
  description: 'Recharge wallet',
  reference_id: null,
  balance_after: 15000,
  created_at: new Date(),
  ...overrides,
});

export const createMockNotification = (overrides = {}) => ({
  id: 1,
  user_id: 1,
  user_type: 'user',
  merchant_id: 1,
  title: 'Test Notification',
  body: 'This is a test notification',
  type: 'booking',
  data: { booking_id: '1' },
  is_read: 0,
  created_at: new Date(),
  ...overrides,
});

export const createMockDriverRequest = (overrides = {}) => ({
  id: 1,
  booking_id: 1,
  driver_id: 1,
  status: 'pending',
  distance_km: 2.5,
  created_at: new Date(),
  responded_at: null,
  ...overrides,
});

export const createMockSurgeZone = (overrides = {}) => ({
  id: 1,
  merchant_id: 1,
  name: 'Concert Zone',
  latitude: 6.1319,
  longitude: 1.2228,
  radius_km: 3,
  multiplier: 1.5,
  reason: 'Concert event',
  start_time: new Date(),
  end_time: new Date(Date.now() + 4 * 60 * 60 * 1000),
  is_active: true,
  created_at: new Date(),
  ...overrides,
});

export const createMockMerchant = (overrides = {}) => ({
  id: 1,
  company_name: 'Test Merchant',
  email: 'merchant@test.com',
  phone: '+22890000000',
  status: 1,
  commission_rate: 20,
  currency: 'XOF',
  surge_pricing_enabled: true,
  surge_max_multiplier: 3.0,
  surge_trigger_threshold: 0.5,
  created_at: new Date(),
  ...overrides,
});

export const createMockRating = (overrides = {}) => ({
  id: 1,
  booking_id: 1,
  user_id: 1,
  driver_id: 1,
  rating: 5,
  comment: 'Great ride!',
  rated_by: 'user',
  created_at: new Date(),
  ...overrides,
});
