// =============================================================================
// Auth E2E Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { INestApplication, ValidationPipe } from '@nestjs/common';
import * as request from 'supertest';
import { AppModule } from '../src/app.module';
import { PrismaService } from '../src/common/prisma/prisma.service';

describe('AuthController (e2e)', () => {
  let app: INestApplication;
  let prisma: PrismaService;

  beforeAll(async () => {
    const moduleFixture: TestingModule = await Test.createTestingModule({
      imports: [AppModule],
    }).compile();

    app = moduleFixture.createNestApplication();
    app.useGlobalPipes(
      new ValidationPipe({
        whitelist: true,
        forbidNonWhitelisted: true,
        transform: true,
      }),
    );
    app.setGlobalPrefix('api');

    prisma = app.get<PrismaService>(PrismaService);
    await app.init();
  });

  afterAll(async () => {
    await app.close();
  });

  // ===========================================================================
  // USER SIGNUP
  // ===========================================================================

  describe('POST /api/auth/user/signup', () => {
    const signupData = {
      firstName: 'Test',
      lastName: 'User',
      email: `test.user.${Date.now()}@example.com`,
      phone: `+2289${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`,
      password: 'password123',
    };

    it('should create a new user', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send(signupData)
        .expect(201);

      expect(response.body).toHaveProperty('accessToken');
      expect(response.body).toHaveProperty('refreshToken');
      expect(response.body).toHaveProperty('user');
      expect(response.body.user.email).toBe(signupData.email);
    });

    it('should fail with invalid email', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({ ...signupData, email: 'invalid-email' })
        .expect(400);

      expect(response.body.message).toContain('email');
    });

    it('should fail with short password', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({ ...signupData, password: '123' })
        .expect(400);

      expect(response.body.message).toContain('password');
    });

    it('should fail with missing required fields', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({ email: signupData.email })
        .expect(400);

      expect(response.body.message).toBeDefined();
    });
  });

  // ===========================================================================
  // USER LOGIN
  // ===========================================================================

  describe('POST /api/auth/user/login', () => {
    const testEmail = 'login.test@example.com';
    const testPassword = 'password123';

    beforeAll(async () => {
      // Create test user for login tests
      await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({
          firstName: 'Login',
          lastName: 'Test',
          email: testEmail,
          phone: '+22890999888',
          password: testPassword,
        });
    });

    it('should login with valid credentials', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/login')
        .set('X-Merchant-Id', '1')
        .send({
          identifier: testEmail,
          password: testPassword,
        })
        .expect(200);

      expect(response.body).toHaveProperty('accessToken');
      expect(response.body).toHaveProperty('refreshToken');
      expect(response.body.tokenType).toBe('Bearer');
    });

    it('should fail with wrong password', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/login')
        .set('X-Merchant-Id', '1')
        .send({
          identifier: testEmail,
          password: 'wrongpassword',
        })
        .expect(401);

      expect(response.body.message).toContain('Invalid');
    });

    it('should fail with non-existent email', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/login')
        .set('X-Merchant-Id', '1')
        .send({
          identifier: 'nonexistent@example.com',
          password: testPassword,
        })
        .expect(401);

      expect(response.body.message).toBeDefined();
    });
  });

  // ===========================================================================
  // DRIVER SIGNUP
  // ===========================================================================

  describe('POST /api/auth/driver/signup', () => {
    const signupData = {
      firstName: 'Driver',
      lastName: 'Test',
      email: `driver.${Date.now()}@example.com`,
      phone: `+2288${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`,
      password: 'password123',
      vehicleTypeId: 1,
    };

    it('should create a new driver (pending approval)', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/driver/signup')
        .set('X-Merchant-Id', '1')
        .send(signupData)
        .expect(201);

      expect(response.body).toHaveProperty('driver');
      expect(response.body.driver.driver_status).toBe(0); // pending
    });

    it('should fail with duplicate phone', async () => {
      // First signup
      await request(app.getHttpServer())
        .post('/api/auth/driver/signup')
        .set('X-Merchant-Id', '1')
        .send({
          ...signupData,
          email: `driver2.${Date.now()}@example.com`,
        });

      // Second signup with same phone
      const response = await request(app.getHttpServer())
        .post('/api/auth/driver/signup')
        .set('X-Merchant-Id', '1')
        .send({
          ...signupData,
          email: `driver3.${Date.now()}@example.com`,
        })
        .expect(400);

      expect(response.body.message).toContain('exists');
    });
  });

  // ===========================================================================
  // OTP LOGIN
  // ===========================================================================

  describe('POST /api/auth/send-otp', () => {
    it('should send OTP to valid phone', async () => {
      // First create a user
      const phone = `+2287${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`;
      await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({
          firstName: 'OTP',
          lastName: 'Test',
          email: `otp.${Date.now()}@example.com`,
          phone,
          password: 'password123',
        });

      const response = await request(app.getHttpServer())
        .post('/api/auth/send-otp')
        .set('X-Merchant-Id', '1')
        .send({ phone, type: 'user' })
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.message).toContain('OTP');
    });

    it('should fail for non-existent phone', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/send-otp')
        .set('X-Merchant-Id', '1')
        .send({ phone: '+22800000000', type: 'user' })
        .expect(404);

      expect(response.body.message).toContain('not found');
    });
  });

  // ===========================================================================
  // REFRESH TOKEN
  // ===========================================================================

  describe('POST /api/auth/refresh', () => {
    let refreshToken: string;

    beforeAll(async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({
          firstName: 'Refresh',
          lastName: 'Test',
          email: `refresh.${Date.now()}@example.com`,
          phone: `+2286${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`,
          password: 'password123',
        });

      refreshToken = response.body.refreshToken;
    });

    it('should return new tokens with valid refresh token', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/refresh')
        .send({ refreshToken })
        .expect(200);

      expect(response.body).toHaveProperty('accessToken');
      expect(response.body).toHaveProperty('refreshToken');
    });

    it('should fail with invalid refresh token', async () => {
      await request(app.getHttpServer())
        .post('/api/auth/refresh')
        .send({ refreshToken: 'invalid-token' })
        .expect(401);
    });
  });

  // ===========================================================================
  // PROTECTED ROUTES
  // ===========================================================================

  describe('Protected routes', () => {
    let accessToken: string;

    beforeAll(async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({
          firstName: 'Protected',
          lastName: 'Test',
          email: `protected.${Date.now()}@example.com`,
          phone: `+2285${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`,
          password: 'password123',
        });

      accessToken = response.body.accessToken;
    });

    it('should access protected route with valid token', async () => {
      const response = await request(app.getHttpServer())
        .get('/api/users/profile')
        .set('Authorization', `Bearer ${accessToken}`)
        .expect(200);

      expect(response.body).toHaveProperty('id');
    });

    it('should fail without token', async () => {
      await request(app.getHttpServer())
        .get('/api/users/profile')
        .expect(401);
    });

    it('should fail with invalid token', async () => {
      await request(app.getHttpServer())
        .get('/api/users/profile')
        .set('Authorization', 'Bearer invalid-token')
        .expect(401);
    });
  });

  // ===========================================================================
  // LOGOUT
  // ===========================================================================

  describe('POST /api/auth/logout', () => {
    let accessToken: string;

    beforeAll(async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/user/signup')
        .set('X-Merchant-Id', '1')
        .send({
          firstName: 'Logout',
          lastName: 'Test',
          email: `logout.${Date.now()}@example.com`,
          phone: `+2284${Math.floor(Math.random() * 10000000).toString().padStart(7, '0')}`,
          password: 'password123',
        });

      accessToken = response.body.accessToken;
    });

    it('should logout successfully', async () => {
      const response = await request(app.getHttpServer())
        .post('/api/auth/logout')
        .set('Authorization', `Bearer ${accessToken}`)
        .expect(200);

      expect(response.body.success).toBe(true);
    });
  });
});
