import { Injectable, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export type ZoneType =
  | 'service_area'      // Zone de service (où les courses sont possibles)
  | 'surge_zone'        // Zone de surge pricing
  | 'restricted'        // Zone interdite
  | 'airport'           // Aéroport (tarif spécial)
  | 'event'             // Zone événementielle temporaire
  | 'delivery_only'     // Livraison uniquement
  | 'pickup_only'       // Prise en charge uniquement
  | 'dropoff_only';     // Dépose uniquement

export interface Coordinate {
  lat: number;
  lng: number;
}

export interface Zone {
  id: number;
  name: string;
  slug: string;
  type: ZoneType;
  description?: string;
  polygon: Coordinate[];
  center: Coordinate;
  radius?: number; // Pour les zones circulaires (en mètres)
  color?: string;
  surgeMultiplier?: number;
  extraFee?: number;
  vehicleTypes?: number[]; // Types de véhicules autorisés
  schedule?: ZoneSchedule[];
  active: boolean;
  priority: number; // Priorité en cas de chevauchement
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface ZoneSchedule {
  dayOfWeek: number; // 0-6 (dimanche-samedi)
  startTime: string; // HH:mm
  endTime: string;   // HH:mm
  surgeMultiplier?: number;
}

export interface CreateZoneDto {
  name: string;
  slug: string;
  type: ZoneType;
  description?: string;
  polygon?: Coordinate[];
  center?: Coordinate;
  radius?: number;
  color?: string;
  surgeMultiplier?: number;
  extraFee?: number;
  vehicleTypes?: number[];
  schedule?: ZoneSchedule[];
  priority?: number;
}

export interface UpdateZoneDto extends Partial<CreateZoneDto> {
  active?: boolean;
}

export interface PointCheckResult {
  inServiceArea: boolean;
  zones: {
    zone: Zone;
    applicable: boolean;
    reason?: string;
  }[];
  restrictions: string[];
  surgeMultiplier: number;
  extraFees: number;
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class ZoneService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // CRUD ZONES
  // ==========================================================================

  /**
   * Lister toutes les zones
   */
  async listZones(
    merchantId: number,
    options: {
      type?: ZoneType;
      active?: boolean;
      search?: string;
    } = {},
  ): Promise<Zone[]> {
    const where: any = { merchantId };

    if (options.type) {
      where.type = options.type;
    }

    if (options.active !== undefined) {
      where.active = options.active;
    }

    if (options.search) {
      where.OR = [
        { name: { contains: options.search } },
        { description: { contains: options.search } },
      ];
    }

    const zones = await this.prisma.zone.findMany({
      where,
      orderBy: [
        { priority: 'desc' },
        { name: 'asc' },
      ],
    });

    return zones.map(this.mapZone);
  }

  /**
   * Obtenir une zone par ID
   */
  async getZone(id: number, merchantId: number): Promise<Zone | null> {
    const zone = await this.prisma.zone.findFirst({
      where: { id, merchantId },
    });

    return zone ? this.mapZone(zone) : null;
  }

  /**
   * Obtenir une zone par slug
   */
  async getZoneBySlug(slug: string, merchantId: number): Promise<Zone | null> {
    const zone = await this.prisma.zone.findFirst({
      where: { slug, merchantId },
    });

    return zone ? this.mapZone(zone) : null;
  }

  /**
   * Créer une zone
   */
  async createZone(merchantId: number, data: CreateZoneDto): Promise<Zone> {
    // Vérifier slug unique
    const existing = await this.prisma.zone.findFirst({
      where: { slug: data.slug, merchantId },
    });

    if (existing) {
      throw new BadRequestException('Ce slug existe déjà');
    }

    // Valider les coordonnées
    if (data.polygon && data.polygon.length < 3) {
      throw new BadRequestException('Un polygone doit avoir au moins 3 points');
    }

    if (!data.polygon && !data.radius) {
      throw new BadRequestException('Spécifiez un polygone ou un rayon');
    }

    // Calculer le centre si non fourni
    let center = data.center;
    if (!center && data.polygon) {
      center = this.calculatePolygonCenter(data.polygon);
    }

    const zone = await this.prisma.zone.create({
      data: {
        merchantId,
        name: data.name,
        slug: data.slug,
        type: data.type,
        description: data.description,
        polygon: data.polygon ? JSON.stringify(data.polygon) : null,
        centerLat: center?.lat,
        centerLng: center?.lng,
        radius: data.radius,
        color: data.color || this.getDefaultColor(data.type),
        surgeMultiplier: data.surgeMultiplier || 1,
        extraFee: data.extraFee || 0,
        vehicleTypes: data.vehicleTypes ? JSON.stringify(data.vehicleTypes) : null,
        schedule: data.schedule ? JSON.stringify(data.schedule) : null,
        priority: data.priority || 0,
        active: true,
      },
    });

    return this.mapZone(zone);
  }

  /**
   * Mettre à jour une zone
   */
  async updateZone(
    id: number,
    merchantId: number,
    data: UpdateZoneDto,
  ): Promise<Zone | null> {
    const zone = await this.prisma.zone.findFirst({
      where: { id, merchantId },
    });

    if (!zone) {
      return null;
    }

    // Vérifier slug unique si modifié
    if (data.slug && data.slug !== zone.slug) {
      const existing = await this.prisma.zone.findFirst({
        where: { slug: data.slug, merchantId, id: { not: id } },
      });

      if (existing) {
        throw new BadRequestException('Ce slug existe déjà');
      }
    }

    // Calculer le centre si polygone modifié
    let center = data.center;
    if (data.polygon && !center) {
      center = this.calculatePolygonCenter(data.polygon);
    }

    const updated = await this.prisma.zone.update({
      where: { id },
      data: {
        name: data.name,
        slug: data.slug,
        type: data.type,
        description: data.description,
        polygon: data.polygon ? JSON.stringify(data.polygon) : undefined,
        centerLat: center?.lat,
        centerLng: center?.lng,
        radius: data.radius,
        color: data.color,
        surgeMultiplier: data.surgeMultiplier,
        extraFee: data.extraFee,
        vehicleTypes: data.vehicleTypes ? JSON.stringify(data.vehicleTypes) : undefined,
        schedule: data.schedule ? JSON.stringify(data.schedule) : undefined,
        priority: data.priority,
        active: data.active,
      },
    });

    return this.mapZone(updated);
  }

  /**
   * Supprimer une zone
   */
  async deleteZone(id: number, merchantId: number): Promise<boolean> {
    const zone = await this.prisma.zone.findFirst({
      where: { id, merchantId },
    });

    if (!zone) {
      return false;
    }

    await this.prisma.zone.delete({ where: { id } });
    return true;
  }

  // ==========================================================================
  // GEOFENCING
  // ==========================================================================

  /**
   * Vérifier si un point est dans une zone
   */
  isPointInZone(point: Coordinate, zone: Zone): boolean {
    // Zone circulaire
    if (zone.radius && zone.center) {
      const distance = this.calculateDistance(point, zone.center);
      return distance <= zone.radius;
    }

    // Zone polygonale
    if (zone.polygon && zone.polygon.length >= 3) {
      return this.isPointInPolygon(point, zone.polygon);
    }

    return false;
  }

  /**
   * Trouver toutes les zones contenant un point
   */
  async findZonesContainingPoint(
    point: Coordinate,
    merchantId: number,
    activeOnly = true,
  ): Promise<Zone[]> {
    const allZones = await this.listZones(merchantId, { active: activeOnly });

    return allZones
      .filter((zone) => this.isPointInZone(point, zone))
      .sort((a, b) => b.priority - a.priority);
  }

  /**
   * Vérifier un point pour les courses
   */
  async checkPoint(
    point: Coordinate,
    merchantId: number,
    vehicleTypeId?: number,
  ): Promise<PointCheckResult> {
    const zones = await this.findZonesContainingPoint(point, merchantId);
    const now = new Date();
    const currentDay = now.getDay();
    const currentTime = `${now.getHours().toString().padStart(2, '0')}:${now.getMinutes().toString().padStart(2, '0')}`;

    let inServiceArea = false;
    let surgeMultiplier = 1;
    let extraFees = 0;
    const restrictions: string[] = [];
    const zoneResults: { zone: Zone; applicable: boolean; reason?: string }[] = [];

    for (const zone of zones) {
      let applicable = true;
      let reason: string | undefined;

      // Vérifier le type de véhicule
      if (vehicleTypeId && zone.vehicleTypes && zone.vehicleTypes.length > 0) {
        if (!zone.vehicleTypes.includes(vehicleTypeId)) {
          applicable = false;
          reason = 'Type de véhicule non autorisé dans cette zone';
        }
      }

      // Vérifier le planning
      if (applicable && zone.schedule && zone.schedule.length > 0) {
        const scheduleForToday = zone.schedule.find(
          (s) => s.dayOfWeek === currentDay,
        );

        if (scheduleForToday) {
          if (currentTime < scheduleForToday.startTime || currentTime > scheduleForToday.endTime) {
            applicable = false;
            reason = 'Hors des heures de service pour cette zone';
          } else if (scheduleForToday.surgeMultiplier) {
            surgeMultiplier = Math.max(surgeMultiplier, scheduleForToday.surgeMultiplier);
          }
        }
      }

      // Traiter selon le type de zone
      if (applicable) {
        switch (zone.type) {
          case 'service_area':
            inServiceArea = true;
            break;

          case 'surge_zone':
            if (zone.surgeMultiplier) {
              surgeMultiplier = Math.max(surgeMultiplier, zone.surgeMultiplier);
            }
            break;

          case 'restricted':
            restrictions.push(`Zone interdite: ${zone.name}`);
            break;

          case 'airport':
            if (zone.extraFee) {
              extraFees += zone.extraFee;
            }
            break;

          case 'event':
            if (zone.surgeMultiplier) {
              surgeMultiplier = Math.max(surgeMultiplier, zone.surgeMultiplier);
            }
            if (zone.extraFee) {
              extraFees += zone.extraFee;
            }
            break;

          case 'delivery_only':
            restrictions.push('Zone réservée aux livraisons');
            break;

          case 'pickup_only':
            // Géré côté booking
            break;

          case 'dropoff_only':
            // Géré côté booking
            break;
        }
      }

      zoneResults.push({ zone, applicable, reason });
    }

    return {
      inServiceArea,
      zones: zoneResults,
      restrictions,
      surgeMultiplier,
      extraFees,
    };
  }

  /**
   * Vérifier un trajet (pickup et dropoff)
   */
  async checkRoute(
    pickup: Coordinate,
    dropoff: Coordinate,
    merchantId: number,
    vehicleTypeId?: number,
  ): Promise<{
    valid: boolean;
    errors: string[];
    pickup: PointCheckResult;
    dropoff: PointCheckResult;
    surgeMultiplier: number;
    extraFees: number;
  }> {
    const [pickupCheck, dropoffCheck] = await Promise.all([
      this.checkPoint(pickup, merchantId, vehicleTypeId),
      this.checkPoint(dropoff, merchantId, vehicleTypeId),
    ]);

    const errors: string[] = [];

    // Vérifier que le pickup est dans une zone de service
    if (!pickupCheck.inServiceArea) {
      errors.push('Le point de départ est en dehors de la zone de service');
    }

    // Vérifier que le dropoff est dans une zone de service
    if (!dropoffCheck.inServiceArea) {
      errors.push('La destination est en dehors de la zone de service');
    }

    // Ajouter les restrictions
    errors.push(...pickupCheck.restrictions);
    errors.push(...dropoffCheck.restrictions);

    // Vérifier les zones pickup_only et dropoff_only
    const pickupZones = pickupCheck.zones.filter((z) => z.applicable);
    const dropoffZones = dropoffCheck.zones.filter((z) => z.applicable);

    const hasDropoffOnlyAtPickup = pickupZones.some(
      (z) => z.zone.type === 'dropoff_only',
    );
    if (hasDropoffOnlyAtPickup) {
      errors.push('Impossible de prendre en charge dans cette zone (dépose uniquement)');
    }

    const hasPickupOnlyAtDropoff = dropoffZones.some(
      (z) => z.zone.type === 'pickup_only',
    );
    if (hasPickupOnlyAtDropoff) {
      errors.push('Impossible de déposer dans cette zone (prise en charge uniquement)');
    }

    // Calculer le surge et les frais
    const surgeMultiplier = Math.max(
      pickupCheck.surgeMultiplier,
      dropoffCheck.surgeMultiplier,
    );
    const extraFees = pickupCheck.extraFees + dropoffCheck.extraFees;

    return {
      valid: errors.length === 0,
      errors,
      pickup: pickupCheck,
      dropoff: dropoffCheck,
      surgeMultiplier,
      extraFees,
    };
  }

  // ==========================================================================
  // ZONE STATISTICS
  // ==========================================================================

  /**
   * Statistiques des zones
   */
  async getStatistics(merchantId: number): Promise<{
    total: number;
    byType: Record<ZoneType, number>;
    active: number;
    inactive: number;
  }> {
    const [total, active, typeCounts] = await Promise.all([
      this.prisma.zone.count({ where: { merchantId } }),
      this.prisma.zone.count({ where: { merchantId, active: true } }),
      this.prisma.zone.groupBy({
        by: ['type'],
        where: { merchantId },
        _count: { type: true },
      }),
    ]);

    const byType: Record<string, number> = {
      service_area: 0,
      surge_zone: 0,
      restricted: 0,
      airport: 0,
      event: 0,
      delivery_only: 0,
      pickup_only: 0,
      dropoff_only: 0,
    };

    typeCounts.forEach((t) => {
      byType[t.type] = t._count.type;
    });

    return {
      total,
      byType: byType as Record<ZoneType, number>,
      active,
      inactive: total - active,
    };
  }

  // ==========================================================================
  // HELPERS - GEOMÉTRIE
  // ==========================================================================

  /**
   * Calculer le centre d'un polygone
   */
  private calculatePolygonCenter(polygon: Coordinate[]): Coordinate {
    const sumLat = polygon.reduce((sum, p) => sum + p.lat, 0);
    const sumLng = polygon.reduce((sum, p) => sum + p.lng, 0);

    return {
      lat: sumLat / polygon.length,
      lng: sumLng / polygon.length,
    };
  }

  /**
   * Calculer la distance entre deux points (en mètres)
   * Formule de Haversine
   */
  private calculateDistance(point1: Coordinate, point2: Coordinate): number {
    const R = 6371000; // Rayon de la Terre en mètres
    const lat1 = (point1.lat * Math.PI) / 180;
    const lat2 = (point2.lat * Math.PI) / 180;
    const deltaLat = ((point2.lat - point1.lat) * Math.PI) / 180;
    const deltaLng = ((point2.lng - point1.lng) * Math.PI) / 180;

    const a =
      Math.sin(deltaLat / 2) * Math.sin(deltaLat / 2) +
      Math.cos(lat1) *
        Math.cos(lat2) *
        Math.sin(deltaLng / 2) *
        Math.sin(deltaLng / 2);

    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

    return R * c;
  }

  /**
   * Vérifier si un point est dans un polygone
   * Algorithme Ray Casting
   */
  private isPointInPolygon(point: Coordinate, polygon: Coordinate[]): boolean {
    let inside = false;
    const x = point.lng;
    const y = point.lat;

    for (let i = 0, j = polygon.length - 1; i < polygon.length; j = i++) {
      const xi = polygon[i].lng;
      const yi = polygon[i].lat;
      const xj = polygon[j].lng;
      const yj = polygon[j].lat;

      if (yi > y !== yj > y && x < ((xj - xi) * (y - yi)) / (yj - yi) + xi) {
        inside = !inside;
      }
    }

    return inside;
  }

  /**
   * Couleur par défaut selon le type
   */
  private getDefaultColor(type: ZoneType): string {
    const colors: Record<ZoneType, string> = {
      service_area: '#4CAF50',    // Vert
      surge_zone: '#FF9800',      // Orange
      restricted: '#F44336',      // Rouge
      airport: '#2196F3',         // Bleu
      event: '#9C27B0',           // Violet
      delivery_only: '#00BCD4',   // Cyan
      pickup_only: '#8BC34A',     // Vert clair
      dropoff_only: '#CDDC39',    // Lime
    };

    return colors[type] || '#607D8B';
  }

  /**
   * Mapper zone depuis DB
   */
  private mapZone(zone: any): Zone {
    return {
      id: zone.id,
      name: zone.name,
      slug: zone.slug,
      type: zone.type,
      description: zone.description,
      polygon: zone.polygon ? JSON.parse(zone.polygon) : [],
      center: {
        lat: zone.centerLat,
        lng: zone.centerLng,
      },
      radius: zone.radius,
      color: zone.color,
      surgeMultiplier: zone.surgeMultiplier ? parseFloat(zone.surgeMultiplier) : undefined,
      extraFee: zone.extraFee ? parseFloat(zone.extraFee) : undefined,
      vehicleTypes: zone.vehicleTypes ? JSON.parse(zone.vehicleTypes) : undefined,
      schedule: zone.schedule ? JSON.parse(zone.schedule) : undefined,
      active: zone.active,
      priority: zone.priority,
      merchantId: zone.merchantId,
      createdAt: zone.createdAt,
      updatedAt: zone.updatedAt,
    };
  }
}
