import {
  WebSocketGateway,
  WebSocketServer,
  SubscribeMessage,
  OnGatewayConnection,
  OnGatewayDisconnect,
  OnGatewayInit,
  ConnectedSocket,
  MessageBody,
} from '@nestjs/websockets';
import { Logger, UseGuards } from '@nestjs/common';
import { Server, Socket } from 'socket.io';
import { JwtService } from '@nestjs/jwt';
import { WebsocketService } from './websocket.service';

interface AuthenticatedSocket extends Socket {
  user?: {
    id: number;
    type: 'user' | 'driver';
    merchantId: number;
  };
}

@WebSocketGateway({
  cors: {
    origin: '*',
    credentials: true,
  },
  namespace: '/',
})
export class WebsocketGateway
  implements OnGatewayInit, OnGatewayConnection, OnGatewayDisconnect
{
  @WebSocketServer()
  server: Server;

  private readonly logger = new Logger(WebsocketGateway.name);

  constructor(
    private jwtService: JwtService,
    private websocketService: WebsocketService,
  ) {}

  afterInit(server: Server) {
    this.logger.log('WebSocket Gateway initialized');
    this.websocketService.setServer(server);
  }

  async handleConnection(client: AuthenticatedSocket) {
    try {
      // Extract token from query or auth header
      const token =
        client.handshake.query.token as string ||
        client.handshake.auth?.token ||
        client.handshake.headers.authorization?.replace('Bearer ', '');

      if (!token) {
        this.logger.warn(`Client ${client.id} connected without token`);
        client.disconnect();
        return;
      }

      // Verify token
      const payload = this.jwtService.verify(token);
      client.user = {
        id: payload.sub,
        type: payload.type,
        merchantId: payload.merchantId,
      };

      // Join rooms based on user type
      const userRoom = `${client.user.type}.${client.user.id}`;
      const merchantRoom = `merchant.${client.user.merchantId}`;

      client.join(userRoom);
      client.join(merchantRoom);

      // Track connection
      this.websocketService.addConnection(client.user.id, client.user.type, client.id);

      this.logger.log(
        `Client connected: ${client.id} (${client.user.type} #${client.user.id})`,
      );
    } catch (error) {
      this.logger.error(`Authentication failed for client ${client.id}: ${error.message}`);
      client.disconnect();
    }
  }

  handleDisconnect(client: AuthenticatedSocket) {
    if (client.user) {
      this.websocketService.removeConnection(client.user.id, client.user.type);
      this.logger.log(
        `Client disconnected: ${client.id} (${client.user.type} #${client.user.id})`,
      );
    }
  }

  // ============================================================================
  // DRIVER LOCATION
  // ============================================================================

  @SubscribeMessage('driver:location:update')
  handleDriverLocationUpdate(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody()
    data: {
      latitude: number;
      longitude: number;
      heading?: number;
      speed?: number;
      bookingId?: number;
    },
  ) {
    if (client.user?.type !== 'driver') {
      return { error: 'Only drivers can update location' };
    }

    const locationData = {
      driverId: client.user.id,
      latitude: data.latitude,
      longitude: data.longitude,
      heading: data.heading,
      speed: data.speed,
      timestamp: new Date().toISOString(),
    };

    // Broadcast to booking room if there's an active booking
    if (data.bookingId) {
      this.server
        .to(`booking.${data.bookingId}`)
        .emit('driver:location:updated', locationData);
    }

    // Broadcast to merchant for tracking all drivers
    this.server
      .to(`merchant.${client.user.merchantId}`)
      .emit('driver:location:updated', locationData);

    return { success: true };
  }

  // ============================================================================
  // BOOKING EVENTS
  // ============================================================================

  @SubscribeMessage('booking:subscribe')
  handleBookingSubscribe(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody() data: { bookingId: number },
  ) {
    const room = `booking.${data.bookingId}`;
    client.join(room);
    this.logger.log(`Client ${client.id} subscribed to ${room}`);
    return { success: true, room };
  }

  @SubscribeMessage('booking:unsubscribe')
  handleBookingUnsubscribe(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody() data: { bookingId: number },
  ) {
    const room = `booking.${data.bookingId}`;
    client.leave(room);
    this.logger.log(`Client ${client.id} unsubscribed from ${room}`);
    return { success: true };
  }

  // ============================================================================
  // CHAT
  // ============================================================================

  @SubscribeMessage('chat:message')
  handleChatMessage(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody()
    data: {
      bookingId: number;
      message: string;
      messageType?: 'text' | 'image' | 'location';
    },
  ) {
    if (!client.user) {
      return { error: 'Not authenticated' };
    }

    const chatMessage = {
      bookingId: data.bookingId,
      senderId: client.user.id,
      senderType: client.user.type,
      message: data.message,
      messageType: data.messageType || 'text',
      timestamp: new Date().toISOString(),
    };

    // Broadcast to booking room
    this.server
      .to(`booking.${data.bookingId}`)
      .emit('chat:message:new', chatMessage);

    return { success: true, message: chatMessage };
  }

  @SubscribeMessage('chat:typing')
  handleChatTyping(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody() data: { bookingId: number; isTyping: boolean },
  ) {
    if (!client.user) return;

    this.server.to(`booking.${data.bookingId}`).emit('chat:typing', {
      senderId: client.user.id,
      senderType: client.user.type,
      isTyping: data.isTyping,
    });
  }

  // ============================================================================
  // DRIVER STATUS
  // ============================================================================

  @SubscribeMessage('driver:status:update')
  handleDriverStatusUpdate(
    @ConnectedSocket() client: AuthenticatedSocket,
    @MessageBody() data: { isOnline: boolean },
  ) {
    if (client.user?.type !== 'driver') {
      return { error: 'Only drivers can update status' };
    }

    // Broadcast to merchant
    this.server.to(`merchant.${client.user.merchantId}`).emit('driver:status:updated', {
      driverId: client.user.id,
      isOnline: data.isOnline,
      timestamp: new Date().toISOString(),
    });

    return { success: true };
  }
}
