import { Injectable, Logger } from '@nestjs/common';
import { HttpService } from '@nestjs/axios';
import { PrismaService } from '../../common/prisma/prisma.service';
import { ConfigService } from '@nestjs/config';
import { firstValueFrom, timeout, catchError } from 'rxjs';
import { of } from 'rxjs';
import * as crypto from 'crypto';

export type WebhookEvent =
  | 'booking.created'
  | 'booking.accepted'
  | 'booking.arrived'
  | 'booking.started'
  | 'booking.completed'
  | 'booking.cancelled'
  | 'delivery.created'
  | 'delivery.picked_up'
  | 'delivery.delivered'
  | 'delivery.cancelled'
  | 'driver.online'
  | 'driver.offline'
  | 'driver.location_updated'
  | 'payment.completed'
  | 'payment.failed'
  | 'payment.refunded'
  | 'user.created'
  | 'user.updated'
  | 'driver.created'
  | 'driver.approved'
  | 'driver.suspended';

export interface WebhookEndpoint {
  id: number;
  merchantId: number;
  url: string;
  events: string[];
  secret: string;
  active: boolean;
  description?: string;
  headers?: Record<string, string>;
  retryCount: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface WebhookDelivery {
  id: number;
  endpointId: number;
  event: string;
  payload: any;
  status: 'pending' | 'delivered' | 'failed';
  statusCode?: number;
  response?: string;
  attempts: number;
  nextRetryAt?: Date;
  deliveredAt?: Date;
  createdAt: Date;
}

export interface WebhookPayload {
  event: WebhookEvent;
  timestamp: string;
  data: any;
  merchantId?: number;
}

@Injectable()
export class WebhookService {
  private readonly logger = new Logger(WebhookService.name);
  private readonly maxRetries: number;
  private readonly retryDelays: number[];

  constructor(
    private httpService: HttpService,
    private prisma: PrismaService,
    private configService: ConfigService,
  ) {
    this.maxRetries = this.configService.get<number>('WEBHOOK_MAX_RETRIES', 5);
    this.retryDelays = [60, 300, 900, 3600, 86400]; // 1min, 5min, 15min, 1h, 24h
  }

  // ============================================================================
  // ENDPOINT MANAGEMENT
  // ============================================================================

  /**
   * Create webhook endpoint
   */
  async createEndpoint(
    merchantId: number,
    data: {
      url: string;
      events: WebhookEvent[];
      description?: string;
      headers?: Record<string, string>;
    },
  ): Promise<WebhookEndpoint> {
    const secret = this.generateSecret();

    const endpoint = await this.prisma.webhookEndpoint.create({
      data: {
        merchant_id: merchantId,
        url: data.url,
        events: JSON.stringify(data.events),
        secret,
        active: true,
        description: data.description,
        headers: data.headers ? JSON.stringify(data.headers) : null,
        retry_count: this.maxRetries,
        created_at: new Date(),
        updated_at: new Date(),
      },
    });

    return this.mapToEndpoint(endpoint);
  }

  /**
   * Update webhook endpoint
   */
  async updateEndpoint(
    endpointId: number,
    merchantId: number,
    data: Partial<{
      url: string;
      events: WebhookEvent[];
      active: boolean;
      description: string;
      headers: Record<string, string>;
    }>,
  ): Promise<WebhookEndpoint | null> {
    const updateData: any = { updated_at: new Date() };

    if (data.url) updateData.url = data.url;
    if (data.events) updateData.events = JSON.stringify(data.events);
    if (typeof data.active === 'boolean') updateData.active = data.active;
    if (data.description !== undefined) updateData.description = data.description;
    if (data.headers) updateData.headers = JSON.stringify(data.headers);

    const endpoint = await this.prisma.webhookEndpoint.updateMany({
      where: { id: endpointId, merchant_id: merchantId },
      data: updateData,
    });

    if (endpoint.count === 0) return null;

    return this.getEndpoint(endpointId, merchantId);
  }

  /**
   * Delete webhook endpoint
   */
  async deleteEndpoint(endpointId: number, merchantId: number): Promise<boolean> {
    const result = await this.prisma.webhookEndpoint.deleteMany({
      where: { id: endpointId, merchant_id: merchantId },
    });

    return result.count > 0;
  }

  /**
   * Get webhook endpoint
   */
  async getEndpoint(
    endpointId: number,
    merchantId: number,
  ): Promise<WebhookEndpoint | null> {
    const endpoint = await this.prisma.webhookEndpoint.findFirst({
      where: { id: endpointId, merchant_id: merchantId },
    });

    return endpoint ? this.mapToEndpoint(endpoint) : null;
  }

  /**
   * List webhook endpoints
   */
  async listEndpoints(merchantId: number): Promise<WebhookEndpoint[]> {
    const endpoints = await this.prisma.webhookEndpoint.findMany({
      where: { merchant_id: merchantId },
      orderBy: { created_at: 'desc' },
    });

    return endpoints.map(this.mapToEndpoint);
  }

  /**
   * Rotate webhook secret
   */
  async rotateSecret(
    endpointId: number,
    merchantId: number,
  ): Promise<{ secret: string } | null> {
    const newSecret = this.generateSecret();

    const result = await this.prisma.webhookEndpoint.updateMany({
      where: { id: endpointId, merchant_id: merchantId },
      data: { secret: newSecret, updated_at: new Date() },
    });

    if (result.count === 0) return null;

    return { secret: newSecret };
  }

  // ============================================================================
  // WEBHOOK DISPATCH
  // ============================================================================

  /**
   * Dispatch webhook event
   */
  async dispatch(
    event: WebhookEvent,
    data: any,
    merchantId?: number,
  ): Promise<number> {
    const endpoints = await this.getEndpointsForEvent(event, merchantId);

    if (!endpoints.length) {
      return 0;
    }

    const payload: WebhookPayload = {
      event,
      timestamp: new Date().toISOString(),
      data,
      merchantId,
    };

    let dispatched = 0;

    for (const endpoint of endpoints) {
      await this.queueDelivery(endpoint, payload);
      dispatched++;
    }

    return dispatched;
  }

  /**
   * Queue webhook delivery
   */
  private async queueDelivery(
    endpoint: WebhookEndpoint,
    payload: WebhookPayload,
  ): Promise<void> {
    const delivery = await this.prisma.webhookDelivery.create({
      data: {
        endpoint_id: endpoint.id,
        event: payload.event,
        payload: JSON.stringify(payload),
        status: 'pending',
        attempts: 0,
        created_at: new Date(),
      },
    });

    // Attempt immediate delivery
    this.deliver(delivery.id).catch((error) => {
      this.logger.debug(`Webhook delivery queued for retry: ${error.message}`);
    });
  }

  /**
   * Deliver webhook
   */
  async deliver(deliveryId: number): Promise<boolean> {
    const delivery = await this.prisma.webhookDelivery.findUnique({
      where: { id: deliveryId },
      include: { endpoint: true },
    });

    if (!delivery || delivery.status === 'delivered') {
      return false;
    }

    const endpoint = this.mapToEndpoint(delivery.endpoint);
    const payload = JSON.parse(delivery.payload);

    // Generate signature
    const signature = this.generateSignature(
      JSON.stringify(payload),
      endpoint.secret,
    );

    // Prepare headers
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      'X-Webhook-Signature': signature,
      'X-Webhook-Event': payload.event,
      'X-Webhook-Timestamp': payload.timestamp,
      ...(endpoint.headers || {}),
    };

    try {
      const response = await firstValueFrom(
        this.httpService
          .post(endpoint.url, payload, { headers })
          .pipe(
            timeout(30000),
            catchError((error) => {
              throw error;
            }),
          ),
      );

      // Success
      await this.prisma.webhookDelivery.update({
        where: { id: deliveryId },
        data: {
          status: 'delivered',
          status_code: response.status,
          response: JSON.stringify(response.data).substring(0, 1000),
          attempts: delivery.attempts + 1,
          delivered_at: new Date(),
        },
      });

      return true;
    } catch (error) {
      const attempts = delivery.attempts + 1;
      const shouldRetry = attempts < this.maxRetries;

      await this.prisma.webhookDelivery.update({
        where: { id: deliveryId },
        data: {
          status: shouldRetry ? 'pending' : 'failed',
          status_code: error.response?.status || 0,
          response: error.message.substring(0, 1000),
          attempts,
          next_retry_at: shouldRetry
            ? new Date(Date.now() + this.retryDelays[attempts - 1] * 1000)
            : null,
        },
      });

      this.logger.warn(
        `Webhook delivery failed: ${error.message} (attempt ${attempts}/${this.maxRetries})`,
      );

      return false;
    }
  }

  /**
   * Retry failed deliveries
   */
  async retryFailedDeliveries(): Promise<number> {
    const pendingDeliveries = await this.prisma.webhookDelivery.findMany({
      where: {
        status: 'pending',
        next_retry_at: { lte: new Date() },
      },
      take: 100,
    });

    let retried = 0;

    for (const delivery of pendingDeliveries) {
      const success = await this.deliver(delivery.id);
      if (success) retried++;
    }

    return retried;
  }

  // ============================================================================
  // DELIVERY HISTORY
  // ============================================================================

  /**
   * Get delivery history for endpoint
   */
  async getDeliveryHistory(
    endpointId: number,
    merchantId: number,
    options: { page?: number; limit?: number; status?: string } = {},
  ): Promise<{
    data: WebhookDelivery[];
    total: number;
    page: number;
    limit: number;
  }> {
    const page = options.page || 1;
    const limit = Math.min(options.limit || 50, 100);
    const skip = (page - 1) * limit;

    // Verify endpoint belongs to merchant
    const endpoint = await this.getEndpoint(endpointId, merchantId);
    if (!endpoint) {
      return { data: [], total: 0, page, limit };
    }

    const where: any = { endpoint_id: endpointId };
    if (options.status) {
      where.status = options.status;
    }

    const [deliveries, total] = await Promise.all([
      this.prisma.webhookDelivery.findMany({
        where,
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
      }),
      this.prisma.webhookDelivery.count({ where }),
    ]);

    return {
      data: deliveries.map(this.mapToDelivery),
      total,
      page,
      limit,
    };
  }

  /**
   * Get delivery by ID
   */
  async getDelivery(
    deliveryId: number,
    merchantId: number,
  ): Promise<WebhookDelivery | null> {
    const delivery = await this.prisma.webhookDelivery.findUnique({
      where: { id: deliveryId },
      include: { endpoint: true },
    });

    if (!delivery || delivery.endpoint.merchant_id !== merchantId) {
      return null;
    }

    return this.mapToDelivery(delivery);
  }

  /**
   * Retry specific delivery
   */
  async retryDelivery(
    deliveryId: number,
    merchantId: number,
  ): Promise<boolean> {
    const delivery = await this.getDelivery(deliveryId, merchantId);

    if (!delivery) {
      return false;
    }

    // Reset attempts to allow retry
    await this.prisma.webhookDelivery.update({
      where: { id: deliveryId },
      data: {
        status: 'pending',
        next_retry_at: null,
      },
    });

    return this.deliver(deliveryId);
  }

  // ============================================================================
  // STATISTICS
  // ============================================================================

  /**
   * Get webhook statistics
   */
  async getStatistics(
    merchantId: number,
    startDate?: Date,
    endDate?: Date,
  ): Promise<{
    endpoints: number;
    activeEndpoints: number;
    deliveries: {
      total: number;
      delivered: number;
      failed: number;
      pending: number;
    };
    byEvent: Record<string, number>;
    recentFailures: WebhookDelivery[];
  }> {
    const where: any = {
      endpoint: { merchant_id: merchantId },
    };

    if (startDate || endDate) {
      where.created_at = {};
      if (startDate) where.created_at.gte = startDate;
      if (endDate) where.created_at.lte = endDate;
    }

    const [
      endpoints,
      activeEndpoints,
      statusGroups,
      eventGroups,
      recentFailures,
    ] = await Promise.all([
      this.prisma.webhookEndpoint.count({
        where: { merchant_id: merchantId },
      }),
      this.prisma.webhookEndpoint.count({
        where: { merchant_id: merchantId, active: true },
      }),
      this.prisma.webhookDelivery.groupBy({
        by: ['status'],
        where,
        _count: true,
      }),
      this.prisma.webhookDelivery.groupBy({
        by: ['event'],
        where,
        _count: true,
      }),
      this.prisma.webhookDelivery.findMany({
        where: { ...where, status: 'failed' },
        orderBy: { created_at: 'desc' },
        take: 10,
      }),
    ]);

    const deliveries = {
      total: 0,
      delivered: 0,
      failed: 0,
      pending: 0,
    };

    for (const group of statusGroups) {
      deliveries[group.status as keyof typeof deliveries] = group._count;
      deliveries.total += group._count;
    }

    return {
      endpoints,
      activeEndpoints,
      deliveries,
      byEvent: eventGroups.reduce(
        (acc, g) => ({ ...acc, [g.event]: g._count }),
        {},
      ),
      recentFailures: recentFailures.map(this.mapToDelivery),
    };
  }

  // ============================================================================
  // PRIVATE METHODS
  // ============================================================================

  private async getEndpointsForEvent(
    event: WebhookEvent,
    merchantId?: number,
  ): Promise<WebhookEndpoint[]> {
    const where: any = { active: true };
    if (merchantId) {
      where.merchant_id = merchantId;
    }

    const endpoints = await this.prisma.webhookEndpoint.findMany({ where });

    return endpoints
      .map(this.mapToEndpoint)
      .filter((e) => e.events.includes(event) || e.events.includes('*'));
  }

  private generateSecret(): string {
    return 'whsec_' + crypto.randomBytes(32).toString('hex');
  }

  private generateSignature(payload: string, secret: string): string {
    const hmac = crypto.createHmac('sha256', secret);
    hmac.update(payload);
    return `sha256=${hmac.digest('hex')}`;
  }

  private mapToEndpoint(endpoint: any): WebhookEndpoint {
    return {
      id: endpoint.id,
      merchantId: endpoint.merchant_id,
      url: endpoint.url,
      events: JSON.parse(endpoint.events),
      secret: endpoint.secret,
      active: endpoint.active,
      description: endpoint.description,
      headers: endpoint.headers ? JSON.parse(endpoint.headers) : null,
      retryCount: endpoint.retry_count,
      createdAt: endpoint.created_at,
      updatedAt: endpoint.updated_at,
    };
  }

  private mapToDelivery(delivery: any): WebhookDelivery {
    return {
      id: delivery.id,
      endpointId: delivery.endpoint_id,
      event: delivery.event,
      payload: JSON.parse(delivery.payload),
      status: delivery.status,
      statusCode: delivery.status_code,
      response: delivery.response,
      attempts: delivery.attempts,
      nextRetryAt: delivery.next_retry_at,
      deliveredAt: delivery.delivered_at,
      createdAt: delivery.created_at,
    };
  }
}
