import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { WebhookService, WebhookEvent } from './webhook.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class CreateEndpointDto {
  url: string;
  events: WebhookEvent[];
  description?: string;
  headers?: Record<string, string>;
}

class UpdateEndpointDto {
  url?: string;
  events?: WebhookEvent[];
  active?: boolean;
  description?: string;
  headers?: Record<string, string>;
}

class TestWebhookDto {
  event?: WebhookEvent;
  data?: any;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('webhooks')
@UseGuards(JwtAuthGuard)
export class WebhookController {
  constructor(private readonly webhookService: WebhookService) {}

  // ============================================================================
  // ENDPOINT MANAGEMENT
  // ============================================================================

  /**
   * List webhook endpoints
   */
  @Get('endpoints')
  async listEndpoints(@MerchantId() merchantId: number) {
    const endpoints = await this.webhookService.listEndpoints(merchantId);

    // Hide secrets in list
    return endpoints.map((e) => ({
      ...e,
      secret: e.secret.substring(0, 12) + '...',
    }));
  }

  /**
   * Create webhook endpoint
   */
  @Post('endpoints')
  async createEndpoint(
    @MerchantId() merchantId: number,
    @Body() dto: CreateEndpointDto,
  ) {
    const endpoint = await this.webhookService.createEndpoint(merchantId, dto);

    return {
      success: true,
      endpoint,
      message: 'Save the secret securely. It will not be shown again.',
    };
  }

  /**
   * Get webhook endpoint
   */
  @Get('endpoints/:id')
  async getEndpoint(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const endpoint = await this.webhookService.getEndpoint(id, merchantId);

    if (!endpoint) {
      return { error: 'Endpoint not found' };
    }

    // Hide secret
    return {
      ...endpoint,
      secret: endpoint.secret.substring(0, 12) + '...',
    };
  }

  /**
   * Update webhook endpoint
   */
  @Put('endpoints/:id')
  async updateEndpoint(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateEndpointDto,
  ) {
    const endpoint = await this.webhookService.updateEndpoint(id, merchantId, dto);

    if (!endpoint) {
      return { error: 'Endpoint not found' };
    }

    return {
      success: true,
      endpoint: {
        ...endpoint,
        secret: endpoint.secret.substring(0, 12) + '...',
      },
    };
  }

  /**
   * Delete webhook endpoint
   */
  @Delete('endpoints/:id')
  async deleteEndpoint(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.webhookService.deleteEndpoint(id, merchantId);

    return {
      success: deleted,
      message: deleted ? 'Endpoint deleted' : 'Endpoint not found',
    };
  }

  /**
   * Rotate webhook secret
   */
  @Post('endpoints/:id/rotate-secret')
  async rotateSecret(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const result = await this.webhookService.rotateSecret(id, merchantId);

    if (!result) {
      return { error: 'Endpoint not found' };
    }

    return {
      success: true,
      secret: result.secret,
      message: 'Save the new secret securely. It will not be shown again.',
    };
  }

  // ============================================================================
  // DELIVERY HISTORY
  // ============================================================================

  /**
   * Get delivery history for endpoint
   */
  @Get('endpoints/:id/deliveries')
  async getDeliveryHistory(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('status') status?: string,
  ) {
    return this.webhookService.getDeliveryHistory(id, merchantId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
      status,
    });
  }

  /**
   * Get delivery details
   */
  @Get('deliveries/:id')
  async getDelivery(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const delivery = await this.webhookService.getDelivery(id, merchantId);

    if (!delivery) {
      return { error: 'Delivery not found' };
    }

    return delivery;
  }

  /**
   * Retry failed delivery
   */
  @Post('deliveries/:id/retry')
  async retryDelivery(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const success = await this.webhookService.retryDelivery(id, merchantId);

    return {
      success,
      message: success ? 'Delivery retried' : 'Delivery not found or retry failed',
    };
  }

  // ============================================================================
  // TESTING
  // ============================================================================

  /**
   * Test webhook endpoint
   */
  @Post('endpoints/:id/test')
  async testEndpoint(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: TestWebhookDto,
  ) {
    const endpoint = await this.webhookService.getEndpoint(id, merchantId);

    if (!endpoint) {
      return { error: 'Endpoint not found' };
    }

    // Dispatch test event
    await this.webhookService.dispatch(
      dto.event || 'booking.created',
      dto.data || {
        test: true,
        message: 'This is a test webhook',
        timestamp: new Date().toISOString(),
      },
      merchantId,
    );

    return {
      success: true,
      message: 'Test webhook dispatched',
    };
  }

  // ============================================================================
  // STATISTICS
  // ============================================================================

  /**
   * Get webhook statistics
   */
  @Get('stats')
  async getStatistics(
    @MerchantId() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.webhookService.getStatistics(
      merchantId,
      startDate ? new Date(startDate) : undefined,
      endDate ? new Date(endDate) : undefined,
    );
  }

  // ============================================================================
  // AVAILABLE EVENTS
  // ============================================================================

  /**
   * Get available webhook events
   */
  @Get('events')
  getAvailableEvents() {
    return {
      events: [
        { event: 'booking.created', description: 'Triggered when a new booking is created' },
        { event: 'booking.accepted', description: 'Triggered when a driver accepts a booking' },
        { event: 'booking.arrived', description: 'Triggered when driver arrives at pickup' },
        { event: 'booking.started', description: 'Triggered when the ride starts' },
        { event: 'booking.completed', description: 'Triggered when the ride is completed' },
        { event: 'booking.cancelled', description: 'Triggered when a booking is cancelled' },
        { event: 'delivery.created', description: 'Triggered when a delivery is created' },
        { event: 'delivery.picked_up', description: 'Triggered when package is picked up' },
        { event: 'delivery.delivered', description: 'Triggered when package is delivered' },
        { event: 'delivery.cancelled', description: 'Triggered when delivery is cancelled' },
        { event: 'driver.online', description: 'Triggered when driver goes online' },
        { event: 'driver.offline', description: 'Triggered when driver goes offline' },
        { event: 'driver.location_updated', description: 'Triggered on driver location update' },
        { event: 'payment.completed', description: 'Triggered when payment is completed' },
        { event: 'payment.failed', description: 'Triggered when payment fails' },
        { event: 'payment.refunded', description: 'Triggered when payment is refunded' },
        { event: 'user.created', description: 'Triggered when new user signs up' },
        { event: 'user.updated', description: 'Triggered when user profile is updated' },
        { event: 'driver.created', description: 'Triggered when new driver signs up' },
        { event: 'driver.approved', description: 'Triggered when driver is approved' },
        { event: 'driver.suspended', description: 'Triggered when driver is suspended' },
      ],
    };
  }

  /**
   * Get signature verification guide
   */
  @Get('verify-guide')
  getVerificationGuide() {
    return {
      headers: {
        'X-Webhook-Signature': 'HMAC SHA256 signature of the payload',
        'X-Webhook-Event': 'The event type (e.g., booking.created)',
        'X-Webhook-Timestamp': 'ISO 8601 timestamp of when the webhook was sent',
      },
      verification: {
        step1: 'Get the X-Webhook-Signature header value',
        step2: 'Compute HMAC SHA256 of the raw request body using your webhook secret',
        step3: 'Compare the computed signature with the header value',
        example: {
          pseudocode: `
            expectedSignature = "sha256=" + hmacSha256(requestBody, webhookSecret)
            isValid = secureCompare(expectedSignature, headerSignature)
          `,
          nodejs: `
            const crypto = require('crypto');
            const expectedSignature = 'sha256=' +
              crypto.createHmac('sha256', webhookSecret)
                    .update(requestBody)
                    .digest('hex');
            const isValid = crypto.timingSafeEqual(
              Buffer.from(expectedSignature),
              Buffer.from(headerSignature)
            );
          `,
        },
      },
    };
  }
}
