import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

export enum TransactionType {
  CREDIT = 'credit',
  DEBIT = 'debit',
}

export enum TransactionReason {
  TOPUP = 'topup',
  BOOKING_PAYMENT = 'booking_payment',
  BOOKING_REFUND = 'booking_refund',
  DELIVERY_PAYMENT = 'delivery_payment',
  DELIVERY_REFUND = 'delivery_refund',
  WITHDRAWAL = 'withdrawal',
  REFERRAL_BONUS = 'referral_bonus',
  PROMO_CREDIT = 'promo_credit',
  CASHBACK = 'cashback',
  TRANSFER_IN = 'transfer_in',
  TRANSFER_OUT = 'transfer_out',
  DRIVER_EARNING = 'driver_earning',
  DRIVER_COMMISSION = 'driver_commission',
  ADJUSTMENT = 'adjustment',
}

export enum WithdrawalStatus {
  PENDING = 'pending',
  PROCESSING = 'processing',
  COMPLETED = 'completed',
  FAILED = 'failed',
  CANCELLED = 'cancelled',
}

@Injectable()
export class WalletService {
  private readonly logger = new Logger(WalletService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  // ============================================================================
  // WALLET MANAGEMENT
  // ============================================================================

  /**
   * Get or create wallet
   */
  async getOrCreateWallet(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
  ): Promise<any> {
    const existingWallet = await this.prisma.wallet.findFirst({
      where: { user_id: userId, user_type: userType },
    });

    if (existingWallet) {
      return existingWallet;
    }

    return this.prisma.wallet.create({
      data: {
        user_id: userId,
        user_type: userType,
        merchant_id: merchantId,
        balance: 0,
        currency: 'XOF',
        is_active: 1,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get wallet balance
   */
  async getBalance(userId: number, userType: 'user' | 'driver'): Promise<number> {
    const wallet = await this.prisma.wallet.findFirst({
      where: { user_id: userId, user_type: userType },
    });

    return Number(wallet?.balance) || 0;
  }

  /**
   * Get wallet details with recent transactions
   */
  async getWalletDetails(userId: number, userType: 'user' | 'driver'): Promise<any> {
    const wallet = await this.prisma.wallet.findFirst({
      where: { user_id: userId, user_type: userType },
    });

    if (!wallet) {
      throw new NotFoundException('Portefeuille non trouve');
    }

    const recentTransactions = await this.prisma.walletTransaction.findMany({
      where: { wallet_id: wallet.id },
      orderBy: { created_at: 'desc' },
      take: 10,
    });

    // Get pending withdrawals
    const pendingWithdrawals = await this.prisma.walletWithdrawal.aggregate({
      where: {
        wallet_id: wallet.id,
        status: { in: [WithdrawalStatus.PENDING, WithdrawalStatus.PROCESSING] },
      },
      _sum: { amount: true },
    });

    return {
      ...wallet,
      balance: Number(wallet.balance),
      pendingWithdrawals: Number(pendingWithdrawals._sum.amount) || 0,
      availableBalance:
        Number(wallet.balance) - (Number(pendingWithdrawals._sum.amount) || 0),
      recentTransactions,
    };
  }

  // ============================================================================
  // TRANSACTIONS
  // ============================================================================

  /**
   * Credit wallet
   */
  async credit(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    reason: TransactionReason,
    referenceId?: string,
    description?: string,
  ): Promise<any> {
    if (amount <= 0) {
      throw new BadRequestException('Le montant doit etre positif');
    }

    const wallet = await this.getOrCreateWallet(userId, userType, merchantId);

    const newBalance = Number(wallet.balance) + amount;

    // Update wallet balance
    await this.prisma.wallet.update({
      where: { id: wallet.id },
      data: { balance: newBalance, updated_at: new Date() },
    });

    // Create transaction record
    const transaction = await this.prisma.walletTransaction.create({
      data: {
        wallet_id: wallet.id,
        user_id: userId,
        user_type: userType,
        merchant_id: merchantId,
        type: TransactionType.CREDIT,
        reason,
        amount,
        balance_before: Number(wallet.balance),
        balance_after: newBalance,
        reference_id: referenceId,
        description: description || this.getTransactionDescription(reason, amount),
        status: 'completed',
        created_at: new Date(),
      },
    });

    this.logger.log(
      `Wallet credited: ${userType} #${userId}, amount: ${amount}, reason: ${reason}`,
    );

    return transaction;
  }

  /**
   * Debit wallet
   */
  async debit(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    reason: TransactionReason,
    referenceId?: string,
    description?: string,
  ): Promise<any> {
    if (amount <= 0) {
      throw new BadRequestException('Le montant doit etre positif');
    }

    const wallet = await this.prisma.wallet.findFirst({
      where: { user_id: userId, user_type: userType },
    });

    if (!wallet) {
      throw new NotFoundException('Portefeuille non trouve');
    }

    if (Number(wallet.balance) < amount) {
      throw new BadRequestException('Solde insuffisant');
    }

    const newBalance = Number(wallet.balance) - amount;

    // Update wallet balance
    await this.prisma.wallet.update({
      where: { id: wallet.id },
      data: { balance: newBalance, updated_at: new Date() },
    });

    // Create transaction record
    const transaction = await this.prisma.walletTransaction.create({
      data: {
        wallet_id: wallet.id,
        user_id: userId,
        user_type: userType,
        merchant_id: merchantId,
        type: TransactionType.DEBIT,
        reason,
        amount,
        balance_before: Number(wallet.balance),
        balance_after: newBalance,
        reference_id: referenceId,
        description: description || this.getTransactionDescription(reason, amount),
        status: 'completed',
        created_at: new Date(),
      },
    });

    this.logger.log(
      `Wallet debited: ${userType} #${userId}, amount: ${amount}, reason: ${reason}`,
    );

    return transaction;
  }

  /**
   * Get transaction history
   */
  async getTransactionHistory(
    userId: number,
    userType: 'user' | 'driver',
    options?: {
      type?: TransactionType;
      reason?: TransactionReason;
      startDate?: Date;
      endDate?: Date;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { user_id: userId, user_type: userType };

    if (options?.type) {
      where.type = options.type;
    }

    if (options?.reason) {
      where.reason = options.reason;
    }

    if (options?.startDate || options?.endDate) {
      where.created_at = {};
      if (options.startDate) {
        where.created_at.gte = options.startDate;
      }
      if (options.endDate) {
        where.created_at.lte = options.endDate;
      }
    }

    const [transactions, total] = await Promise.all([
      this.prisma.walletTransaction.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.walletTransaction.count({ where }),
    ]);

    return { data: transactions, total };
  }

  // ============================================================================
  // TOP-UP
  // ============================================================================

  /**
   * Create top-up request
   */
  async createTopUpRequest(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    paymentMethod: string,
  ): Promise<any> {
    if (amount < 100) {
      throw new BadRequestException('Le montant minimum est de 100 XOF');
    }

    const wallet = await this.getOrCreateWallet(userId, userType, merchantId);

    const topup = await this.prisma.walletTopup.create({
      data: {
        wallet_id: wallet.id,
        user_id: userId,
        user_type: userType,
        merchant_id: merchantId,
        amount,
        payment_method: paymentMethod,
        reference: this.generateReference('TOP'),
        status: 'pending',
        created_at: new Date(),
      },
    });

    return topup;
  }

  /**
   * Confirm top-up (after payment success)
   */
  async confirmTopUp(
    topupId: number,
    paymentReference: string,
  ): Promise<any> {
    const topup = await this.prisma.walletTopup.findUnique({
      where: { id: topupId },
    });

    if (!topup) {
      throw new NotFoundException('Rechargement non trouve');
    }

    if (topup.status !== 'pending') {
      throw new BadRequestException('Ce rechargement a deja ete traite');
    }

    // Update top-up status
    await this.prisma.walletTopup.update({
      where: { id: topupId },
      data: {
        status: 'completed',
        payment_reference: paymentReference,
        completed_at: new Date(),
      },
    });

    // Credit wallet
    await this.credit(
      topup.user_id,
      topup.user_type as 'user' | 'driver',
      topup.merchant_id,
      Number(topup.amount),
      TransactionReason.TOPUP,
      topup.reference,
      `Rechargement de ${topup.amount} XOF`,
    );

    // Notify user
    await this.notificationService.sendToUser(
      topup.user_id,
      topup.user_type as 'user' | 'driver',
      {
        title: 'Rechargement reussi',
        body: `Votre portefeuille a ete credite de ${topup.amount} XOF`,
        data: { type: 'wallet_topup', topup_id: String(topupId) },
      },
    );

    return topup;
  }

  // ============================================================================
  // WITHDRAWAL
  // ============================================================================

  /**
   * Request withdrawal
   */
  async requestWithdrawal(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    withdrawalMethod: string,
    accountDetails: {
      accountNumber?: string;
      accountName?: string;
      bankName?: string;
      mobileNumber?: string;
      provider?: string;
    },
  ): Promise<any> {
    const wallet = await this.prisma.wallet.findFirst({
      where: { user_id: userId, user_type: userType },
    });

    if (!wallet) {
      throw new NotFoundException('Portefeuille non trouve');
    }

    // Check minimum withdrawal
    const minWithdrawal = 1000;
    if (amount < minWithdrawal) {
      throw new BadRequestException(`Le montant minimum de retrait est de ${minWithdrawal} XOF`);
    }

    // Check available balance (excluding pending withdrawals)
    const pendingWithdrawals = await this.prisma.walletWithdrawal.aggregate({
      where: {
        wallet_id: wallet.id,
        status: { in: [WithdrawalStatus.PENDING, WithdrawalStatus.PROCESSING] },
      },
      _sum: { amount: true },
    });

    const availableBalance =
      Number(wallet.balance) - (Number(pendingWithdrawals._sum.amount) || 0);

    if (availableBalance < amount) {
      throw new BadRequestException('Solde disponible insuffisant');
    }

    const withdrawal = await this.prisma.walletWithdrawal.create({
      data: {
        wallet_id: wallet.id,
        user_id: userId,
        user_type: userType,
        merchant_id: merchantId,
        amount,
        withdrawal_method: withdrawalMethod,
        account_number: accountDetails.accountNumber,
        account_name: accountDetails.accountName,
        bank_name: accountDetails.bankName,
        mobile_number: accountDetails.mobileNumber,
        provider: accountDetails.provider,
        reference: this.generateReference('WTH'),
        status: WithdrawalStatus.PENDING,
        created_at: new Date(),
      },
    });

    this.logger.log(
      `Withdrawal requested: ${userType} #${userId}, amount: ${amount}`,
    );

    return withdrawal;
  }

  /**
   * Get withdrawal requests
   */
  async getWithdrawals(
    userId: number,
    userType: 'user' | 'driver',
    options?: { status?: WithdrawalStatus; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { user_id: userId, user_type: userType };
    if (options?.status) {
      where.status = options.status;
    }

    const [withdrawals, total] = await Promise.all([
      this.prisma.walletWithdrawal.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.walletWithdrawal.count({ where }),
    ]);

    return { data: withdrawals, total };
  }

  /**
   * Cancel withdrawal
   */
  async cancelWithdrawal(withdrawalId: number, userId: number): Promise<void> {
    const withdrawal = await this.prisma.walletWithdrawal.findFirst({
      where: { id: withdrawalId, user_id: userId },
    });

    if (!withdrawal) {
      throw new NotFoundException('Demande de retrait non trouvee');
    }

    if (withdrawal.status !== WithdrawalStatus.PENDING) {
      throw new BadRequestException('Cette demande ne peut pas etre annulee');
    }

    await this.prisma.walletWithdrawal.update({
      where: { id: withdrawalId },
      data: {
        status: WithdrawalStatus.CANCELLED,
        cancelled_at: new Date(),
      },
    });
  }

  // ============================================================================
  // ADMIN FUNCTIONS
  // ============================================================================

  /**
   * Process withdrawal (admin)
   */
  async processWithdrawal(
    withdrawalId: number,
    adminId: number,
  ): Promise<any> {
    const withdrawal = await this.prisma.walletWithdrawal.findUnique({
      where: { id: withdrawalId },
    });

    if (!withdrawal) {
      throw new NotFoundException('Demande de retrait non trouvee');
    }

    if (withdrawal.status !== WithdrawalStatus.PENDING) {
      throw new BadRequestException('Cette demande a deja ete traitee');
    }

    return this.prisma.walletWithdrawal.update({
      where: { id: withdrawalId },
      data: {
        status: WithdrawalStatus.PROCESSING,
        processed_by: adminId,
        processed_at: new Date(),
      },
    });
  }

  /**
   * Complete withdrawal (admin)
   */
  async completeWithdrawal(
    withdrawalId: number,
    adminId: number,
    transactionReference?: string,
  ): Promise<any> {
    const withdrawal = await this.prisma.walletWithdrawal.findUnique({
      where: { id: withdrawalId },
    });

    if (!withdrawal) {
      throw new NotFoundException('Demande de retrait non trouvee');
    }

    if (withdrawal.status !== WithdrawalStatus.PROCESSING) {
      throw new BadRequestException('Cette demande doit etre en cours de traitement');
    }

    // Debit wallet
    await this.debit(
      withdrawal.user_id,
      withdrawal.user_type as 'user' | 'driver',
      withdrawal.merchant_id,
      Number(withdrawal.amount),
      TransactionReason.WITHDRAWAL,
      withdrawal.reference,
      `Retrait de ${withdrawal.amount} XOF`,
    );

    const updated = await this.prisma.walletWithdrawal.update({
      where: { id: withdrawalId },
      data: {
        status: WithdrawalStatus.COMPLETED,
        transaction_reference: transactionReference,
        completed_by: adminId,
        completed_at: new Date(),
      },
    });

    // Notify user
    await this.notificationService.sendToUser(
      withdrawal.user_id,
      withdrawal.user_type as 'user' | 'driver',
      {
        title: 'Retrait effectue',
        body: `Votre retrait de ${withdrawal.amount} XOF a ete effectue`,
        data: { type: 'withdrawal_completed', withdrawal_id: String(withdrawalId) },
      },
    );

    return updated;
  }

  /**
   * Reject withdrawal (admin)
   */
  async rejectWithdrawal(
    withdrawalId: number,
    adminId: number,
    reason: string,
  ): Promise<any> {
    const withdrawal = await this.prisma.walletWithdrawal.findUnique({
      where: { id: withdrawalId },
    });

    if (!withdrawal) {
      throw new NotFoundException('Demande de retrait non trouvee');
    }

    const updated = await this.prisma.walletWithdrawal.update({
      where: { id: withdrawalId },
      data: {
        status: WithdrawalStatus.FAILED,
        rejection_reason: reason,
        rejected_by: adminId,
        rejected_at: new Date(),
      },
    });

    // Notify user
    await this.notificationService.sendToUser(
      withdrawal.user_id,
      withdrawal.user_type as 'user' | 'driver',
      {
        title: 'Retrait refuse',
        body: `Votre demande de retrait a ete refusee. Raison: ${reason}`,
        data: { type: 'withdrawal_rejected', withdrawal_id: String(withdrawalId) },
      },
    );

    return updated;
  }

  /**
   * Get pending withdrawals (admin)
   */
  async getPendingWithdrawals(
    merchantId: number,
    options?: { page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where = {
      merchant_id: merchantId,
      status: { in: [WithdrawalStatus.PENDING, WithdrawalStatus.PROCESSING] },
    };

    const [withdrawals, total] = await Promise.all([
      this.prisma.walletWithdrawal.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'asc' },
      }),
      this.prisma.walletWithdrawal.count({ where }),
    ]);

    return { data: withdrawals, total };
  }

  /**
   * Manual adjustment (admin)
   */
  async manualAdjustment(
    adminId: number,
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    type: TransactionType,
    reason: string,
  ): Promise<any> {
    if (type === TransactionType.CREDIT) {
      return this.credit(
        userId,
        userType,
        merchantId,
        Math.abs(amount),
        TransactionReason.ADJUSTMENT,
        `ADJ-${adminId}-${Date.now()}`,
        reason,
      );
    } else {
      return this.debit(
        userId,
        userType,
        merchantId,
        Math.abs(amount),
        TransactionReason.ADJUSTMENT,
        `ADJ-${adminId}-${Date.now()}`,
        reason,
      );
    }
  }

  // ============================================================================
  // TRANSFER
  // ============================================================================

  /**
   * Transfer between wallets
   */
  async transfer(
    fromUserId: number,
    fromUserType: 'user' | 'driver',
    toUserId: number,
    toUserType: 'user' | 'driver',
    merchantId: number,
    amount: number,
    description?: string,
  ): Promise<{ debitTransaction: any; creditTransaction: any }> {
    if (fromUserId === toUserId && fromUserType === toUserType) {
      throw new BadRequestException('Impossible de transferer vers le meme portefeuille');
    }

    const reference = this.generateReference('TRF');

    // Debit source
    const debitTransaction = await this.debit(
      fromUserId,
      fromUserType,
      merchantId,
      amount,
      TransactionReason.TRANSFER_OUT,
      reference,
      description || `Transfert vers ${toUserType} #${toUserId}`,
    );

    // Credit destination
    const creditTransaction = await this.credit(
      toUserId,
      toUserType,
      merchantId,
      amount,
      TransactionReason.TRANSFER_IN,
      reference,
      description || `Transfert de ${fromUserType} #${fromUserId}`,
    );

    return { debitTransaction, creditTransaction };
  }

  // ============================================================================
  // DRIVER EARNINGS
  // ============================================================================

  /**
   * Add driver earning
   */
  async addDriverEarning(
    driverId: number,
    merchantId: number,
    bookingId: number,
    totalFare: number,
    commissionRate: number,
  ): Promise<any> {
    const commission = Math.round((totalFare * commissionRate) / 100);
    const driverEarning = totalFare - commission;

    // Credit driver wallet
    const transaction = await this.credit(
      driverId,
      'driver',
      merchantId,
      driverEarning,
      TransactionReason.DRIVER_EARNING,
      `BK-${bookingId}`,
      `Gain course #${bookingId}: ${totalFare} - ${commission} (commission)`,
    );

    // Record commission
    await this.prisma.walletTransaction.create({
      data: {
        wallet_id: transaction.wallet_id,
        user_id: driverId,
        user_type: 'driver',
        merchant_id: merchantId,
        type: TransactionType.DEBIT,
        reason: TransactionReason.DRIVER_COMMISSION,
        amount: commission,
        balance_before: transaction.balance_after,
        balance_after: transaction.balance_after,
        reference_id: `COM-BK-${bookingId}`,
        description: `Commission course #${bookingId}: ${commissionRate}%`,
        status: 'completed',
        created_at: new Date(),
      },
    });

    return {
      totalFare,
      commission,
      driverEarning,
      transaction,
    };
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  private generateReference(prefix: string): string {
    const timestamp = Date.now().toString(36).toUpperCase();
    const random = Math.random().toString(36).substring(2, 6).toUpperCase();
    return `${prefix}${timestamp}${random}`;
  }

  private getTransactionDescription(reason: TransactionReason, amount: number): string {
    const descriptions: Record<TransactionReason, string> = {
      [TransactionReason.TOPUP]: `Rechargement de ${amount} XOF`,
      [TransactionReason.BOOKING_PAYMENT]: `Paiement course: ${amount} XOF`,
      [TransactionReason.BOOKING_REFUND]: `Remboursement course: ${amount} XOF`,
      [TransactionReason.DELIVERY_PAYMENT]: `Paiement livraison: ${amount} XOF`,
      [TransactionReason.DELIVERY_REFUND]: `Remboursement livraison: ${amount} XOF`,
      [TransactionReason.WITHDRAWAL]: `Retrait: ${amount} XOF`,
      [TransactionReason.REFERRAL_BONUS]: `Bonus parrainage: ${amount} XOF`,
      [TransactionReason.PROMO_CREDIT]: `Credit promotionnel: ${amount} XOF`,
      [TransactionReason.CASHBACK]: `Cashback: ${amount} XOF`,
      [TransactionReason.TRANSFER_IN]: `Transfert recu: ${amount} XOF`,
      [TransactionReason.TRANSFER_OUT]: `Transfert envoye: ${amount} XOF`,
      [TransactionReason.DRIVER_EARNING]: `Gain chauffeur: ${amount} XOF`,
      [TransactionReason.DRIVER_COMMISSION]: `Commission: ${amount} XOF`,
      [TransactionReason.ADJUSTMENT]: `Ajustement: ${amount} XOF`,
    };

    return descriptions[reason] || `Transaction: ${amount} XOF`;
  }
}
