import { Injectable, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../shared/database/prisma.service';
import { UpdateProfileDto } from './dto/update-profile.dto';

@Injectable()
export class UserService {
  constructor(private prisma: PrismaService) {}

  async getProfile(userId: number) {
    const user = await this.prisma.user.findUnique({
      where: { id: userId },
    });

    if (!user) {
      throw new NotFoundException('User not found');
    }

    const { password, otp, otp_expires_at, ...profile } = user;

    return {
      message: 'User profile',
      data: profile,
    };
  }

  async updateProfile(userId: number, dto: UpdateProfileDto) {
    const user = await this.prisma.user.update({
      where: { id: userId },
      data: {
        first_name: dto.first_name,
        last_name: dto.last_name,
        email: dto.email,
        profile_image: dto.profile_image,
      },
    });

    const { password, otp, otp_expires_at, ...profile } = user;

    return {
      message: 'Profile updated successfully',
      data: profile,
    };
  }

  async getBookings(userId: number, page: number, limit: number) {
    const skip = (page - 1) * limit;

    const [bookings, total] = await Promise.all([
      this.prisma.booking.findMany({
        where: { user_id: userId },
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
        include: {
          driver: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              phoneNumber: true,
              profile_image: true,
            },
          },
        },
      }),
      this.prisma.booking.count({ where: { user_id: userId } }),
    ]);

    return {
      message: 'User bookings',
      data: {
        bookings,
        pagination: {
          page,
          limit,
          total,
          total_pages: Math.ceil(total / limit),
        },
      },
    };
  }

  async getWallet(userId: number, page: number, limit: number) {
    const skip = (page - 1) * limit;

    // Get wallet balance (sum of all transactions)
    const balanceResult = await this.prisma.userWalletTransaction.aggregate({
      where: { user_id: userId },
      _sum: {
        amount: true,
      },
    });

    const balance = balanceResult._sum.amount || 0;

    // Get transactions
    const [transactions, total] = await Promise.all([
      this.prisma.userWalletTransaction.findMany({
        where: { user_id: userId },
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
      }),
      this.prisma.userWalletTransaction.count({ where: { user_id: userId } }),
    ]);

    return {
      message: 'User wallet',
      data: {
        balance,
        transactions,
        pagination: {
          page,
          limit,
          total,
          total_pages: Math.ceil(total / limit),
        },
      },
    };
  }

  async getFavoriteDrivers(userId: number) {
    // TODO: Implement favorite drivers logic
    // This requires a favorite_drivers table

    return {
      message: 'Favorite drivers',
      data: [],
    };
  }
}
