import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as sharp from 'sharp';
import * as path from 'path';
import * as crypto from 'crypto';
import { StorageProviderFactory } from './providers/storage.factory';
import { UploadResult } from './providers/storage.interface';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface UploadFileOptions {
  provider?: string;
  folder?: string;
  filename?: string;
  isPublic?: boolean;
  merchantId?: number;
  userId?: number;
  entityType?: string;
  entityId?: number;
  resize?: {
    width?: number;
    height?: number;
    fit?: 'cover' | 'contain' | 'fill' | 'inside' | 'outside';
  };
  compress?: boolean;
  quality?: number;
}

export interface FileUploadResult extends UploadResult {
  id?: number;
  originalName?: string;
  thumbnailUrl?: string;
  variants?: Record<string, string>;
}

@Injectable()
export class StorageService {
  private readonly logger = new Logger(StorageService.name);

  constructor(
    private configService: ConfigService,
    private prisma: PrismaService,
    private providerFactory: StorageProviderFactory,
  ) {}

  /**
   * Upload file
   */
  async uploadFile(
    buffer: Buffer,
    originalName: string,
    mimeType: string,
    options?: UploadFileOptions,
  ): Promise<FileUploadResult> {
    const providerName = options?.provider;
    const provider = providerName
      ? this.providerFactory.getProvider(providerName)
      : this.providerFactory.getBestProviderForType(mimeType);

    if (!provider) {
      return { success: false, error: 'No storage provider available' };
    }

    // Process image if needed
    let processedBuffer = buffer;
    if (mimeType.startsWith('image/') && (options?.resize || options?.compress)) {
      processedBuffer = await this.processImage(buffer, mimeType, options);
    }

    // Generate unique filename
    const ext = path.extname(originalName);
    const filename = options?.filename ||
      `${Date.now()}_${crypto.randomBytes(8).toString('hex')}${ext}`;

    // Upload to provider
    const result = await provider.upload(processedBuffer, filename, {
      folder: options?.folder || this.getDefaultFolder(mimeType),
      contentType: mimeType,
      isPublic: options?.isPublic ?? true,
    });

    if (!result.success) {
      return result;
    }

    // Generate thumbnail for images
    let thumbnailUrl: string | undefined;
    if (mimeType.startsWith('image/') && result.url) {
      thumbnailUrl = await this.generateThumbnail(buffer, filename, mimeType, provider.name, options);
    }

    // Save to database
    let fileId: number | undefined;
    if (options?.merchantId || options?.userId) {
      try {
        const file = await this.prisma.file.create({
          data: {
            merchant_id: options?.merchantId,
            user_id: options?.userId,
            original_name: originalName,
            filename,
            path: result.key,
            url: result.url,
            thumbnail_url: thumbnailUrl,
            mime_type: mimeType,
            size: result.size || buffer.length,
            provider: provider.name,
            entity_type: options?.entityType,
            entity_id: options?.entityId,
            is_public: options?.isPublic ?? true,
            created_at: new Date(),
          },
        });
        fileId = file.id;
      } catch (error) {
        this.logger.error(`Failed to save file record: ${error.message}`);
      }
    }

    return {
      ...result,
      id: fileId,
      originalName,
      thumbnailUrl,
    };
  }

  /**
   * Upload multiple files
   */
  async uploadFiles(
    files: Array<{ buffer: Buffer; originalName: string; mimeType: string }>,
    options?: UploadFileOptions,
  ): Promise<FileUploadResult[]> {
    const results = await Promise.all(
      files.map(file => this.uploadFile(file.buffer, file.originalName, file.mimeType, options)),
    );
    return results;
  }

  /**
   * Upload from URL
   */
  async uploadFromUrl(
    url: string,
    options?: UploadFileOptions,
  ): Promise<FileUploadResult> {
    const provider = this.providerFactory.getProvider(options?.provider);

    if (!provider || !provider.uploadFromUrl) {
      return { success: false, error: 'Provider does not support URL upload' };
    }

    const filename = options?.filename ||
      `${Date.now()}_${crypto.randomBytes(8).toString('hex')}`;

    return provider.uploadFromUrl(url, filename, {
      folder: options?.folder,
      isPublic: options?.isPublic ?? true,
    });
  }

  /**
   * Delete file
   */
  async deleteFile(key: string, provider?: string): Promise<boolean> {
    const storageProvider = this.providerFactory.getProvider(provider);

    if (!storageProvider) {
      return false;
    }

    const result = await storageProvider.delete(key);

    // Also delete from database
    if (result.success) {
      try {
        await this.prisma.file.deleteMany({
          where: { path: key },
        });
      } catch (error) {
        this.logger.error(`Failed to delete file record: ${error.message}`);
      }
    }

    return result.success;
  }

  /**
   * Delete file by ID
   */
  async deleteFileById(fileId: number): Promise<boolean> {
    try {
      const file = await this.prisma.file.findUnique({
        where: { id: fileId },
      });

      if (!file) return false;

      return this.deleteFile(file.path, file.provider);
    } catch (error) {
      this.logger.error(`Failed to delete file: ${error.message}`);
      return false;
    }
  }

  /**
   * Get signed URL for private files
   */
  async getSignedUrl(key: string, provider?: string, expiresIn?: number): Promise<string | null> {
    const storageProvider = this.providerFactory.getProvider(provider);

    if (!storageProvider || !storageProvider.getSignedUrl) {
      return null;
    }

    try {
      return await storageProvider.getSignedUrl(key, { expiresIn });
    } catch {
      return null;
    }
  }

  /**
   * Get files for entity
   */
  async getFilesForEntity(entityType: string, entityId: number): Promise<any[]> {
    return this.prisma.file.findMany({
      where: {
        entity_type: entityType,
        entity_id: entityId,
      },
      orderBy: { created_at: 'desc' },
    });
  }

  /**
   * Get user files
   */
  async getUserFiles(
    userId: number,
    options?: { mimeType?: string; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 50;
    const skip = (page - 1) * limit;

    const where: any = { user_id: userId };
    if (options?.mimeType) {
      where.mime_type = { startsWith: options.mimeType };
    }

    const [data, total] = await Promise.all([
      this.prisma.file.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.file.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Process image (resize, compress)
   */
  private async processImage(
    buffer: Buffer,
    mimeType: string,
    options?: UploadFileOptions,
  ): Promise<Buffer> {
    let sharpInstance = sharp(buffer);

    // Resize
    if (options?.resize) {
      sharpInstance = sharpInstance.resize({
        width: options.resize.width,
        height: options.resize.height,
        fit: options.resize.fit || 'inside',
        withoutEnlargement: true,
      });
    }

    // Compress based on format
    const quality = options?.quality || 80;
    if (mimeType === 'image/jpeg' || mimeType === 'image/jpg') {
      sharpInstance = sharpInstance.jpeg({ quality });
    } else if (mimeType === 'image/png') {
      sharpInstance = sharpInstance.png({ quality });
    } else if (mimeType === 'image/webp') {
      sharpInstance = sharpInstance.webp({ quality });
    }

    return sharpInstance.toBuffer();
  }

  /**
   * Generate thumbnail
   */
  private async generateThumbnail(
    buffer: Buffer,
    originalFilename: string,
    mimeType: string,
    providerName: string,
    options?: UploadFileOptions,
  ): Promise<string | undefined> {
    try {
      const thumbnailBuffer = await sharp(buffer)
        .resize(200, 200, { fit: 'cover' })
        .jpeg({ quality: 70 })
        .toBuffer();

      const ext = path.extname(originalFilename);
      const thumbFilename = `thumb_${originalFilename.replace(ext, '.jpg')}`;

      const provider = this.providerFactory.getProvider(providerName);
      if (!provider) return undefined;

      const result = await provider.upload(thumbnailBuffer, thumbFilename, {
        folder: options?.folder ? `${options.folder}/thumbnails` : 'thumbnails',
        contentType: 'image/jpeg',
        isPublic: options?.isPublic ?? true,
      });

      return result.url;
    } catch (error) {
      this.logger.error(`Failed to generate thumbnail: ${error.message}`);
      return undefined;
    }
  }

  /**
   * Get default folder based on mime type
   */
  private getDefaultFolder(mimeType: string): string {
    if (mimeType.startsWith('image/')) return 'images';
    if (mimeType.startsWith('video/')) return 'videos';
    if (mimeType.startsWith('audio/')) return 'audio';
    if (mimeType === 'application/pdf') return 'documents';
    return 'files';
  }

  /**
   * Get available providers
   */
  getAvailableProviders() {
    return this.providerFactory.getProviderInfo();
  }

  // ============================================================================
  // SPECIALIZED UPLOAD METHODS
  // ============================================================================

  /**
   * Upload driver document
   */
  async uploadDriverDocument(
    buffer: Buffer,
    originalName: string,
    mimeType: string,
    driverId: number,
    documentType: string,
  ): Promise<FileUploadResult> {
    return this.uploadFile(buffer, originalName, mimeType, {
      folder: `drivers/${driverId}/documents`,
      entityType: 'driver_document',
      entityId: driverId,
      compress: true,
    });
  }

  /**
   * Upload user profile picture
   */
  async uploadProfilePicture(
    buffer: Buffer,
    originalName: string,
    mimeType: string,
    userId: number,
    userType: 'user' | 'driver',
  ): Promise<FileUploadResult> {
    return this.uploadFile(buffer, originalName, mimeType, {
      folder: `${userType}s/${userId}/profile`,
      entityType: `${userType}_profile`,
      entityId: userId,
      resize: { width: 500, height: 500, fit: 'cover' },
      compress: true,
      quality: 85,
    });
  }

  /**
   * Upload vehicle photo
   */
  async uploadVehiclePhoto(
    buffer: Buffer,
    originalName: string,
    mimeType: string,
    vehicleId: number,
  ): Promise<FileUploadResult> {
    return this.uploadFile(buffer, originalName, mimeType, {
      folder: `vehicles/${vehicleId}`,
      entityType: 'vehicle_photo',
      entityId: vehicleId,
      resize: { width: 1200, height: 800, fit: 'inside' },
      compress: true,
    });
  }

  /**
   * Upload chat attachment
   */
  async uploadChatAttachment(
    buffer: Buffer,
    originalName: string,
    mimeType: string,
    chatId: number,
  ): Promise<FileUploadResult> {
    return this.uploadFile(buffer, originalName, mimeType, {
      folder: `chats/${chatId}`,
      entityType: 'chat_attachment',
      entityId: chatId,
      compress: mimeType.startsWith('image/'),
    });
  }
}
