import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { StorageProvider } from './storage.interface';
import { S3Provider } from './s3.provider';
import { LocalStorageProvider } from './local.provider';
import { CloudinaryProvider } from './cloudinary.provider';

@Injectable()
export class StorageProviderFactory implements OnModuleInit {
  private readonly logger = new Logger(StorageProviderFactory.name);
  private providers: Map<string, StorageProvider> = new Map();
  private defaultProvider: string;

  constructor(
    private configService: ConfigService,
    private s3Provider: S3Provider,
    private localProvider: LocalStorageProvider,
    private cloudinaryProvider: CloudinaryProvider,
  ) {
    this.defaultProvider = this.configService.get<string>('STORAGE_DEFAULT_PROVIDER') || 'local';
  }

  onModuleInit() {
    // Register all providers
    this.registerProvider(this.s3Provider);
    this.registerProvider(this.localProvider);
    this.registerProvider(this.cloudinaryProvider);

    this.logger.log(`Registered ${this.providers.size} storage providers. Default: ${this.defaultProvider}`);
  }

  /**
   * Register a provider
   */
  registerProvider(provider: StorageProvider): void {
    this.providers.set(provider.name, provider);
    this.logger.debug(`Registered storage provider: ${provider.name}`);
  }

  /**
   * Get provider by name
   */
  getProvider(name?: string): StorageProvider | undefined {
    return this.providers.get(name || this.defaultProvider);
  }

  /**
   * Get default provider
   */
  getDefaultProvider(): StorageProvider | undefined {
    return this.getProvider(this.defaultProvider);
  }

  /**
   * Get all providers
   */
  getAllProviders(): StorageProvider[] {
    return Array.from(this.providers.values());
  }

  /**
   * Get provider info for display
   */
  getProviderInfo(): Array<{ name: string; displayName: string; isDefault: boolean }> {
    return this.getAllProviders().map(provider => ({
      name: provider.name,
      displayName: provider.displayName,
      isDefault: provider.name === this.defaultProvider,
    }));
  }

  /**
   * Get best provider for file type
   */
  getBestProviderForType(mimeType: string): StorageProvider {
    // Use Cloudinary for images (better optimization)
    if (mimeType.startsWith('image/') && this.providers.has('cloudinary')) {
      const cloudinary = this.providers.get('cloudinary');
      if (cloudinary) return cloudinary;
    }

    // Use S3 for large files and videos
    if (
      mimeType.startsWith('video/') ||
      mimeType === 'application/pdf' ||
      mimeType.startsWith('application/')
    ) {
      const s3 = this.providers.get('s3');
      if (s3) return s3;
    }

    // Default provider for everything else
    return this.getDefaultProvider();
  }
}
