import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import {
  S3Client,
  PutObjectCommand,
  DeleteObjectCommand,
  GetObjectCommand,
  HeadObjectCommand,
  CopyObjectCommand,
} from '@aws-sdk/client-s3';
import { getSignedUrl } from '@aws-sdk/s3-request-presigner';
import {
  StorageProvider,
  UploadOptions,
  UploadResult,
  DeleteResult,
  SignedUrlOptions,
} from './storage.interface';
import axios from 'axios';

@Injectable()
export class S3Provider implements StorageProvider {
  readonly name = 's3';
  readonly displayName = 'Amazon S3';

  private readonly logger = new Logger(S3Provider.name);
  private client: S3Client;
  private bucket: string;
  private region: string;
  private cdnUrl: string;

  constructor(private configService: ConfigService) {
    this.region = this.configService.get<string>('AWS_REGION') || 'us-east-1';
    this.bucket = this.configService.get<string>('AWS_S3_BUCKET');
    this.cdnUrl = this.configService.get<string>('AWS_CLOUDFRONT_URL');

    this.client = new S3Client({
      region: this.region,
      credentials: {
        accessKeyId: this.configService.get<string>('AWS_ACCESS_KEY_ID'),
        secretAccessKey: this.configService.get<string>('AWS_SECRET_ACCESS_KEY'),
      },
    });
  }

  async upload(
    buffer: Buffer,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult> {
    try {
      const key = options?.folder
        ? `${options.folder}/${filename}`
        : filename;

      const command = new PutObjectCommand({
        Bucket: this.bucket,
        Key: key,
        Body: buffer,
        ContentType: options?.contentType || 'application/octet-stream',
        ACL: options?.isPublic ? 'public-read' : 'private',
        Metadata: options?.metadata,
      });

      await this.client.send(command);

      const url = this.cdnUrl
        ? `${this.cdnUrl}/${key}`
        : `https://${this.bucket}.s3.${this.region}.amazonaws.com/${key}`;

      return {
        success: true,
        url,
        key,
        bucket: this.bucket,
        size: buffer.length,
        contentType: options?.contentType,
      };
    } catch (error) {
      this.logger.error(`S3 upload error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async uploadFromUrl(
    url: string,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult> {
    try {
      const response = await axios.get(url, { responseType: 'arraybuffer' });
      const buffer = Buffer.from(response.data);
      const contentType = response.headers['content-type'] || options?.contentType;

      return this.upload(buffer, filename, { ...options, contentType });
    } catch (error) {
      this.logger.error(`S3 uploadFromUrl error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async delete(key: string): Promise<DeleteResult> {
    try {
      const command = new DeleteObjectCommand({
        Bucket: this.bucket,
        Key: key,
      });

      await this.client.send(command);
      return { success: true };
    } catch (error) {
      this.logger.error(`S3 delete error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async getSignedUrl(key: string, options?: SignedUrlOptions): Promise<string> {
    try {
      const command = new GetObjectCommand({
        Bucket: this.bucket,
        Key: key,
      });

      const expiresIn = options?.expiresIn || 3600; // 1 hour default
      return await getSignedUrl(this.client, command, { expiresIn });
    } catch (error) {
      this.logger.error(`S3 getSignedUrl error: ${error.message}`);
      throw error;
    }
  }

  async getUploadSignedUrl(key: string, options?: SignedUrlOptions): Promise<string> {
    try {
      const command = new PutObjectCommand({
        Bucket: this.bucket,
        Key: key,
        ContentType: options?.contentType || 'application/octet-stream',
      });

      const expiresIn = options?.expiresIn || 3600;
      return await getSignedUrl(this.client, command, { expiresIn });
    } catch (error) {
      this.logger.error(`S3 getUploadSignedUrl error: ${error.message}`);
      throw error;
    }
  }

  async exists(key: string): Promise<boolean> {
    try {
      const command = new HeadObjectCommand({
        Bucket: this.bucket,
        Key: key,
      });

      await this.client.send(command);
      return true;
    } catch (error) {
      if (error.name === 'NotFound') return false;
      throw error;
    }
  }

  async getMetadata(key: string): Promise<Record<string, any>> {
    try {
      const command = new HeadObjectCommand({
        Bucket: this.bucket,
        Key: key,
      });

      const response = await this.client.send(command);
      return {
        contentType: response.ContentType,
        contentLength: response.ContentLength,
        lastModified: response.LastModified,
        metadata: response.Metadata,
        etag: response.ETag,
      };
    } catch (error) {
      this.logger.error(`S3 getMetadata error: ${error.message}`);
      throw error;
    }
  }

  async copy(sourceKey: string, destKey: string): Promise<UploadResult> {
    try {
      const command = new CopyObjectCommand({
        Bucket: this.bucket,
        CopySource: `${this.bucket}/${sourceKey}`,
        Key: destKey,
      });

      await this.client.send(command);

      const url = this.cdnUrl
        ? `${this.cdnUrl}/${destKey}`
        : `https://${this.bucket}.s3.${this.region}.amazonaws.com/${destKey}`;

      return {
        success: true,
        url,
        key: destKey,
        bucket: this.bucket,
      };
    } catch (error) {
      this.logger.error(`S3 copy error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }
}
