import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { PrismaService } from '../../common/prisma/prisma.service';
import { SmsProviderFactory } from './providers/provider.factory';
import { SendSmsResult, SendBulkSmsResult } from './providers/provider.interface';

export interface SmsOptions {
  provider?: string;
  from?: string;
  reference?: string;
  saveToLog?: boolean;
}

@Injectable()
export class SmsService {
  private readonly logger = new Logger(SmsService.name);
  private defaultProvider: string;

  constructor(
    private configService: ConfigService,
    private prisma: PrismaService,
    private providerFactory: SmsProviderFactory,
  ) {
    this.defaultProvider = this.configService.get<string>('SMS_DEFAULT_PROVIDER') || 'twilio';
  }

  /**
   * Send SMS
   */
  async send(
    to: string,
    message: string,
    options?: SmsOptions,
  ): Promise<SendSmsResult> {
    const providerName = options?.provider || this.defaultProvider;
    const provider = this.providerFactory.getProvider(providerName);

    if (!provider) {
      this.logger.error(`SMS provider not found: ${providerName}`);
      return { success: false, message: `Provider not found: ${providerName}` };
    }

    const result = await provider.sendSms({
      to,
      message,
      from: options?.from,
      reference: options?.reference,
    });

    // Log to database
    if (options?.saveToLog !== false) {
      await this.logSms({
        provider: providerName,
        to,
        message,
        messageId: result.messageId,
        status: result.success ? 'sent' : 'failed',
        error: result.message,
        cost: result.cost,
        reference: options?.reference,
      });
    }

    if (result.success) {
      this.logger.log(`SMS sent via ${providerName} to ${to}`);
    } else {
      this.logger.error(`SMS failed via ${providerName} to ${to}: ${result.message}`);
    }

    return result;
  }

  /**
   * Send SMS with automatic provider selection based on country
   */
  async sendSmart(
    to: string,
    message: string,
    countryCode: string,
    options?: SmsOptions,
  ): Promise<SendSmsResult> {
    const provider = this.providerFactory.getBestProvider(countryCode);

    if (!provider) {
      return { success: false, message: `No provider available for country: ${countryCode}` };
    }

    return this.send(to, message, { ...options, provider: provider.name });
  }

  /**
   * Send bulk SMS
   */
  async sendBulk(
    recipients: string[],
    message: string,
    options?: SmsOptions,
  ): Promise<SendBulkSmsResult> {
    const providerName = options?.provider || this.defaultProvider;
    const provider = this.providerFactory.getProvider(providerName);

    if (!provider) {
      return {
        success: false,
        totalSent: 0,
        totalFailed: recipients.length,
        results: recipients.map(to => ({
          to,
          status: 'failed' as const,
          error: `Provider not found: ${providerName}`,
        })),
      };
    }

    if (provider.sendBulkSms) {
      const result = await provider.sendBulkSms({
        recipients,
        message,
        from: options?.from,
        reference: options?.reference,
      });

      // Log all messages
      for (const r of result.results) {
        await this.logSms({
          provider: providerName,
          to: r.to,
          message,
          messageId: r.messageId,
          status: r.status,
          error: r.error,
          reference: options?.reference,
        });
      }

      return result;
    }

    // Fallback to individual sends
    const results: SendBulkSmsResult['results'] = [];
    let totalSent = 0;
    let totalFailed = 0;

    for (const to of recipients) {
      const result = await this.send(to, message, options);
      if (result.success) {
        totalSent++;
        results.push({ to, messageId: result.messageId, status: 'sent' });
      } else {
        totalFailed++;
        results.push({ to, status: 'failed', error: result.message });
      }
    }

    return { success: totalSent > 0, totalSent, totalFailed, results };
  }

  /**
   * Send OTP
   */
  async sendOtp(to: string, otp: string, options?: SmsOptions): Promise<SendSmsResult> {
    const appName = this.configService.get<string>('APP_NAME') || 'MonkAPI';
    const message = `Votre code de verification ${appName} est: ${otp}. Ce code expire dans 5 minutes.`;

    return this.send(to, message, options);
  }

  /**
   * Send booking notification
   */
  async sendBookingNotification(
    to: string,
    bookingNumber: string,
    driverName: string,
    driverPhone: string,
    vehicleInfo: string,
    options?: SmsOptions,
  ): Promise<SendSmsResult> {
    const message = `Votre course #${bookingNumber} est confirmee.\nChauffeur: ${driverName}\nTel: ${driverPhone}\nVehicule: ${vehicleInfo}`;

    return this.send(to, message, { ...options, reference: `booking_${bookingNumber}` });
  }

  /**
   * Send driver arrival notification
   */
  async sendDriverArrivalNotification(
    to: string,
    bookingNumber: string,
    driverName: string,
    options?: SmsOptions,
  ): Promise<SendSmsResult> {
    const message = `Votre chauffeur ${driverName} est arrive pour la course #${bookingNumber}.`;

    return this.send(to, message, { ...options, reference: `arrival_${bookingNumber}` });
  }

  /**
   * Send payment confirmation
   */
  async sendPaymentConfirmation(
    to: string,
    amount: number,
    currency: string,
    reference: string,
    options?: SmsOptions,
  ): Promise<SendSmsResult> {
    const message = `Paiement de ${amount} ${currency} confirme. Ref: ${reference}. Merci pour votre confiance.`;

    return this.send(to, message, { ...options, reference });
  }

  /**
   * Check delivery status
   */
  async checkDeliveryStatus(
    messageId: string,
    providerName?: string,
  ): Promise<{ success: boolean; status: string; deliveredAt?: Date }> {
    const provider = this.providerFactory.getProvider(providerName || this.defaultProvider);

    if (!provider || !provider.checkDeliveryStatus) {
      return { success: false, status: 'unknown' };
    }

    return provider.checkDeliveryStatus(messageId);
  }

  /**
   * Get provider balance
   */
  async getProviderBalance(
    providerName?: string,
  ): Promise<{ balance: number; currency: string }> {
    const provider = this.providerFactory.getProvider(providerName || this.defaultProvider);

    if (!provider || !provider.getBalance) {
      return { balance: 0, currency: 'USD' };
    }

    return provider.getBalance();
  }

  /**
   * Get SMS logs
   */
  async getSmsLogs(
    merchantId: number,
    options?: {
      status?: string;
      provider?: string;
      startDate?: Date;
      endDate?: Date;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 50;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };

    if (options?.status) where.status = options.status;
    if (options?.provider) where.provider = options.provider;
    if (options?.startDate || options?.endDate) {
      where.created_at = {};
      if (options.startDate) where.created_at.gte = options.startDate;
      if (options.endDate) where.created_at.lte = options.endDate;
    }

    const [logs, total] = await Promise.all([
      this.prisma.smsLog.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.smsLog.count({ where }),
    ]);

    return { data: logs, total };
  }

  /**
   * Log SMS
   */
  private async logSms(data: {
    provider: string;
    to: string;
    message: string;
    messageId?: string;
    status: string;
    error?: string;
    cost?: number;
    reference?: string;
  }): Promise<void> {
    try {
      await this.prisma.smsLog.create({
        data: {
          provider: data.provider,
          recipient: data.to,
          message: data.message,
          message_id: data.messageId,
          status: data.status,
          error: data.error,
          cost: data.cost,
          reference: data.reference,
          created_at: new Date(),
        },
      });
    } catch (error) {
      this.logger.error(`Failed to log SMS: ${error.message}`);
    }
  }

  /**
   * Get available providers
   */
  getAvailableProviders() {
    return this.providerFactory.getProviderInfo();
  }
}
