import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as twilio from 'twilio';
import {
  SmsProvider,
  SendSmsParams,
  SendSmsResult,
  SendBulkSmsParams,
  SendBulkSmsResult,
  DeliveryStatusResult,
} from './provider.interface';

@Injectable()
export class TwilioProvider implements SmsProvider {
  readonly name = 'twilio';
  readonly displayName = 'Twilio';
  readonly supportedCountries = ['*']; // Twilio supports most countries

  private readonly logger = new Logger(TwilioProvider.name);
  private client: twilio.Twilio;
  private fromNumber: string;

  constructor(private configService: ConfigService) {
    const accountSid = this.configService.get<string>('TWILIO_ACCOUNT_SID');
    const authToken = this.configService.get<string>('TWILIO_AUTH_TOKEN');
    this.fromNumber = this.configService.get<string>('TWILIO_FROM_NUMBER');

    if (accountSid && authToken) {
      this.client = twilio(accountSid, authToken);
    }
  }

  isSupported(country: string): boolean {
    return true; // Twilio supports most countries
  }

  async sendSms(params: SendSmsParams): Promise<SendSmsResult> {
    try {
      if (!this.client) {
        return { success: false, message: 'Twilio not configured' };
      }

      const message = await this.client.messages.create({
        body: params.message,
        to: this.formatPhoneNumber(params.to),
        from: params.from || this.fromNumber,
      });

      return {
        success: true,
        messageId: message.sid,
        status: message.status,
        rawResponse: message,
      };
    } catch (error) {
      this.logger.error(`Twilio sendSms error: ${error.message}`);
      return {
        success: false,
        message: error.message,
      };
    }
  }

  async sendBulkSms(params: SendBulkSmsParams): Promise<SendBulkSmsResult> {
    const results: SendBulkSmsResult['results'] = [];
    let totalSent = 0;
    let totalFailed = 0;

    for (const recipient of params.recipients) {
      const result = await this.sendSms({
        to: recipient,
        message: params.message,
        from: params.from,
      });

      if (result.success) {
        totalSent++;
        results.push({
          to: recipient,
          messageId: result.messageId,
          status: 'sent',
        });
      } else {
        totalFailed++;
        results.push({
          to: recipient,
          status: 'failed',
          error: result.message,
        });
      }
    }

    return {
      success: totalSent > 0,
      totalSent,
      totalFailed,
      results,
    };
  }

  async checkDeliveryStatus(messageId: string): Promise<DeliveryStatusResult> {
    try {
      if (!this.client) {
        return { success: false, messageId, status: 'unknown', error: 'Twilio not configured' };
      }

      const message = await this.client.messages(messageId).fetch();

      const statusMap: Record<string, DeliveryStatusResult['status']> = {
        queued: 'pending',
        sending: 'pending',
        sent: 'sent',
        delivered: 'delivered',
        undelivered: 'failed',
        failed: 'failed',
      };

      return {
        success: true,
        messageId,
        status: statusMap[message.status] || 'unknown',
        deliveredAt: message.dateUpdated,
      };
    } catch (error) {
      return {
        success: false,
        messageId,
        status: 'unknown',
        error: error.message,
      };
    }
  }

  async getBalance(): Promise<{ balance: number; currency: string }> {
    try {
      if (!this.client) {
        return { balance: 0, currency: 'USD' };
      }

      const balance = await this.client.balance.fetch();
      return {
        balance: parseFloat(balance.balance),
        currency: balance.currency,
      };
    } catch (error) {
      this.logger.error(`Twilio getBalance error: ${error.message}`);
      return { balance: 0, currency: 'USD' };
    }
  }

  private formatPhoneNumber(phone: string): string {
    // Ensure phone starts with +
    let formatted = phone.replace(/[^0-9+]/g, '');
    if (!formatted.startsWith('+')) {
      formatted = '+' + formatted;
    }
    return formatted;
  }

  // ============================================================================
  // WHATSAPP (via Twilio)
  // ============================================================================

  async sendWhatsApp(
    to: string,
    message: string,
    mediaUrl?: string,
  ): Promise<SendSmsResult> {
    try {
      if (!this.client) {
        return { success: false, message: 'Twilio not configured' };
      }

      const whatsappFrom = this.configService.get<string>('TWILIO_WHATSAPP_FROM');

      const messageParams: any = {
        body: message,
        to: `whatsapp:${this.formatPhoneNumber(to)}`,
        from: `whatsapp:${whatsappFrom || this.fromNumber}`,
      };

      if (mediaUrl) {
        messageParams.mediaUrl = [mediaUrl];
      }

      const result = await this.client.messages.create(messageParams);

      return {
        success: true,
        messageId: result.sid,
        status: result.status,
        rawResponse: result,
      };
    } catch (error) {
      this.logger.error(`Twilio WhatsApp error: ${error.message}`);
      return {
        success: false,
        message: error.message,
      };
    }
  }
}
