/**
 * SMS Provider Interface
 * All SMS providers must implement this interface
 */
export interface SmsProvider {
  /**
   * Provider unique identifier
   */
  readonly name: string;

  /**
   * Provider display name
   */
  readonly displayName: string;

  /**
   * Supported countries (ISO codes)
   */
  readonly supportedCountries: string[];

  /**
   * Send SMS
   */
  sendSms(params: SendSmsParams): Promise<SendSmsResult>;

  /**
   * Send bulk SMS
   */
  sendBulkSms?(params: SendBulkSmsParams): Promise<SendBulkSmsResult>;

  /**
   * Check delivery status
   */
  checkDeliveryStatus?(messageId: string): Promise<DeliveryStatusResult>;

  /**
   * Get account balance
   */
  getBalance?(): Promise<{ balance: number; currency: string }>;

  /**
   * Check if provider supports a country
   */
  isSupported(country: string): boolean;
}

export interface SendSmsParams {
  to: string;
  message: string;
  from?: string;
  reference?: string;
}

export interface SendSmsResult {
  success: boolean;
  messageId?: string;
  status?: string;
  cost?: number;
  message?: string;
  rawResponse?: any;
}

export interface SendBulkSmsParams {
  recipients: string[];
  message: string;
  from?: string;
  reference?: string;
}

export interface SendBulkSmsResult {
  success: boolean;
  totalSent: number;
  totalFailed: number;
  results: Array<{
    to: string;
    messageId?: string;
    status: 'sent' | 'failed';
    error?: string;
  }>;
}

export interface DeliveryStatusResult {
  success: boolean;
  messageId: string;
  status: 'pending' | 'sent' | 'delivered' | 'failed' | 'unknown';
  deliveredAt?: Date;
  error?: string;
}

export enum SmsProviderType {
  TWILIO = 'twilio',
  PLIVO = 'plivo',
  INFOBIP = 'infobip',
  AFRICASTALKING = 'africastalking',
  NEXMO = 'nexmo',
  MESSAGEBIRD = 'messagebird',
  TERMII = 'termii',
  MNOTIFY = 'mnotify',
}
