import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { SmsProvider } from './provider.interface';
import { TwilioProvider } from './twilio.provider';
import { PlivoProvider } from './plivo.provider';
import { InfobipProvider } from './infobip.provider';
import { AfricasTalkingProvider } from './africastalking.provider';

@Injectable()
export class SmsProviderFactory implements OnModuleInit {
  private readonly logger = new Logger(SmsProviderFactory.name);
  private providers: Map<string, SmsProvider> = new Map();

  constructor(
    private twilioProvider: TwilioProvider,
    private plivoProvider: PlivoProvider,
    private infobipProvider: InfobipProvider,
    private africasTalkingProvider: AfricasTalkingProvider,
  ) {}

  onModuleInit() {
    // Register all providers
    this.registerProvider(this.twilioProvider);
    this.registerProvider(this.plivoProvider);
    this.registerProvider(this.infobipProvider);
    this.registerProvider(this.africasTalkingProvider);

    this.logger.log(`Registered ${this.providers.size} SMS providers`);
  }

  /**
   * Register a provider
   */
  registerProvider(provider: SmsProvider): void {
    this.providers.set(provider.name, provider);
    this.logger.debug(`Registered SMS provider: ${provider.name}`);
  }

  /**
   * Get provider by name
   */
  getProvider(name: string): SmsProvider | undefined {
    return this.providers.get(name);
  }

  /**
   * Get all available providers
   */
  getAllProviders(): SmsProvider[] {
    return Array.from(this.providers.values());
  }

  /**
   * Get providers that support a specific country
   */
  getProvidersForCountry(country: string): SmsProvider[] {
    return this.getAllProviders().filter(provider => provider.isSupported(country));
  }

  /**
   * Get the best provider for a country
   * Priority: Local providers first, then global
   */
  getBestProvider(country: string): SmsProvider | undefined {
    const supported = this.getProvidersForCountry(country);

    if (supported.length === 0) return undefined;

    // Priority order for different regions
    const priorityMap: Record<string, string[]> = {
      // East Africa
      KE: ['africastalking', 'twilio', 'infobip'],
      UG: ['africastalking', 'twilio', 'infobip'],
      TZ: ['africastalking', 'twilio', 'infobip'],
      RW: ['africastalking', 'twilio', 'infobip'],
      // West Africa
      NG: ['africastalking', 'infobip', 'twilio'],
      GH: ['africastalking', 'infobip', 'twilio'],
      SN: ['africastalking', 'infobip', 'twilio'],
      CI: ['africastalking', 'infobip', 'twilio'],
      // Default
      DEFAULT: ['twilio', 'infobip', 'plivo'],
    };

    const priorities = priorityMap[country.toUpperCase()] || priorityMap.DEFAULT;

    for (const providerName of priorities) {
      const provider = supported.find(p => p.name === providerName);
      if (provider) return provider;
    }

    return supported[0];
  }

  /**
   * Get provider information for display
   */
  getProviderInfo(): Array<{
    name: string;
    displayName: string;
    countries: string[];
  }> {
    return this.getAllProviders().map(provider => ({
      name: provider.name,
      displayName: provider.displayName,
      countries: provider.supportedCountries,
    }));
  }
}
