import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  SmsProvider,
  SendSmsParams,
  SendSmsResult,
  SendBulkSmsParams,
  SendBulkSmsResult,
  DeliveryStatusResult,
} from './provider.interface';

@Injectable()
export class InfobipProvider implements SmsProvider {
  readonly name = 'infobip';
  readonly displayName = 'Infobip';
  readonly supportedCountries = ['*']; // Global coverage

  private readonly logger = new Logger(InfobipProvider.name);
  private baseUrl: string;
  private apiKey: string;
  private senderId: string;

  constructor(private configService: ConfigService) {
    this.baseUrl = this.configService.get<string>('INFOBIP_BASE_URL') || 'https://api.infobip.com';
    this.apiKey = this.configService.get<string>('INFOBIP_API_KEY');
    this.senderId = this.configService.get<string>('INFOBIP_SENDER_ID');
  }

  isSupported(country: string): boolean {
    return true;
  }

  async sendSms(params: SendSmsParams): Promise<SendSmsResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/sms/2/text/advanced`,
        {
          messages: [
            {
              from: params.from || this.senderId,
              destinations: [{ to: this.formatPhoneNumber(params.to) }],
              text: params.message,
            },
          ],
        },
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const message = response.data.messages?.[0];

      if (message && message.status?.groupName === 'PENDING') {
        return {
          success: true,
          messageId: message.messageId,
          status: message.status.name,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: message?.status?.description || 'Failed to send',
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Infobip sendSms error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.requestError?.serviceException?.text || error.message,
      };
    }
  }

  async sendBulkSms(params: SendBulkSmsParams): Promise<SendBulkSmsResult> {
    try {
      const destinations = params.recipients.map(to => ({
        to: this.formatPhoneNumber(to),
      }));

      const response = await axios.post(
        `${this.baseUrl}/sms/2/text/advanced`,
        {
          messages: [
            {
              from: params.from || this.senderId,
              destinations,
              text: params.message,
            },
          ],
        },
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const messages = response.data.messages || [];
      const results: SendBulkSmsResult['results'] = messages.map((msg: any) => ({
        to: msg.to,
        messageId: msg.messageId,
        status: msg.status?.groupName === 'PENDING' ? 'sent' : 'failed',
        error: msg.status?.groupName !== 'PENDING' ? msg.status?.description : undefined,
      }));

      const totalSent = results.filter(r => r.status === 'sent').length;
      const totalFailed = results.filter(r => r.status === 'failed').length;

      return {
        success: totalSent > 0,
        totalSent,
        totalFailed,
        results,
      };
    } catch (error) {
      this.logger.error(`Infobip sendBulkSms error: ${error.message}`);
      return {
        success: false,
        totalSent: 0,
        totalFailed: params.recipients.length,
        results: params.recipients.map(to => ({
          to,
          status: 'failed' as const,
          error: error.message,
        })),
      };
    }
  }

  async checkDeliveryStatus(messageId: string): Promise<DeliveryStatusResult> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/sms/1/reports?messageId=${messageId}`,
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
          },
        },
      );

      const report = response.data.results?.[0];

      if (!report) {
        return { success: false, messageId, status: 'unknown' };
      }

      const statusMap: Record<string, DeliveryStatusResult['status']> = {
        PENDING: 'pending',
        DELIVERED: 'delivered',
        UNDELIVERABLE: 'failed',
        EXPIRED: 'failed',
        REJECTED: 'failed',
      };

      return {
        success: true,
        messageId,
        status: statusMap[report.status?.groupName] || 'unknown',
        deliveredAt: report.doneAt ? new Date(report.doneAt) : undefined,
      };
    } catch (error) {
      return {
        success: false,
        messageId,
        status: 'unknown',
        error: error.message,
      };
    }
  }

  async getBalance(): Promise<{ balance: number; currency: string }> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/account/1/balance`,
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
          },
        },
      );

      return {
        balance: response.data.balance || 0,
        currency: response.data.currency || 'EUR',
      };
    } catch (error) {
      this.logger.error(`Infobip getBalance error: ${error.message}`);
      return { balance: 0, currency: 'EUR' };
    }
  }

  private formatPhoneNumber(phone: string): string {
    return phone.replace(/[^0-9]/g, '');
  }

  // ============================================================================
  // WHATSAPP (via Infobip)
  // ============================================================================

  async sendWhatsApp(
    to: string,
    templateName: string,
    templateData: Record<string, string>,
    language: string = 'en',
  ): Promise<SendSmsResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/whatsapp/1/message/template`,
        {
          messages: [
            {
              from: this.configService.get('INFOBIP_WHATSAPP_NUMBER'),
              to: this.formatPhoneNumber(to),
              content: {
                templateName,
                templateData,
                language,
              },
            },
          ],
        },
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const message = response.data.messages?.[0];

      if (message?.status?.groupName === 'PENDING') {
        return {
          success: true,
          messageId: message.messageId,
          status: 'sent',
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: message?.status?.description || 'Failed to send',
      };
    } catch (error) {
      this.logger.error(`Infobip WhatsApp error: ${error.message}`);
      return {
        success: false,
        message: error.message,
      };
    }
  }

  // ============================================================================
  // VIBER
  // ============================================================================

  async sendViber(
    to: string,
    message: string,
    imageUrl?: string,
    buttonText?: string,
    buttonUrl?: string,
  ): Promise<SendSmsResult> {
    try {
      const content: any = { text: message };

      if (imageUrl) {
        content.imageUrl = imageUrl;
      }

      if (buttonText && buttonUrl) {
        content.button = {
          text: buttonText,
          url: buttonUrl,
        };
      }

      const response = await axios.post(
        `${this.baseUrl}/viber/1/message/text`,
        {
          messages: [
            {
              from: this.configService.get('INFOBIP_VIBER_SENDER'),
              to: this.formatPhoneNumber(to),
              content,
            },
          ],
        },
        {
          headers: {
            Authorization: `App ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const msg = response.data.messages?.[0];

      if (msg?.status?.groupName === 'PENDING') {
        return {
          success: true,
          messageId: msg.messageId,
          status: 'sent',
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: msg?.status?.description || 'Failed to send',
      };
    } catch (error) {
      this.logger.error(`Infobip Viber error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }
}
