import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import * as ExcelJS from 'exceljs';

// ============================================================================
// TYPES
// ============================================================================

export type ReportType =
  | 'bookings'
  | 'earnings'
  | 'drivers'
  | 'users'
  | 'payments'
  | 'deliveries'
  | 'ratings'
  | 'support'
  | 'financial'
  | 'operational';

export type ReportFormat = 'json' | 'csv' | 'excel' | 'pdf';
export type ReportPeriod = 'today' | 'yesterday' | 'week' | 'month' | 'quarter' | 'year' | 'custom';

export interface ReportOptions {
  type: ReportType;
  format: ReportFormat;
  period: ReportPeriod;
  startDate?: Date;
  endDate?: Date;
  filters?: Record<string, any>;
  groupBy?: string;
  includeCharts?: boolean;
}

export interface ReportResult {
  id: string;
  type: ReportType;
  format: ReportFormat;
  generatedAt: Date;
  period: { start: Date; end: Date };
  summary: Record<string, any>;
  data: any[];
  fileUrl?: string;
  fileName?: string;
}

export interface ScheduledReport {
  id: number;
  name: string;
  type: ReportType;
  format: ReportFormat;
  schedule: string; // Cron expression
  recipients: string[];
  filters?: Record<string, any>;
  active: boolean;
  lastRunAt?: Date;
  nextRunAt?: Date;
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class ReportService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // GENERATION RAPPORTS
  // ==========================================================================

  /**
   * Générer un rapport
   */
  async generateReport(
    merchantId: number,
    options: ReportOptions,
  ): Promise<ReportResult> {
    const { start, end } = this.getPeriodDates(options.period, options.startDate, options.endDate);
    const reportId = `RPT-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;

    let data: any[];
    let summary: Record<string, any>;

    switch (options.type) {
      case 'bookings':
        ({ data, summary } = await this.generateBookingsReport(merchantId, start, end, options.filters));
        break;
      case 'earnings':
        ({ data, summary } = await this.generateEarningsReport(merchantId, start, end, options.filters));
        break;
      case 'drivers':
        ({ data, summary } = await this.generateDriversReport(merchantId, start, end, options.filters));
        break;
      case 'users':
        ({ data, summary } = await this.generateUsersReport(merchantId, start, end, options.filters));
        break;
      case 'payments':
        ({ data, summary } = await this.generatePaymentsReport(merchantId, start, end, options.filters));
        break;
      case 'deliveries':
        ({ data, summary } = await this.generateDeliveriesReport(merchantId, start, end, options.filters));
        break;
      case 'ratings':
        ({ data, summary } = await this.generateRatingsReport(merchantId, start, end, options.filters));
        break;
      case 'support':
        ({ data, summary } = await this.generateSupportReport(merchantId, start, end, options.filters));
        break;
      case 'financial':
        ({ data, summary } = await this.generateFinancialReport(merchantId, start, end, options.filters));
        break;
      case 'operational':
        ({ data, summary } = await this.generateOperationalReport(merchantId, start, end, options.filters));
        break;
      default:
        throw new Error(`Type de rapport inconnu: ${options.type}`);
    }

    const result: ReportResult = {
      id: reportId,
      type: options.type,
      format: options.format,
      generatedAt: new Date(),
      period: { start, end },
      summary,
      data,
    };

    // Générer le fichier si nécessaire
    if (options.format !== 'json') {
      const file = await this.exportToFormat(result, options.format, options.type);
      result.fileUrl = file.url;
      result.fileName = file.name;
    }

    // Sauvegarder l'historique
    await this.saveReportHistory(merchantId, result);

    return result;
  }

  // ==========================================================================
  // RAPPORTS SPECIFIQUES
  // ==========================================================================

  /**
   * Rapport des réservations
   */
  private async generateBookingsReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const where: any = {
      merchantId,
      createdAt: { gte: start, lte: end },
    };

    if (filters?.status) where.status = filters.status;
    if (filters?.vehicleTypeId) where.vehicleTypeId = filters.vehicleTypeId;
    if (filters?.driverId) where.driverId = filters.driverId;

    const [bookings, stats] = await Promise.all([
      this.prisma.booking.findMany({
        where,
        include: {
          user: { select: { id: true, firstName: true, lastName: true, phone: true } },
          driver: { select: { id: true, firstName: true, lastName: true, phone: true } },
          vehicleType: { select: { id: true, name: true } },
        },
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.booking.aggregate({
        where,
        _count: { id: true },
        _sum: { totalFare: true, distance: true },
        _avg: { totalFare: true, distance: true, rating: true },
      }),
    ]);

    // Stats par statut
    const statusCounts = await this.prisma.booking.groupBy({
      by: ['status'],
      where,
      _count: { status: true },
    });

    const summary = {
      totalBookings: stats._count.id || 0,
      totalRevenue: parseFloat(stats._sum.totalFare || '0'),
      totalDistance: parseFloat(stats._sum.distance || '0'),
      avgFare: parseFloat(stats._avg.totalFare || '0'),
      avgDistance: parseFloat(stats._avg.distance || '0'),
      avgRating: parseFloat(stats._avg.rating || '0'),
      byStatus: Object.fromEntries(statusCounts.map((s) => [s.status, s._count.status])),
      completionRate: this.calculateCompletionRate(statusCounts),
    };

    const data = bookings.map((b) => ({
      id: b.id,
      bookingNumber: b.bookingNumber,
      status: b.status,
      user: b.user ? `${b.user.firstName} ${b.user.lastName}` : 'N/A',
      driver: b.driver ? `${b.driver.firstName} ${b.driver.lastName}` : 'N/A',
      vehicleType: b.vehicleType?.name || 'N/A',
      pickupAddress: b.pickupAddress,
      dropoffAddress: b.dropoffAddress,
      distance: b.distance,
      fare: b.totalFare,
      rating: b.rating,
      createdAt: b.createdAt,
      completedAt: b.completedAt,
    }));

    return { data, summary };
  }

  /**
   * Rapport des revenus
   */
  private async generateEarningsReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const where: any = {
      merchantId,
      createdAt: { gte: start, lte: end },
      status: 'completed',
    };

    // Revenus par jour
    const dailyEarnings = await this.prisma.$queryRaw<any[]>`
      SELECT
        DATE(created_at) as date,
        COUNT(*) as bookings,
        SUM(total_fare) as revenue,
        SUM(platform_fee) as platformFee,
        SUM(driver_earnings) as driverEarnings
      FROM bookings
      WHERE merchant_id = ${merchantId}
        AND created_at >= ${start}
        AND created_at <= ${end}
        AND status = 'completed'
      GROUP BY DATE(created_at)
      ORDER BY date ASC
    `;

    // Revenus par type de véhicule
    const byVehicleType = await this.prisma.booking.groupBy({
      by: ['vehicleTypeId'],
      where,
      _count: { id: true },
      _sum: { totalFare: true, platformFee: true },
    });

    // Revenus par méthode de paiement
    const byPaymentMethod = await this.prisma.booking.groupBy({
      by: ['paymentMethod'],
      where,
      _count: { id: true },
      _sum: { totalFare: true },
    });

    const totals = await this.prisma.booking.aggregate({
      where,
      _sum: { totalFare: true, platformFee: true, driverEarnings: true },
    });

    const summary = {
      totalRevenue: parseFloat(totals._sum.totalFare || '0'),
      platformFees: parseFloat(totals._sum.platformFee || '0'),
      driverEarnings: parseFloat(totals._sum.driverEarnings || '0'),
      byVehicleType: byVehicleType.map((v) => ({
        vehicleTypeId: v.vehicleTypeId,
        bookings: v._count.id,
        revenue: parseFloat(v._sum.totalFare || '0'),
      })),
      byPaymentMethod: byPaymentMethod.map((p) => ({
        method: p.paymentMethod,
        bookings: p._count.id,
        revenue: parseFloat(p._sum.totalFare || '0'),
      })),
    };

    return { data: dailyEarnings, summary };
  }

  /**
   * Rapport des chauffeurs
   */
  private async generateDriversReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const driverStats = await this.prisma.$queryRaw<any[]>`
      SELECT
        d.id,
        d.first_name as firstName,
        d.last_name as lastName,
        d.phone,
        d.status,
        d.rating,
        COUNT(b.id) as totalBookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completedBookings,
        SUM(CASE WHEN b.status = 'cancelled' THEN 1 ELSE 0 END) as cancelledBookings,
        SUM(b.driver_earnings) as totalEarnings,
        AVG(b.rating) as avgRating
      FROM drivers d
      LEFT JOIN bookings b ON b.driver_id = d.id
        AND b.created_at >= ${start}
        AND b.created_at <= ${end}
      WHERE d.merchant_id = ${merchantId}
      GROUP BY d.id, d.first_name, d.last_name, d.phone, d.status, d.rating
      ORDER BY totalEarnings DESC
    `;

    const [totalDrivers, activeDrivers, newDrivers] = await Promise.all([
      this.prisma.driver.count({ where: { merchantId } }),
      this.prisma.driver.count({ where: { merchantId, status: 'active' } }),
      this.prisma.driver.count({
        where: { merchantId, createdAt: { gte: start, lte: end } },
      }),
    ]);

    const summary = {
      totalDrivers,
      activeDrivers,
      newDrivers,
      topPerformers: driverStats.slice(0, 10),
      avgEarningsPerDriver: driverStats.length > 0
        ? driverStats.reduce((sum, d) => sum + parseFloat(d.totalEarnings || 0), 0) / driverStats.length
        : 0,
    };

    return { data: driverStats, summary };
  }

  /**
   * Rapport des utilisateurs
   */
  private async generateUsersReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const userStats = await this.prisma.$queryRaw<any[]>`
      SELECT
        u.id,
        u.first_name as firstName,
        u.last_name as lastName,
        u.phone,
        u.email,
        u.created_at as createdAt,
        COUNT(b.id) as totalBookings,
        SUM(b.total_fare) as totalSpent,
        MAX(b.created_at) as lastBooking
      FROM users u
      LEFT JOIN bookings b ON b.user_id = u.id
        AND b.created_at >= ${start}
        AND b.created_at <= ${end}
        AND b.status = 'completed'
      WHERE u.merchant_id = ${merchantId}
      GROUP BY u.id, u.first_name, u.last_name, u.phone, u.email, u.created_at
      ORDER BY totalSpent DESC
    `;

    const [totalUsers, activeUsers, newUsers] = await Promise.all([
      this.prisma.user.count({ where: { merchantId } }),
      this.prisma.user.count({
        where: {
          merchantId,
          bookings: { some: { createdAt: { gte: start, lte: end } } },
        },
      }),
      this.prisma.user.count({
        where: { merchantId, createdAt: { gte: start, lte: end } },
      }),
    ]);

    const summary = {
      totalUsers,
      activeUsers,
      newUsers,
      retentionRate: totalUsers > 0 ? (activeUsers / totalUsers) * 100 : 0,
      topSpenders: userStats.slice(0, 10),
    };

    return { data: userStats, summary };
  }

  /**
   * Rapport des paiements
   */
  private async generatePaymentsReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const where: any = {
      merchantId,
      createdAt: { gte: start, lte: end },
    };

    if (filters?.status) where.status = filters.status;
    if (filters?.method) where.paymentMethod = filters.method;

    const [payments, byMethod, byStatus] = await Promise.all([
      this.prisma.payment.findMany({
        where,
        include: {
          booking: { select: { bookingNumber: true } },
          user: { select: { firstName: true, lastName: true } },
        },
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.payment.groupBy({
        by: ['paymentMethod'],
        where,
        _count: { id: true },
        _sum: { amount: true },
      }),
      this.prisma.payment.groupBy({
        by: ['status'],
        where,
        _count: { id: true },
        _sum: { amount: true },
      }),
    ]);

    const totals = await this.prisma.payment.aggregate({
      where: { ...where, status: 'completed' },
      _sum: { amount: true },
      _count: { id: true },
    });

    const summary = {
      totalPayments: totals._count.id || 0,
      totalAmount: parseFloat(totals._sum.amount || '0'),
      byMethod: byMethod.map((m) => ({
        method: m.paymentMethod,
        count: m._count.id,
        amount: parseFloat(m._sum.amount || '0'),
      })),
      byStatus: byStatus.map((s) => ({
        status: s.status,
        count: s._count.id,
        amount: parseFloat(s._sum.amount || '0'),
      })),
    };

    const data = payments.map((p) => ({
      id: p.id,
      bookingNumber: p.booking?.bookingNumber || 'N/A',
      user: p.user ? `${p.user.firstName} ${p.user.lastName}` : 'N/A',
      amount: p.amount,
      method: p.paymentMethod,
      status: p.status,
      transactionId: p.transactionId,
      createdAt: p.createdAt,
    }));

    return { data, summary };
  }

  /**
   * Rapport des livraisons
   */
  private async generateDeliveriesReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const where: any = {
      merchantId,
      createdAt: { gte: start, lte: end },
    };

    const [deliveries, stats] = await Promise.all([
      this.prisma.delivery.findMany({
        where,
        include: {
          user: { select: { firstName: true, lastName: true } },
          driver: { select: { firstName: true, lastName: true } },
        },
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.delivery.aggregate({
        where,
        _count: { id: true },
        _sum: { totalFare: true },
        _avg: { totalFare: true },
      }),
    ]);

    const summary = {
      totalDeliveries: stats._count.id || 0,
      totalRevenue: parseFloat(stats._sum.totalFare || '0'),
      avgFare: parseFloat(stats._avg.totalFare || '0'),
    };

    const data = deliveries.map((d) => ({
      id: d.id,
      deliveryNumber: d.deliveryNumber,
      status: d.status,
      user: d.user ? `${d.user.firstName} ${d.user.lastName}` : 'N/A',
      driver: d.driver ? `${d.driver.firstName} ${d.driver.lastName}` : 'N/A',
      packageType: d.packageType,
      fare: d.totalFare,
      createdAt: d.createdAt,
    }));

    return { data, summary };
  }

  /**
   * Rapport des évaluations
   */
  private async generateRatingsReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const ratings = await this.prisma.rating.findMany({
      where: {
        merchantId,
        createdAt: { gte: start, lte: end },
      },
      include: {
        booking: { select: { bookingNumber: true } },
        user: { select: { firstName: true, lastName: true } },
        driver: { select: { firstName: true, lastName: true } },
      },
      orderBy: { createdAt: 'desc' },
    });

    const distribution = await this.prisma.rating.groupBy({
      by: ['rating'],
      where: { merchantId, createdAt: { gte: start, lte: end } },
      _count: { rating: true },
    });

    const avgRating = await this.prisma.rating.aggregate({
      where: { merchantId, createdAt: { gte: start, lte: end } },
      _avg: { rating: true },
      _count: { id: true },
    });

    const summary = {
      totalRatings: avgRating._count.id || 0,
      avgRating: parseFloat(avgRating._avg.rating?.toFixed(2) || '0'),
      distribution: Object.fromEntries(distribution.map((d) => [d.rating, d._count.rating])),
      positiveRate: this.calculatePositiveRate(distribution),
    };

    const data = ratings.map((r) => ({
      id: r.id,
      bookingNumber: r.booking?.bookingNumber || 'N/A',
      user: r.user ? `${r.user.firstName} ${r.user.lastName}` : 'N/A',
      driver: r.driver ? `${r.driver.firstName} ${r.driver.lastName}` : 'N/A',
      rating: r.rating,
      comment: r.comment,
      createdAt: r.createdAt,
    }));

    return { data, summary };
  }

  /**
   * Rapport support
   */
  private async generateSupportReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    const where: any = {
      merchantId,
      createdAt: { gte: start, lte: end },
    };

    const [tickets, byStatus, byPriority] = await Promise.all([
      this.prisma.supportTicket.findMany({
        where,
        include: {
          category: { select: { name: true } },
          assignedAgent: { select: { firstName: true, lastName: true } },
        },
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.supportTicket.groupBy({
        by: ['status'],
        where,
        _count: { status: true },
      }),
      this.prisma.supportTicket.groupBy({
        by: ['priority'],
        where,
        _count: { priority: true },
      }),
    ]);

    const avgResolution = await this.prisma.supportTicket.findMany({
      where: { ...where, resolvedAt: { not: null } },
      select: { createdAt: true, resolvedAt: true },
    });

    const avgResolutionTime = avgResolution.length > 0
      ? avgResolution.reduce((sum, t) => {
          const diff = new Date(t.resolvedAt!).getTime() - new Date(t.createdAt).getTime();
          return sum + diff / (1000 * 60 * 60);
        }, 0) / avgResolution.length
      : 0;

    const summary = {
      totalTickets: tickets.length,
      byStatus: Object.fromEntries(byStatus.map((s) => [s.status, s._count.status])),
      byPriority: Object.fromEntries(byPriority.map((p) => [p.priority, p._count.priority])),
      avgResolutionTime: Math.round(avgResolutionTime * 10) / 10,
    };

    const data = tickets.map((t) => ({
      id: t.id,
      ticketNumber: t.ticketNumber,
      subject: t.subject,
      status: t.status,
      priority: t.priority,
      category: t.category?.name || 'N/A',
      assignedTo: t.assignedAgent ? `${t.assignedAgent.firstName} ${t.assignedAgent.lastName}` : 'Non assigné',
      createdAt: t.createdAt,
      resolvedAt: t.resolvedAt,
    }));

    return { data, summary };
  }

  /**
   * Rapport financier
   */
  private async generateFinancialReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    // Revenus des courses
    const bookingRevenue = await this.prisma.booking.aggregate({
      where: { merchantId, status: 'completed', createdAt: { gte: start, lte: end } },
      _sum: { totalFare: true, platformFee: true, driverEarnings: true },
    });

    // Revenus des livraisons
    const deliveryRevenue = await this.prisma.delivery.aggregate({
      where: { merchantId, status: 'delivered', createdAt: { gte: start, lte: end } },
      _sum: { totalFare: true },
    });

    // Transactions wallet
    const walletTransactions = await this.prisma.walletTransaction.aggregate({
      where: { merchantId, createdAt: { gte: start, lte: end } },
      _sum: { amount: true },
    });

    // Remboursements
    const refunds = await this.prisma.payment.aggregate({
      where: { merchantId, status: 'refunded', createdAt: { gte: start, lte: end } },
      _sum: { amount: true },
    });

    const summary = {
      grossRevenue: parseFloat(bookingRevenue._sum.totalFare || '0') + parseFloat(deliveryRevenue._sum.totalFare || '0'),
      bookingRevenue: parseFloat(bookingRevenue._sum.totalFare || '0'),
      deliveryRevenue: parseFloat(deliveryRevenue._sum.totalFare || '0'),
      platformFees: parseFloat(bookingRevenue._sum.platformFee || '0'),
      driverPayouts: parseFloat(bookingRevenue._sum.driverEarnings || '0'),
      refunds: parseFloat(refunds._sum.amount || '0'),
      netRevenue: parseFloat(bookingRevenue._sum.platformFee || '0') - parseFloat(refunds._sum.amount || '0'),
    };

    // Données journalières
    const dailyData = await this.prisma.$queryRaw<any[]>`
      SELECT
        DATE(created_at) as date,
        SUM(total_fare) as revenue,
        SUM(platform_fee) as platformFee,
        COUNT(*) as transactions
      FROM bookings
      WHERE merchant_id = ${merchantId}
        AND status = 'completed'
        AND created_at >= ${start}
        AND created_at <= ${end}
      GROUP BY DATE(created_at)
      ORDER BY date ASC
    `;

    return { data: dailyData, summary };
  }

  /**
   * Rapport opérationnel
   */
  private async generateOperationalReport(
    merchantId: number,
    start: Date,
    end: Date,
    filters?: Record<string, any>,
  ): Promise<{ data: any[]; summary: Record<string, any> }> {
    // Métriques opérationnelles
    const [
      totalBookings,
      completedBookings,
      cancelledBookings,
      avgWaitTime,
      peakHours,
    ] = await Promise.all([
      this.prisma.booking.count({ where: { merchantId, createdAt: { gte: start, lte: end } } }),
      this.prisma.booking.count({ where: { merchantId, status: 'completed', createdAt: { gte: start, lte: end } } }),
      this.prisma.booking.count({ where: { merchantId, status: 'cancelled', createdAt: { gte: start, lte: end } } }),
      this.prisma.$queryRaw<any[]>`
        SELECT AVG(TIMESTAMPDIFF(MINUTE, created_at, accepted_at)) as avgWaitTime
        FROM bookings
        WHERE merchant_id = ${merchantId}
          AND accepted_at IS NOT NULL
          AND created_at >= ${start}
          AND created_at <= ${end}
      `,
      this.prisma.$queryRaw<any[]>`
        SELECT HOUR(created_at) as hour, COUNT(*) as count
        FROM bookings
        WHERE merchant_id = ${merchantId}
          AND created_at >= ${start}
          AND created_at <= ${end}
        GROUP BY HOUR(created_at)
        ORDER BY count DESC
        LIMIT 5
      `,
    ]);

    const summary = {
      totalBookings,
      completedBookings,
      cancelledBookings,
      completionRate: totalBookings > 0 ? (completedBookings / totalBookings) * 100 : 0,
      cancellationRate: totalBookings > 0 ? (cancelledBookings / totalBookings) * 100 : 0,
      avgWaitTime: avgWaitTime[0]?.avgWaitTime || 0,
      peakHours: peakHours.map((h) => ({ hour: h.hour, count: h.count })),
    };

    const hourlyData = await this.prisma.$queryRaw<any[]>`
      SELECT
        HOUR(created_at) as hour,
        COUNT(*) as bookings,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
      FROM bookings
      WHERE merchant_id = ${merchantId}
        AND created_at >= ${start}
        AND created_at <= ${end}
      GROUP BY HOUR(created_at)
      ORDER BY hour ASC
    `;

    return { data: hourlyData, summary };
  }

  // ==========================================================================
  // EXPORT
  // ==========================================================================

  /**
   * Exporter vers format
   */
  private async exportToFormat(
    report: ReportResult,
    format: ReportFormat,
    type: ReportType,
  ): Promise<{ url: string; name: string }> {
    const fileName = `${type}-report-${new Date().toISOString().split('T')[0]}.${format === 'excel' ? 'xlsx' : format}`;

    switch (format) {
      case 'csv':
        return this.exportToCsv(report.data, fileName);
      case 'excel':
        return this.exportToExcel(report, fileName, type);
      case 'pdf':
        return this.exportToPdf(report, fileName, type);
      default:
        throw new Error(`Format non supporté: ${format}`);
    }
  }

  /**
   * Export CSV
   */
  private async exportToCsv(data: any[], fileName: string): Promise<{ url: string; name: string }> {
    if (data.length === 0) {
      return { url: '', name: fileName };
    }

    const headers = Object.keys(data[0]);
    const csvRows = [
      headers.join(','),
      ...data.map((row) =>
        headers.map((h) => {
          const val = row[h];
          if (val === null || val === undefined) return '';
          if (typeof val === 'string' && val.includes(',')) return `"${val}"`;
          return val;
        }).join(','),
      ),
    ];

    const csvContent = csvRows.join('\n');
    // Sauvegarder le fichier (utiliser StorageService en production)
    const url = `/reports/${fileName}`;

    return { url, name: fileName };
  }

  /**
   * Export Excel
   */
  private async exportToExcel(
    report: ReportResult,
    fileName: string,
    type: ReportType,
  ): Promise<{ url: string; name: string }> {
    const workbook = new ExcelJS.Workbook();
    workbook.creator = 'MonkAPI';
    workbook.created = new Date();

    // Feuille résumé
    const summarySheet = workbook.addWorksheet('Résumé');
    summarySheet.columns = [
      { header: 'Métrique', key: 'metric', width: 30 },
      { header: 'Valeur', key: 'value', width: 20 },
    ];

    Object.entries(report.summary).forEach(([key, value]) => {
      if (typeof value !== 'object') {
        summarySheet.addRow({ metric: key, value });
      }
    });

    // Feuille données
    if (report.data.length > 0) {
      const dataSheet = workbook.addWorksheet('Données');
      const headers = Object.keys(report.data[0]);
      dataSheet.columns = headers.map((h) => ({ header: h, key: h, width: 20 }));
      report.data.forEach((row) => dataSheet.addRow(row));

      // Style en-têtes
      dataSheet.getRow(1).font = { bold: true };
      dataSheet.getRow(1).fill = {
        type: 'pattern',
        pattern: 'solid',
        fgColor: { argb: 'FFE0E0E0' },
      };
    }

    // Sauvegarder (utiliser StorageService en production)
    const buffer = await workbook.xlsx.writeBuffer();
    const url = `/reports/${fileName}`;

    return { url, name: fileName };
  }

  /**
   * Export PDF
   */
  private async exportToPdf(
    report: ReportResult,
    fileName: string,
    type: ReportType,
  ): Promise<{ url: string; name: string }> {
    // En production, utiliser une librairie comme pdfkit ou puppeteer
    // Pour l'instant, on retourne un placeholder
    const url = `/reports/${fileName}`;
    return { url, name: fileName };
  }

  // ==========================================================================
  // RAPPORTS PLANIFIES
  // ==========================================================================

  /**
   * Lister les rapports planifiés
   */
  async listScheduledReports(merchantId: number): Promise<ScheduledReport[]> {
    const reports = await this.prisma.scheduledReport.findMany({
      where: { merchantId },
      orderBy: { createdAt: 'desc' },
    });

    return reports.map(this.mapScheduledReport);
  }

  /**
   * Créer un rapport planifié
   */
  async createScheduledReport(
    merchantId: number,
    data: {
      name: string;
      type: ReportType;
      format: ReportFormat;
      schedule: string;
      recipients: string[];
      filters?: Record<string, any>;
    },
  ): Promise<ScheduledReport> {
    const report = await this.prisma.scheduledReport.create({
      data: {
        merchantId,
        name: data.name,
        type: data.type,
        format: data.format,
        schedule: data.schedule,
        recipients: JSON.stringify(data.recipients),
        filters: data.filters ? JSON.stringify(data.filters) : null,
        active: true,
        nextRunAt: this.calculateNextRun(data.schedule),
      },
    });

    return this.mapScheduledReport(report);
  }

  /**
   * Mettre à jour un rapport planifié
   */
  async updateScheduledReport(
    id: number,
    merchantId: number,
    data: Partial<{
      name: string;
      schedule: string;
      recipients: string[];
      filters: Record<string, any>;
      active: boolean;
    }>,
  ): Promise<ScheduledReport | null> {
    const report = await this.prisma.scheduledReport.findFirst({
      where: { id, merchantId },
    });

    if (!report) return null;

    const updated = await this.prisma.scheduledReport.update({
      where: { id },
      data: {
        name: data.name,
        schedule: data.schedule,
        recipients: data.recipients ? JSON.stringify(data.recipients) : undefined,
        filters: data.filters ? JSON.stringify(data.filters) : undefined,
        active: data.active,
        nextRunAt: data.schedule ? this.calculateNextRun(data.schedule) : undefined,
      },
    });

    return this.mapScheduledReport(updated);
  }

  /**
   * Supprimer un rapport planifié
   */
  async deleteScheduledReport(id: number, merchantId: number): Promise<boolean> {
    const report = await this.prisma.scheduledReport.findFirst({
      where: { id, merchantId },
    });

    if (!report) return false;

    await this.prisma.scheduledReport.delete({ where: { id } });
    return true;
  }

  // ==========================================================================
  // HISTORIQUE
  // ==========================================================================

  /**
   * Sauvegarder l'historique
   */
  private async saveReportHistory(merchantId: number, report: ReportResult): Promise<void> {
    await this.prisma.reportHistory.create({
      data: {
        merchantId,
        reportId: report.id,
        type: report.type,
        format: report.format,
        periodStart: report.period.start,
        periodEnd: report.period.end,
        summary: JSON.stringify(report.summary),
        fileUrl: report.fileUrl,
        fileName: report.fileName,
      },
    });
  }

  /**
   * Obtenir l'historique des rapports
   */
  async getReportHistory(
    merchantId: number,
    options: { page?: number; limit?: number; type?: ReportType } = {},
  ): Promise<{ reports: any[]; total: number }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };
    if (options.type) where.type = options.type;

    const [reports, total] = await Promise.all([
      this.prisma.reportHistory.findMany({
        where,
        skip,
        take: limit,
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.reportHistory.count({ where }),
    ]);

    return { reports, total };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  /**
   * Obtenir les dates de période
   */
  private getPeriodDates(
    period: ReportPeriod,
    startDate?: Date,
    endDate?: Date,
  ): { start: Date; end: Date } {
    const now = new Date();
    let start: Date;
    let end: Date = new Date(now.setHours(23, 59, 59, 999));

    switch (period) {
      case 'today':
        start = new Date(now.setHours(0, 0, 0, 0));
        break;
      case 'yesterday':
        start = new Date(now.setDate(now.getDate() - 1));
        start.setHours(0, 0, 0, 0);
        end = new Date(start);
        end.setHours(23, 59, 59, 999);
        break;
      case 'week':
        start = new Date(now.setDate(now.getDate() - 7));
        start.setHours(0, 0, 0, 0);
        break;
      case 'month':
        start = new Date(now.getFullYear(), now.getMonth(), 1);
        break;
      case 'quarter':
        const quarter = Math.floor(now.getMonth() / 3);
        start = new Date(now.getFullYear(), quarter * 3, 1);
        break;
      case 'year':
        start = new Date(now.getFullYear(), 0, 1);
        break;
      case 'custom':
        start = startDate || new Date();
        end = endDate || new Date();
        break;
      default:
        start = new Date(now.setDate(now.getDate() - 30));
    }

    return { start, end };
  }

  /**
   * Calculer le taux de complétion
   */
  private calculateCompletionRate(statusCounts: any[]): number {
    const completed = statusCounts.find((s) => s.status === 'completed')?._count.status || 0;
    const total = statusCounts.reduce((sum, s) => sum + s._count.status, 0);
    return total > 0 ? (completed / total) * 100 : 0;
  }

  /**
   * Calculer le taux d'évaluations positives
   */
  private calculatePositiveRate(distribution: any[]): number {
    const positive = distribution
      .filter((d) => d.rating >= 4)
      .reduce((sum, d) => sum + d._count.rating, 0);
    const total = distribution.reduce((sum, d) => sum + d._count.rating, 0);
    return total > 0 ? (positive / total) * 100 : 0;
  }

  /**
   * Calculer la prochaine exécution
   */
  private calculateNextRun(cronExpression: string): Date {
    // Simplification - en production utiliser cron-parser
    return new Date(Date.now() + 24 * 60 * 60 * 1000);
  }

  private mapScheduledReport(report: any): ScheduledReport {
    return {
      id: report.id,
      name: report.name,
      type: report.type,
      format: report.format,
      schedule: report.schedule,
      recipients: JSON.parse(report.recipients || '[]'),
      filters: report.filters ? JSON.parse(report.filters) : undefined,
      active: report.active,
      lastRunAt: report.lastRunAt,
      nextRunAt: report.nextRunAt,
    };
  }
}
