import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

// Loyalty levels
const LOYALTY_LEVELS = {
  BRONZE: { name: 'Bronze', minPoints: 0, multiplier: 1.0 },
  SILVER: { name: 'Argent', minPoints: 500, multiplier: 1.2 },
  GOLD: { name: 'Or', minPoints: 2000, multiplier: 1.5 },
  PLATINUM: { name: 'Platine', minPoints: 5000, multiplier: 2.0 },
  DIAMOND: { name: 'Diamant', minPoints: 10000, multiplier: 2.5 },
};

// Points earned per action
const POINTS_CONFIG = {
  RIDE_COMPLETED: 10, // Points per completed ride
  RIDE_PER_1000_XOF: 1, // Points per 1000 XOF spent
  DELIVERY_COMPLETED: 15, // Points per completed delivery
  RATING_GIVEN: 2, // Points for rating a ride
  FIRST_RIDE: 50, // Bonus for first ride
  REFERRAL_COMPLETED: 100, // Points for successful referral
  PROFILE_COMPLETED: 20, // Points for completing profile
};

interface RewardsProfile {
  totalPoints: number;
  availablePoints: number;
  level: string;
  levelName: string;
  nextLevel?: string;
  pointsToNextLevel?: number;
  multiplier: number;
  totalRides: number;
  memberSince: Date;
}

interface RedeemableReward {
  id: number;
  name: string;
  description: string;
  pointsCost: number;
  type: 'discount' | 'free_ride' | 'gift' | 'cashback';
  value: number;
  isAvailable: boolean;
}

@Injectable()
export class RewardsService {
  private readonly logger = new Logger(RewardsService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  /**
   * Award points to user/driver
   */
  async awardPoints(
    userId: number,
    userType: 'user' | 'driver',
    action: string,
    amount: number,
    description: string,
    bookingId?: number,
    merchantId?: number,
  ): Promise<{ pointsAwarded: number; newTotal: number; levelUp?: boolean }> {
    // Get current points
    let currentPoints = 0;
    let currentLevel = 'BRONZE';

    if (userType === 'user') {
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { reward_points: true, loyalty_level: true },
      });
      currentPoints = user?.reward_points || 0;
      currentLevel = user?.loyalty_level || 'BRONZE';
    } else {
      const driver = await this.prisma.driver.findUnique({
        where: { id: userId },
        select: { reward_points: true, loyalty_level: true },
      });
      currentPoints = driver?.reward_points || 0;
      currentLevel = driver?.loyalty_level || 'BRONZE';
    }

    const newTotal = currentPoints + amount;
    const newLevel = this.calculateLevel(newTotal);
    const levelUp = newLevel !== currentLevel;

    // Update points and level
    if (userType === 'user') {
      await this.prisma.user.update({
        where: { id: userId },
        data: {
          reward_points: newTotal,
          loyalty_level: newLevel,
        },
      });
    } else {
      await this.prisma.driver.update({
        where: { id: userId },
        data: {
          reward_points: newTotal,
          loyalty_level: newLevel,
        },
      });
    }

    // Record transaction
    await this.prisma.rewardTransaction.create({
      data: {
        user_id: userType === 'user' ? userId : null,
        driver_id: userType === 'driver' ? userId : null,
        merchant_id: merchantId,
        type: 'earn',
        action,
        points: amount,
        description,
        booking_id: bookingId,
        balance_after: newTotal,
        created_at: new Date(),
      },
    });

    // Notify on level up
    if (levelUp) {
      const levelInfo = LOYALTY_LEVELS[newLevel];
      await this.notificationService.sendToUser(userId, userType, {
        title: 'Niveau superieur atteint!',
        body: `Felicitations! Vous etes maintenant niveau ${levelInfo.name}`,
        data: { type: 'level_up', level: newLevel },
      });
    }

    this.logger.log(
      `Awarded ${amount} points to ${userType} #${userId} for ${action}`,
    );

    return { pointsAwarded: amount, newTotal, levelUp };
  }

  /**
   * Award points for completed ride
   */
  async awardRidePoints(
    userId: number,
    userType: 'user' | 'driver',
    bookingId: number,
    fareAmount: number,
    merchantId: number,
  ): Promise<void> {
    // Base points for ride
    let points = POINTS_CONFIG.RIDE_COMPLETED;

    // Additional points based on fare
    points += Math.floor(fareAmount / 1000) * POINTS_CONFIG.RIDE_PER_1000_XOF;

    // Check if first ride for user
    if (userType === 'user') {
      const rideCount = await this.prisma.booking.count({
        where: { user_id: userId, booking_status: 'completed' },
      });

      if (rideCount === 1) {
        points += POINTS_CONFIG.FIRST_RIDE;
      }
    }

    // Apply level multiplier
    const profile = await this.getRewardsProfile(userId, userType);
    points = Math.round(points * profile.multiplier);

    await this.awardPoints(
      userId,
      userType,
      'ride_completed',
      points,
      `Points course #${bookingId}`,
      bookingId,
      merchantId,
    );
  }

  /**
   * Award points for delivery
   */
  async awardDeliveryPoints(
    userId: number,
    deliveryId: number,
    fareAmount: number,
    merchantId: number,
  ): Promise<void> {
    let points = POINTS_CONFIG.DELIVERY_COMPLETED;
    points += Math.floor(fareAmount / 1000) * POINTS_CONFIG.RIDE_PER_1000_XOF;

    const profile = await this.getRewardsProfile(userId, 'user');
    points = Math.round(points * profile.multiplier);

    await this.awardPoints(
      userId,
      'user',
      'delivery_completed',
      points,
      `Points livraison #${deliveryId}`,
      deliveryId,
      merchantId,
    );
  }

  /**
   * Award points for rating
   */
  async awardRatingPoints(
    userId: number,
    userType: 'user' | 'driver',
    bookingId: number,
    merchantId: number,
  ): Promise<void> {
    await this.awardPoints(
      userId,
      userType,
      'rating_given',
      POINTS_CONFIG.RATING_GIVEN,
      `Points evaluation course #${bookingId}`,
      bookingId,
      merchantId,
    );
  }

  /**
   * Get user's rewards profile
   */
  async getRewardsProfile(
    userId: number,
    userType: 'user' | 'driver',
  ): Promise<RewardsProfile> {
    let data: any;

    if (userType === 'user') {
      data = await this.prisma.user.findUnique({
        where: { id: userId },
        select: {
          reward_points: true,
          loyalty_level: true,
          created_at: true,
          _count: { select: { bookings: { where: { booking_status: 'completed' } } } },
        },
      });
    } else {
      data = await this.prisma.driver.findUnique({
        where: { id: userId },
        select: {
          reward_points: true,
          loyalty_level: true,
          created_at: true,
          total_rides: true,
        },
      });
    }

    const totalPoints = data?.reward_points || 0;
    const currentLevel = data?.loyalty_level || 'BRONZE';
    const levelInfo = LOYALTY_LEVELS[currentLevel];

    // Calculate next level
    const levels = Object.entries(LOYALTY_LEVELS);
    const currentIndex = levels.findIndex(([key]) => key === currentLevel);
    const nextLevelEntry = levels[currentIndex + 1];

    return {
      totalPoints,
      availablePoints: totalPoints, // Can be different if points are "spent"
      level: currentLevel,
      levelName: levelInfo.name,
      nextLevel: nextLevelEntry?.[0],
      pointsToNextLevel: nextLevelEntry
        ? nextLevelEntry[1].minPoints - totalPoints
        : undefined,
      multiplier: levelInfo.multiplier,
      totalRides: userType === 'user'
        ? data?._count?.bookings || 0
        : data?.total_rides || 0,
      memberSince: data?.created_at,
    };
  }

  /**
   * Get available rewards to redeem
   */
  async getAvailableRewards(
    merchantId: number,
    userPoints: number,
  ): Promise<RedeemableReward[]> {
    const rewards = await this.prisma.reward.findMany({
      where: {
        merchant_id: merchantId,
        is_active: 1,
        OR: [
          { start_date: null },
          { start_date: { lte: new Date() } },
        ],
        AND: [
          { OR: [{ end_date: null }, { end_date: { gte: new Date() } }] },
        ],
      },
      orderBy: { points_cost: 'asc' },
    });

    return rewards.map((reward) => ({
      id: reward.id,
      name: reward.name,
      description: reward.description,
      pointsCost: reward.points_cost,
      type: reward.type as 'discount' | 'free_ride' | 'gift' | 'cashback',
      value: Number(reward.value),
      isAvailable: userPoints >= reward.points_cost,
    }));
  }

  /**
   * Redeem a reward
   */
  async redeemReward(
    userId: number,
    userType: 'user' | 'driver',
    rewardId: number,
    merchantId: number,
  ): Promise<{ success: boolean; message: string; voucher?: any }> {
    // Get reward
    const reward = await this.prisma.reward.findFirst({
      where: { id: rewardId, merchant_id: merchantId, is_active: 1 },
    });

    if (!reward) {
      return { success: false, message: 'Recompense non disponible' };
    }

    // Check user points
    const profile = await this.getRewardsProfile(userId, userType);

    if (profile.availablePoints < reward.points_cost) {
      return {
        success: false,
        message: `Points insuffisants (${profile.availablePoints}/${reward.points_cost})`,
      };
    }

    // Deduct points
    if (userType === 'user') {
      await this.prisma.user.update({
        where: { id: userId },
        data: { reward_points: { decrement: reward.points_cost } },
      });
    } else {
      await this.prisma.driver.update({
        where: { id: userId },
        data: { reward_points: { decrement: reward.points_cost } },
      });
    }

    // Record transaction
    await this.prisma.rewardTransaction.create({
      data: {
        user_id: userType === 'user' ? userId : null,
        driver_id: userType === 'driver' ? userId : null,
        merchant_id: merchantId,
        type: 'redeem',
        action: 'reward_redeemed',
        points: -reward.points_cost,
        description: `Echange: ${reward.name}`,
        balance_after: profile.availablePoints - reward.points_cost,
        created_at: new Date(),
      },
    });

    // Create voucher/coupon based on reward type
    let voucher: any = null;

    if (reward.type === 'discount' || reward.type === 'free_ride') {
      // Create a promo code for the user
      const code = await this.generateVoucherCode();

      voucher = await this.prisma.userVoucher.create({
        data: {
          user_id: userType === 'user' ? userId : null,
          driver_id: userType === 'driver' ? userId : null,
          merchant_id: merchantId,
          reward_id: reward.id,
          code,
          type: reward.type,
          value: reward.value,
          expires_at: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 days
          is_used: 0,
          created_at: new Date(),
        },
      });
    } else if (reward.type === 'cashback') {
      // Credit wallet directly
      if (userType === 'user') {
        await this.prisma.user.update({
          where: { id: userId },
          data: { wallet_balance: { increment: Number(reward.value) } },
        });
      } else {
        await this.prisma.driver.update({
          where: { id: userId },
          data: { wallet_balance: { increment: Number(reward.value) } },
        });
      }

      voucher = { type: 'cashback', amount: reward.value };
    }

    this.logger.log(
      `${userType} #${userId} redeemed reward #${rewardId} for ${reward.points_cost} points`,
    );

    return {
      success: true,
      message: 'Recompense echangee avec succes!',
      voucher,
    };
  }

  /**
   * Get points history
   */
  async getPointsHistory(
    userId: number,
    userType: 'user' | 'driver',
    options?: { page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where = userType === 'user'
      ? { user_id: userId }
      : { driver_id: userId };

    const [data, total] = await Promise.all([
      this.prisma.rewardTransaction.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.rewardTransaction.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Calculate loyalty level based on points
   */
  private calculateLevel(points: number): string {
    const levels = Object.entries(LOYALTY_LEVELS).reverse();

    for (const [key, value] of levels) {
      if (points >= value.minPoints) {
        return key;
      }
    }

    return 'BRONZE';
  }

  /**
   * Generate unique voucher code
   */
  private async generateVoucherCode(): Promise<string> {
    const characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    let code: string;

    do {
      code = 'RWD';
      for (let i = 0; i < 8; i++) {
        code += characters.charAt(Math.floor(Math.random() * characters.length));
      }
    } while (
      await this.prisma.userVoucher.findFirst({ where: { code } })
    );

    return code;
  }
}
