import {
  Controller,
  Get,
  Post,
  Body,
  Query,
  UseGuards,
  ParseIntPipe,
  Param,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { ReferralService } from './referral.service';
import { RewardsService } from './rewards.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';
import { ApplyReferralDto } from './dto/apply-referral.dto';
import { RedeemRewardDto } from './dto/redeem-reward.dto';

@ApiTags('Referral & Rewards')
@Controller('api')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class ReferralController {
  constructor(
    private referralService: ReferralService,
    private rewardsService: RewardsService,
  ) {}

  // ============================================================================
  // REFERRAL ENDPOINTS
  // ============================================================================

  @Get('referral/code')
  @ApiOperation({ summary: 'Get or generate referral code' })
  async getReferralCode(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    const userType = user.type || 'user';
    const code = await this.referralService.generateReferralCode(
      user.id,
      userType,
      merchantId,
    );

    return {
      code,
      shareMessage: `Utilisez mon code ${code} pour obtenir un bonus sur votre premiere course!`,
    };
  }

  @Post('referral/apply')
  @ApiOperation({ summary: 'Apply a referral code' })
  async applyReferralCode(
    @Body() dto: ApplyReferralDto,
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    const userType = user.type || 'user';
    return this.referralService.applyReferralCode(
      user.id,
      userType,
      dto.code,
      merchantId,
    );
  }

  @Get('referral/stats')
  @ApiOperation({ summary: 'Get referral statistics' })
  async getReferralStats(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    const userType = user.type || 'user';
    return this.referralService.getReferralStats(user.id, userType, merchantId);
  }

  @Get('referral/history')
  @ApiOperation({ summary: 'Get referral history' })
  async getReferralHistory(
    @CurrentUser() user: any,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    const userType = user.type || 'user';
    return this.referralService.getReferralHistory(user.id, userType, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get('referral/settings')
  @ApiOperation({ summary: 'Get referral program settings' })
  async getReferralSettings(@MerchantId() merchantId: number) {
    return this.referralService.getReferralSettings(merchantId);
  }

  // ============================================================================
  // REWARDS ENDPOINTS
  // ============================================================================

  @Get('rewards/profile')
  @ApiOperation({ summary: 'Get rewards profile' })
  async getRewardsProfile(@CurrentUser() user: any) {
    const userType = user.type || 'user';
    return this.rewardsService.getRewardsProfile(user.id, userType);
  }

  @Get('rewards/available')
  @ApiOperation({ summary: 'Get available rewards to redeem' })
  async getAvailableRewards(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    const profile = await this.rewardsService.getRewardsProfile(
      user.id,
      user.type || 'user',
    );
    return this.rewardsService.getAvailableRewards(
      merchantId,
      profile.availablePoints,
    );
  }

  @Post('rewards/redeem')
  @ApiOperation({ summary: 'Redeem a reward' })
  async redeemReward(
    @Body() dto: RedeemRewardDto,
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    const userType = user.type || 'user';
    return this.rewardsService.redeemReward(
      user.id,
      userType,
      dto.rewardId,
      merchantId,
    );
  }

  @Get('rewards/history')
  @ApiOperation({ summary: 'Get points history' })
  async getPointsHistory(
    @CurrentUser() user: any,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    const userType = user.type || 'user';
    return this.rewardsService.getPointsHistory(user.id, userType, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get('rewards/levels')
  @ApiOperation({ summary: 'Get loyalty levels information' })
  async getLoyaltyLevels() {
    return {
      levels: [
        { key: 'BRONZE', name: 'Bronze', minPoints: 0, multiplier: 1.0, benefits: ['Points de base'] },
        { key: 'SILVER', name: 'Argent', minPoints: 500, multiplier: 1.2, benefits: ['20% de points bonus', 'Support prioritaire'] },
        { key: 'GOLD', name: 'Or', minPoints: 2000, multiplier: 1.5, benefits: ['50% de points bonus', 'Courses prioritaires'] },
        { key: 'PLATINUM', name: 'Platine', minPoints: 5000, multiplier: 2.0, benefits: ['100% de points bonus', 'Offres exclusives'] },
        { key: 'DIAMOND', name: 'Diamant', minPoints: 10000, multiplier: 2.5, benefits: ['150% de points bonus', 'Avantages VIP'] },
      ],
    };
  }
}
