import { Process, Processor, OnQueueFailed, OnQueueCompleted } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { ConfigService } from '@nestjs/config';
import { PrismaService } from '../../../shared/database/prisma.service';
import { NotificationService } from '../../notification/notification.service';
import { QUEUE_NAMES } from '../queue.module';
import { BOOKING_STATUS } from '../../booking/booking.constants';

interface PaymentJobData {
  bookingId: number;
  userId: number;
  driverId: number;
  amount: number;
  currency: string;
  paymentMethod: string;
  paymentMethodId?: string;
  merchantId: number;
}

interface RefundJobData {
  bookingId: number;
  transactionId: number;
  amount: number;
  reason: string;
}

interface EarningsJobData {
  bookingId: number;
  driverId: number;
  totalAmount: number;
  commissionRate: number;
  merchantId: number;
}

@Processor(QUEUE_NAMES.PAYMENT)
export class PaymentProcessor {
  private readonly logger = new Logger(PaymentProcessor.name);
  private stripe: any = null;

  constructor(
    private prisma: PrismaService,
    private configService: ConfigService,
    private notificationService: NotificationService,
  ) {
    this.initializeStripe();
  }

  private async initializeStripe() {
    try {
      const stripeKey = this.configService.get<string>('STRIPE_SECRET_KEY');
      if (stripeKey) {
        const Stripe = (await import('stripe')).default;
        this.stripe = new Stripe(stripeKey, { apiVersion: '2024-12-18.acacia' });
      }
    } catch (error) {
      this.logger.warn(`Stripe non initialisé: ${error.message}`);
    }
  }

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job paiement ${job.name} (${job.id}) échoué: ${error.message}`,
      error.stack,
    );
  }

  @OnQueueCompleted()
  onCompleted(job: Job, result: any) {
    this.logger.log(`Job paiement ${job.name} (${job.id}) terminé`);
  }

  /**
   * Traiter le paiement d'une course
   */
  @Process('process-payment')
  async handleProcessPayment(job: Job<PaymentJobData>) {
    const {
      bookingId,
      userId,
      driverId,
      amount,
      currency,
      paymentMethod,
      paymentMethodId,
      merchantId,
    } = job.data;

    this.logger.log(
      `Traitement paiement réservation #${bookingId}: ${amount} ${currency} via ${paymentMethod}`,
    );

    try {
      let paymentResult: { success: boolean; transactionId?: string; error?: string };

      switch (paymentMethod) {
        case 'card':
          paymentResult = await this.processCardPayment(
            userId,
            amount,
            currency,
            paymentMethodId,
            bookingId,
          );
          break;

        case 'wallet':
          paymentResult = await this.processWalletPayment(
            userId,
            amount,
            bookingId,
          );
          break;

        case 'mobile_money':
          paymentResult = await this.processMobileMoneyPayment(
            userId,
            amount,
            currency,
            bookingId,
          );
          break;

        case 'cash':
          paymentResult = { success: true, transactionId: `CASH-${bookingId}` };
          break;

        case 'corporate':
          paymentResult = await this.processCorporatePayment(
            userId,
            amount,
            bookingId,
          );
          break;

        default:
          paymentResult = { success: false, error: 'Méthode de paiement non supportée' };
      }

      if (!paymentResult.success) {
        // Mettre à jour la réservation avec échec de paiement
        await this.prisma.booking.update({
          where: { id: bookingId },
          data: { payment_status: 'failed' },
        });

        await this.notificationService.sendToUser(
          userId,
          {
            title: 'Échec du paiement',
            body: paymentResult.error || 'Le paiement n\'a pas pu être effectué',
            data: { type: 'payment_failed', booking_id: String(bookingId) },
          },
          { push: true },
        );

        return { success: false, error: paymentResult.error };
      }

      // Créer la transaction
      const transaction = await this.prisma.bookingTransaction.create({
        data: {
          booking_id: bookingId,
          user_id: userId,
          merchant_id: merchantId,
          payment_method: paymentMethod,
          amount,
          currency,
          status: 'completed',
          transaction_id: paymentResult.transactionId,
          created_at: new Date(),
        },
      });

      // Mettre à jour la réservation
      await this.prisma.booking.update({
        where: { id: bookingId },
        data: {
          payment_status: 'paid',
          paid_amount: amount,
        },
      });

      // Distribuer les gains au chauffeur
      if (driverId) {
        await this.distributeEarnings(bookingId, driverId, amount, merchantId);
      }

      this.logger.log(
        `Paiement réusssi pour réservation #${bookingId}: transaction ${transaction.id}`,
      );

      return {
        success: true,
        transactionId: transaction.id,
        externalId: paymentResult.transactionId,
      };
    } catch (error) {
      this.logger.error(`Erreur traitement paiement: ${error.message}`);
      throw error;
    }
  }

  /**
   * Traiter un remboursement
   */
  @Process('process-refund')
  async handleProcessRefund(job: Job<RefundJobData>) {
    const { bookingId, transactionId, amount, reason } = job.data;

    this.logger.log(`Traitement remboursement réservation #${bookingId}: ${amount}`);

    try {
      const transaction = await this.prisma.bookingTransaction.findUnique({
        where: { id: transactionId },
        include: { booking: true },
      });

      if (!transaction) {
        return { success: false, error: 'Transaction non trouvée' };
      }

      let refundResult: { success: boolean; refundId?: string; error?: string };

      switch (transaction.payment_method) {
        case 'card':
          refundResult = await this.refundCardPayment(
            transaction.transaction_id,
            amount,
          );
          break;

        case 'wallet':
          refundResult = await this.refundToWallet(
            transaction.user_id,
            amount,
            bookingId,
            reason,
          );
          break;

        default:
          refundResult = { success: false, error: 'Remboursement non supporté pour cette méthode' };
      }

      if (!refundResult.success) {
        return { success: false, error: refundResult.error };
      }

      // Créer la transaction de remboursement
      await this.prisma.bookingTransaction.create({
        data: {
          booking_id: bookingId,
          user_id: transaction.user_id,
          merchant_id: transaction.merchant_id,
          payment_method: transaction.payment_method,
          amount: -amount,
          currency: transaction.currency,
          status: 'refunded',
          transaction_id: refundResult.refundId,
          notes: reason,
          created_at: new Date(),
        },
      });

      // Mettre à jour la réservation
      await this.prisma.booking.update({
        where: { id: bookingId },
        data: { payment_status: 'refunded' },
      });

      await this.notificationService.sendToUser(
        transaction.user_id,
        {
          title: 'Remboursement effectué',
          body: `${amount} ${transaction.currency} ont été remboursés`,
          data: { type: 'refund', booking_id: String(bookingId) },
        },
        { push: true },
      );

      return { success: true, refundId: refundResult.refundId };
    } catch (error) {
      this.logger.error(`Erreur remboursement: ${error.message}`);
      throw error;
    }
  }

  /**
   * Distribuer les gains au chauffeur
   */
  @Process('distribute-earnings')
  async handleDistributeEarnings(job: Job<EarningsJobData>) {
    const { bookingId, driverId, totalAmount, commissionRate, merchantId } = job.data;

    return this.distributeEarnings(bookingId, driverId, totalAmount, merchantId, commissionRate);
  }

  /**
   * Traiter le paiement des pourboires
   */
  @Process('process-tip')
  async handleProcessTip(
    job: Job<{
      bookingId: number;
      driverId: number;
      userId: number;
      amount: number;
      currency: string;
    }>,
  ) {
    const { bookingId, driverId, userId, amount, currency } = job.data;

    this.logger.log(`Traitement pourboire réservation #${bookingId}: ${amount} ${currency}`);

    try {
      // Vérifier le solde wallet de l'utilisateur
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });

      if (!user || user.wallet_balance < amount) {
        return { success: false, error: 'Solde insuffisant' };
      }

      await this.prisma.$transaction([
        // Débiter l'utilisateur
        this.prisma.user.update({
          where: { id: userId },
          data: { wallet_balance: { decrement: amount } },
        }),
        // Créditer le chauffeur (100% du pourboire)
        this.prisma.driver.update({
          where: { id: driverId },
          data: { wallet_balance: { increment: amount } },
        }),
        // Enregistrer la transaction utilisateur
        this.prisma.userWalletTransaction.create({
          data: {
            user_id: userId,
            amount: -amount,
            type: 'tip',
            description: `Pourboire course #${bookingId}`,
            reference_id: bookingId,
            created_at: new Date(),
          },
        }),
        // Enregistrer la transaction chauffeur
        this.prisma.driverWalletTransaction.create({
          data: {
            driver_id: driverId,
            amount,
            type: 'tip',
            description: `Pourboire course #${bookingId}`,
            reference_id: bookingId,
            created_at: new Date(),
          },
        }),
        // Mettre à jour le pourboire de la réservation
        this.prisma.booking.update({
          where: { id: bookingId },
          data: { tip_amount: { increment: amount } },
        }),
      ]);

      await this.notificationService.sendToDriver(
        driverId,
        {
          title: 'Pourboire reçu!',
          body: `Vous avez reçu ${amount} ${currency} de pourboire`,
          data: { type: 'tip_received', booking_id: String(bookingId) },
        },
        { push: true },
      );

      return { success: true };
    } catch (error) {
      this.logger.error(`Erreur traitement pourboire: ${error.message}`);
      throw error;
    }
  }

  // =============================================================================
  // MÉTHODES PRIVÉES DE PAIEMENT
  // =============================================================================

  private async processCardPayment(
    userId: number,
    amount: number,
    currency: string,
    paymentMethodId: string,
    bookingId: number,
  ): Promise<{ success: boolean; transactionId?: string; error?: string }> {
    if (!this.stripe) {
      return { success: false, error: 'Stripe non configuré' };
    }

    try {
      // Récupérer le customer Stripe de l'utilisateur
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { stripe_customer_id: true },
      });

      if (!user?.stripe_customer_id) {
        return { success: false, error: 'Aucune carte enregistrée' };
      }

      const paymentIntent = await this.stripe.paymentIntents.create({
        amount: Math.round(amount * 100), // Stripe utilise les centimes
        currency: currency.toLowerCase(),
        customer: user.stripe_customer_id,
        payment_method: paymentMethodId,
        off_session: true,
        confirm: true,
        metadata: {
          booking_id: String(bookingId),
          user_id: String(userId),
        },
      });

      if (paymentIntent.status === 'succeeded') {
        return { success: true, transactionId: paymentIntent.id };
      }

      return { success: false, error: `Statut paiement: ${paymentIntent.status}` };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }

  private async processWalletPayment(
    userId: number,
    amount: number,
    bookingId: number,
  ): Promise<{ success: boolean; transactionId?: string; error?: string }> {
    try {
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });

      if (!user || user.wallet_balance < amount) {
        return { success: false, error: 'Solde wallet insuffisant' };
      }

      // Débiter le wallet
      await this.prisma.$transaction([
        this.prisma.user.update({
          where: { id: userId },
          data: { wallet_balance: { decrement: amount } },
        }),
        this.prisma.userWalletTransaction.create({
          data: {
            user_id: userId,
            amount: -amount,
            type: 'payment',
            description: `Paiement course #${bookingId}`,
            reference_id: bookingId,
            created_at: new Date(),
          },
        }),
      ]);

      return { success: true, transactionId: `WALLET-${bookingId}-${Date.now()}` };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }

  private async processMobileMoneyPayment(
    userId: number,
    amount: number,
    currency: string,
    bookingId: number,
  ): Promise<{ success: boolean; transactionId?: string; error?: string }> {
    // TODO: Intégration Mobile Money (M-Pesa, MTN, etc.)
    this.logger.warn('Paiement Mobile Money non implémenté');
    return { success: false, error: 'Mobile Money en cours d\'intégration' };
  }

  private async processCorporatePayment(
    userId: number,
    amount: number,
    bookingId: number,
  ): Promise<{ success: boolean; transactionId?: string; error?: string }> {
    try {
      // Récupérer le compte entreprise de l'utilisateur
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        select: { corporate_id: true },
      });

      if (!booking?.corporate_id) {
        return { success: false, error: 'Aucun compte entreprise associé' };
      }

      // Vérifier le solde/crédit entreprise
      const corporate = await this.prisma.corporate.findUnique({
        where: { id: booking.corporate_id },
        select: { credit_balance: true, credit_limit: true },
      });

      if (!corporate) {
        return { success: false, error: 'Compte entreprise non trouvé' };
      }

      const availableCredit = (corporate.credit_limit || 0) + (corporate.credit_balance || 0);
      if (availableCredit < amount) {
        return { success: false, error: 'Crédit entreprise insuffisant' };
      }

      // Débiter le compte entreprise
      await this.prisma.corporate.update({
        where: { id: booking.corporate_id },
        data: { credit_balance: { decrement: amount } },
      });

      return { success: true, transactionId: `CORP-${booking.corporate_id}-${bookingId}` };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }

  private async refundCardPayment(
    transactionId: string,
    amount: number,
  ): Promise<{ success: boolean; refundId?: string; error?: string }> {
    if (!this.stripe) {
      return { success: false, error: 'Stripe non configuré' };
    }

    try {
      const refund = await this.stripe.refunds.create({
        payment_intent: transactionId,
        amount: Math.round(amount * 100),
      });

      return { success: true, refundId: refund.id };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }

  private async refundToWallet(
    userId: number,
    amount: number,
    bookingId: number,
    reason: string,
  ): Promise<{ success: boolean; refundId?: string; error?: string }> {
    try {
      await this.prisma.$transaction([
        this.prisma.user.update({
          where: { id: userId },
          data: { wallet_balance: { increment: amount } },
        }),
        this.prisma.userWalletTransaction.create({
          data: {
            user_id: userId,
            amount,
            type: 'refund',
            description: `Remboursement course #${bookingId}: ${reason}`,
            reference_id: bookingId,
            created_at: new Date(),
          },
        }),
      ]);

      return { success: true, refundId: `REFUND-${bookingId}-${Date.now()}` };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }

  private async distributeEarnings(
    bookingId: number,
    driverId: number,
    totalAmount: number,
    merchantId: number,
    commissionRate?: number,
  ): Promise<{ success: boolean; driverEarnings: number; commission: number }> {
    try {
      // Récupérer le taux de commission si non fourni
      if (commissionRate === undefined) {
        const merchant = await this.prisma.merchant.findUnique({
          where: { id: merchantId },
          select: { commission_rate: true },
        });
        commissionRate = merchant?.commission_rate || 20;
      }

      const commission = totalAmount * (commissionRate / 100);
      const driverEarnings = totalAmount - commission;

      await this.prisma.$transaction([
        // Créditer le wallet du chauffeur
        this.prisma.driver.update({
          where: { id: driverId },
          data: {
            wallet_balance: { increment: driverEarnings },
            total_earnings: { increment: driverEarnings },
          },
        }),
        // Enregistrer la transaction
        this.prisma.driverWalletTransaction.create({
          data: {
            driver_id: driverId,
            amount: driverEarnings,
            type: 'earning',
            description: `Gain course #${bookingId}`,
            reference_id: bookingId,
            created_at: new Date(),
          },
        }),
        // Mettre à jour la réservation
        this.prisma.booking.update({
          where: { id: bookingId },
          data: {
            driver_commission: driverEarnings,
            admin_commission: commission,
          },
        }),
      ]);

      await this.notificationService.sendToDriver(
        driverId,
        {
          title: 'Nouveau gain!',
          body: `+${driverEarnings.toFixed(0)} XOF pour la course`,
          data: { type: 'earning', booking_id: String(bookingId) },
        },
        { push: true },
      );

      return { success: true, driverEarnings, commission };
    } catch (error) {
      this.logger.error(`Erreur distribution gains: ${error.message}`);
      return { success: false, driverEarnings: 0, commission: 0 };
    }
  }
}
