import { Process, Processor, OnQueueFailed } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { ConfigService } from '@nestjs/config';
import * as nodemailer from 'nodemailer';
import { PrismaService } from '../../../shared/database/prisma.service';
import { QUEUE_NAMES } from '../queue.module';

@Processor(QUEUE_NAMES.EMAIL)
export class EmailQueueProcessor {
  private readonly logger = new Logger(EmailQueueProcessor.name);
  private transporter: nodemailer.Transporter;

  constructor(
    private prisma: PrismaService,
    private configService: ConfigService,
  ) {
    // Initialize email transporter
    this.transporter = nodemailer.createTransport({
      host: this.configService.get<string>('MAIL_HOST', 'smtp.gmail.com'),
      port: this.configService.get<number>('MAIL_PORT', 587),
      secure: false,
      auth: {
        user: this.configService.get<string>('MAIL_USER'),
        pass: this.configService.get<string>('MAIL_PASSWORD'),
      },
    });
  }

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job ${job.name} (${job.id}) failed: ${error.message}`,
      error.stack,
    );
  }

  /**
   * Send generic email
   */
  @Process('send-email')
  async handleSendEmail(
    job: Job<{
      to: string;
      subject: string;
      template: string;
      context: Record<string, any>;
    }>,
  ) {
    const { to, subject, template, context } = job.data;
    this.logger.log(`Sending email to ${to}: ${subject}`);

    try {
      // TODO: Implement template rendering
      const html = this.renderTemplate(template, context);

      await this.transporter.sendMail({
        from: this.configService.get<string>('MAIL_FROM', 'noreply@monkapi.com'),
        to,
        subject,
        html,
      });

      this.logger.log(`Email sent to ${to}`);
      return { success: true };
    } catch (error) {
      this.logger.error(`Email failed: ${error.message}`);
      throw error;
    }
  }

  /**
   * Send booking invoice email
   */
  @Process('send-invoice')
  async handleSendInvoice(
    job: Job<{ bookingId: number; userEmail: string }>,
  ) {
    const { bookingId, userEmail } = job.data;
    this.logger.log(`Sending invoice for booking #${bookingId} to ${userEmail}`);

    try {
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        include: {
          user: true,
          driver: true,
        },
      });

      if (!booking) {
        return { success: false, reason: 'Booking not found' };
      }

      const html = this.renderInvoiceTemplate(booking);

      await this.transporter.sendMail({
        from: this.configService.get<string>('MAIL_FROM'),
        to: userEmail,
        subject: `Facture - Course #${booking.merchant_booking_id}`,
        html,
      });

      this.logger.log(`Invoice sent for booking #${bookingId}`);
      return { success: true };
    } catch (error) {
      this.logger.error(`Invoice email failed: ${error.message}`);
      throw error;
    }
  }

  /**
   * Send welcome email
   */
  @Process('send-welcome')
  async handleSendWelcome(
    job: Job<{
      userId: number;
      userType: 'user' | 'driver';
      email: string;
      name: string;
    }>,
  ) {
    const { userId, userType, email, name } = job.data;
    this.logger.log(`Sending welcome email to ${email}`);

    try {
      const subject = userType === 'user'
        ? 'Bienvenue sur MonkAPI!'
        : 'Bienvenue dans notre equipe de chauffeurs!';

      const html = this.renderWelcomeTemplate(name, userType);

      await this.transporter.sendMail({
        from: this.configService.get<string>('MAIL_FROM'),
        to: email,
        subject,
        html,
      });

      this.logger.log(`Welcome email sent to ${email}`);
      return { success: true };
    } catch (error) {
      this.logger.error(`Welcome email failed: ${error.message}`);
      throw error;
    }
  }

  // ============================================================================
  // TEMPLATE HELPERS
  // ============================================================================

  private renderTemplate(template: string, context: Record<string, any>): string {
    // Simple template rendering
    // TODO: Use proper template engine like Handlebars
    let html = template;
    for (const [key, value] of Object.entries(context)) {
      html = html.replace(new RegExp(`{{${key}}}`, 'g'), String(value));
    }
    return html;
  }

  private renderInvoiceTemplate(booking: any): string {
    return `
      <!DOCTYPE html>
      <html>
      <head>
        <style>
          body { font-family: Arial, sans-serif; }
          .invoice { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { text-align: center; border-bottom: 2px solid #333; padding-bottom: 20px; }
          .details { margin: 20px 0; }
          .row { display: flex; justify-content: space-between; padding: 10px 0; border-bottom: 1px solid #eee; }
          .total { font-size: 1.2em; font-weight: bold; }
        </style>
      </head>
      <body>
        <div class="invoice">
          <div class="header">
            <h1>MonkAPI</h1>
            <p>Facture de course</p>
          </div>

          <div class="details">
            <div class="row">
              <span>Numero de course:</span>
              <span>${booking.merchant_booking_id}</span>
            </div>
            <div class="row">
              <span>Date:</span>
              <span>${new Date(booking.completed_time).toLocaleDateString('fr-FR')}</span>
            </div>
            <div class="row">
              <span>Depart:</span>
              <span>${booking.pickup_address}</span>
            </div>
            <div class="row">
              <span>Arrivee:</span>
              <span>${booking.drop_address}</span>
            </div>
            <div class="row">
              <span>Distance:</span>
              <span>${booking.travel_distance} km</span>
            </div>
            <div class="row">
              <span>Duree:</span>
              <span>${booking.travel_time} min</span>
            </div>
            <div class="row">
              <span>Chauffeur:</span>
              <span>${booking.driver?.first_name} ${booking.driver?.last_name}</span>
            </div>
            <div class="row total">
              <span>Total:</span>
              <span>${booking.final_amount} XOF</span>
            </div>
          </div>

          <p style="text-align: center; color: #666;">
            Merci d'avoir choisi MonkAPI!
          </p>
        </div>
      </body>
      </html>
    `;
  }

  private renderWelcomeTemplate(name: string, userType: 'user' | 'driver'): string {
    const greeting = userType === 'user'
      ? `Merci de nous avoir rejoints! Commencez a reserver vos courses des maintenant.`
      : `Bienvenue dans notre equipe! Votre compte est en cours de verification.`;

    return `
      <!DOCTYPE html>
      <html>
      <head>
        <style>
          body { font-family: Arial, sans-serif; }
          .container { max-width: 600px; margin: 0 auto; padding: 20px; }
          .header { text-align: center; background: #4CAF50; color: white; padding: 30px; }
          .content { padding: 30px; text-align: center; }
          .button { display: inline-block; background: #4CAF50; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <h1>Bienvenue, ${name}!</h1>
          </div>

          <div class="content">
            <p>${greeting}</p>
            <p>Si vous avez des questions, n'hesitez pas a nous contacter.</p>
            <a href="#" class="button">Ouvrir l'application</a>
          </div>
        </div>
      </body>
      </html>
    `;
  }
}
