import { Process, Processor, OnQueueFailed, OnQueueCompleted } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { PrismaService } from '../../../shared/database/prisma.service';
import { NotificationService } from '../../notification/notification.service';
import { PusherService } from '../../notification/pusher.service';
import { OneSignalService } from '../../notification/onesignal.service';
import { QUEUE_NAMES } from '../queue.module';
import { BOOKING_STATUS } from '../../booking/booking.constants';

interface DriverSearchJobData {
  bookingId: number;
  merchantId: number;
  vehicleTypeId: number;
  pickupLatitude: number;
  pickupLongitude: number;
  currentRadius: number;
  maxRadius: number;
  attempt: number;
  excludedDriverIds: number[];
  userId: number;
}

interface NearbyDriver {
  id: number;
  first_name: string;
  last_name: string;
  latitude: number;
  longitude: number;
  average_rating: number;
  distance: number;
  player_id?: string;
}

@Processor(QUEUE_NAMES.DRIVER_SEARCH)
export class DriverSearchProcessor {
  private readonly logger = new Logger(DriverSearchProcessor.name);

  // Configuration de recherche
  private readonly DRIVER_RESPONSE_TIMEOUT = 30000; // 30 secondes pour répondre
  private readonly RADIUS_INCREMENT = 2; // km
  private readonly MAX_DRIVERS_PER_REQUEST = 5;

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
    private pusherService: PusherService,
    private oneSignalService: OneSignalService,
  ) {}

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job recherche chauffeur ${job.id} échoué: ${error.message}`,
      error.stack,
    );
  }

  @OnQueueCompleted()
  onCompleted(job: Job, result: any) {
    this.logger.log(`Job recherche chauffeur ${job.id} terminé: ${JSON.stringify(result)}`);
  }

  /**
   * Recherche de chauffeurs disponibles à proximité
   */
  @Process('search-drivers')
  async handleDriverSearch(job: Job<DriverSearchJobData>) {
    const {
      bookingId,
      merchantId,
      vehicleTypeId,
      pickupLatitude,
      pickupLongitude,
      currentRadius,
      maxRadius,
      attempt,
      excludedDriverIds,
      userId,
    } = job.data;

    this.logger.log(
      `Recherche chauffeurs pour réservation #${bookingId} - rayon: ${currentRadius}km, tentative: ${attempt}`,
    );

    try {
      // Vérifier que la réservation est toujours en attente
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        include: { user: true },
      });

      if (!booking) {
        return { success: false, reason: 'Réservation non trouvée' };
      }

      // Si la réservation n'est plus en recherche, arrêter
      if (booking.booking_status !== BOOKING_STATUS.SEARCHING) {
        this.logger.log(`Réservation #${bookingId} n'est plus en recherche (statut: ${booking.booking_status})`);
        return { success: false, reason: 'Réservation plus en recherche' };
      }

      // Trouver les chauffeurs disponibles dans le rayon actuel
      const nearbyDrivers = await this.findNearbyDrivers(
        merchantId,
        vehicleTypeId,
        pickupLatitude,
        pickupLongitude,
        currentRadius,
        excludedDriverIds,
      );

      if (nearbyDrivers.length === 0) {
        // Aucun chauffeur trouvé, augmenter le rayon ou abandonner
        if (currentRadius >= maxRadius) {
          this.logger.warn(`Aucun chauffeur trouvé pour réservation #${bookingId} après rayon max`);

          // Marquer la réservation comme sans chauffeur
          await this.prisma.booking.update({
            where: { id: bookingId },
            data: { booking_status: BOOKING_STATUS.NO_DRIVER },
          });

          // Notifier l'utilisateur
          await this.notificationService.sendToUser(
            userId,
            {
              title: 'Aucun chauffeur disponible',
              body: 'Désolé, aucun chauffeur n\'est disponible actuellement. Veuillez réessayer plus tard.',
              data: { type: 'no_driver', booking_id: String(bookingId) },
            },
            { push: true, inApp: true },
          );

          return { success: false, reason: 'Aucun chauffeur disponible' };
        }

        // Relancer avec un rayon plus grand
        this.logger.log(`Extension du rayon de recherche pour réservation #${bookingId}`);
        return {
          success: false,
          reason: 'Extension rayon nécessaire',
          nextRadius: currentRadius + this.RADIUS_INCREMENT,
        };
      }

      // Envoyer la demande aux chauffeurs trouvés
      const driversNotified = await this.notifyDrivers(nearbyDrivers, booking);

      // Créer les entrées DriverRequest
      await this.createDriverRequests(bookingId, nearbyDrivers);

      this.logger.log(
        `${driversNotified} chauffeurs notifiés pour réservation #${bookingId}`,
      );

      return {
        success: true,
        driversNotified,
        radius: currentRadius,
        driverIds: nearbyDrivers.map((d) => d.id),
      };
    } catch (error) {
      this.logger.error(`Erreur recherche chauffeur: ${error.message}`);
      throw error;
    }
  }

  /**
   * Traiter la réponse d'un chauffeur
   */
  @Process('driver-response')
  async handleDriverResponse(
    job: Job<{
      bookingId: number;
      driverId: number;
      accepted: boolean;
      merchantId: number;
    }>,
  ) {
    const { bookingId, driverId, accepted, merchantId } = job.data;

    this.logger.log(
      `Réponse chauffeur #${driverId} pour réservation #${bookingId}: ${accepted ? 'ACCEPTÉE' : 'REFUSÉE'}`,
    );

    try {
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
        include: { user: true },
      });

      if (!booking || booking.booking_status !== BOOKING_STATUS.SEARCHING) {
        return { success: false, reason: 'Réservation plus disponible' };
      }

      // Mettre à jour la demande chauffeur
      await this.prisma.driverRequest.updateMany({
        where: {
          booking_id: bookingId,
          driver_id: driverId,
        },
        data: {
          status: accepted ? 'accepted' : 'rejected',
          responded_at: new Date(),
        },
      });

      if (!accepted) {
        return { success: true, action: 'rejected' };
      }

      // Chauffeur a accepté - assigner à la réservation
      const driver = await this.prisma.driver.findUnique({
        where: { id: driverId },
        include: {
          driverVehicles: {
            where: { is_default: 1 },
            take: 1,
          },
        },
      });

      if (!driver) {
        return { success: false, reason: 'Chauffeur non trouvé' };
      }

      // Vérifier que le chauffeur est toujours disponible
      if (driver.free_busy !== 2 || driver.is_online !== 1) {
        return { success: false, reason: 'Chauffeur plus disponible' };
      }

      // Assigner le chauffeur
      await this.prisma.$transaction([
        // Mettre à jour la réservation
        this.prisma.booking.update({
          where: { id: bookingId },
          data: {
            driver_id: driverId,
            booking_status: BOOKING_STATUS.ACCEPTED,
            accepted_time: new Date(),
          },
        }),
        // Marquer le chauffeur comme occupé
        this.prisma.driver.update({
          where: { id: driverId },
          data: { free_busy: 1 },
        }),
        // Annuler les autres demandes
        this.prisma.driverRequest.updateMany({
          where: {
            booking_id: bookingId,
            driver_id: { not: driverId },
            status: 'pending',
          },
          data: { status: 'cancelled' },
        }),
      ]);

      // Notifier l'utilisateur
      const vehicle = driver.driverVehicles?.[0];

      await this.pusherService.broadcastBookingAccepted(
        booking.user_id,
        booking,
        driver,
        vehicle,
      );

      await this.notificationService.sendToUser(
        booking.user_id,
        {
          title: 'Chauffeur en route!',
          body: `${driver.first_name} arrive dans quelques minutes`,
          data: {
            type: 'booking_accepted',
            booking_id: String(bookingId),
            driver_id: String(driverId),
          },
        },
        { push: true, inApp: true },
      );

      this.logger.log(`Chauffeur #${driverId} assigné à réservation #${bookingId}`);

      return {
        success: true,
        action: 'assigned',
        driverId,
      };
    } catch (error) {
      this.logger.error(`Erreur traitement réponse chauffeur: ${error.message}`);
      throw error;
    }
  }

  /**
   * Vérifier les demandes expirées
   */
  @Process('check-request-timeout')
  async handleRequestTimeout(
    job: Job<{
      bookingId: number;
      driverIds: number[];
      merchantId: number;
      nextRadius: number;
      maxRadius: number;
      excludedDriverIds: number[];
    }>,
  ) {
    const { bookingId, driverIds, merchantId, nextRadius, maxRadius, excludedDriverIds } = job.data;

    this.logger.log(`Vérification timeout pour réservation #${bookingId}`);

    try {
      // Vérifier si la réservation a été acceptée
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
      });

      if (!booking || booking.booking_status !== BOOKING_STATUS.SEARCHING) {
        return { success: true, reason: 'Réservation plus en recherche' };
      }

      // Marquer les demandes sans réponse comme expirées
      await this.prisma.driverRequest.updateMany({
        where: {
          booking_id: bookingId,
          driver_id: { in: driverIds },
          status: 'pending',
        },
        data: { status: 'expired' },
      });

      // Ajouter les chauffeurs qui n'ont pas répondu à la liste d'exclusion
      const newExcludedDrivers = [...excludedDriverIds, ...driverIds];

      return {
        success: true,
        action: 'continue_search',
        nextRadius,
        excludedDriverIds: newExcludedDrivers,
      };
    } catch (error) {
      this.logger.error(`Erreur vérification timeout: ${error.message}`);
      throw error;
    }
  }

  // =============================================================================
  // MÉTHODES PRIVÉES
  // =============================================================================

  /**
   * Trouver les chauffeurs disponibles à proximité
   */
  private async findNearbyDrivers(
    merchantId: number,
    vehicleTypeId: number,
    latitude: number,
    longitude: number,
    radiusKm: number,
    excludeIds: number[],
  ): Promise<NearbyDriver[]> {
    // Calcul du bounding box pour la requête
    const latDelta = radiusKm / 111; // 1 degré ≈ 111 km
    const lngDelta = radiusKm / (111 * Math.cos(latitude * (Math.PI / 180)));

    const minLat = latitude - latDelta;
    const maxLat = latitude + latDelta;
    const minLng = longitude - lngDelta;
    const maxLng = longitude + lngDelta;

    // Requête pour les chauffeurs disponibles
    const drivers = await this.prisma.driver.findMany({
      where: {
        merchant_id: merchantId,
        driver_status: 1, // Approuvé
        is_online: 1, // En ligne
        free_busy: 2, // Libre
        latitude: { gte: minLat, lte: maxLat },
        longitude: { gte: minLng, lte: maxLng },
        id: { notIn: excludeIds.length > 0 ? excludeIds : undefined },
        // Vérifier le type de véhicule
        driverVehicles: {
          some: {
            vehicle_type_id: vehicleTypeId,
            is_default: 1,
            status: 1,
          },
        },
      },
      select: {
        id: true,
        first_name: true,
        last_name: true,
        latitude: true,
        longitude: true,
        average_rating: true,
        driverDevices: {
          where: { is_active: 1 },
          select: { player_id: true },
          take: 1,
        },
      },
      take: this.MAX_DRIVERS_PER_REQUEST * 2, // Prendre plus pour filtrer ensuite
    });

    // Calculer la distance réelle et filtrer
    const driversWithDistance = drivers
      .map((driver) => ({
        ...driver,
        distance: this.calculateDistance(
          latitude,
          longitude,
          driver.latitude,
          driver.longitude,
        ),
        player_id: driver.driverDevices?.[0]?.player_id,
      }))
      .filter((d) => d.distance <= radiusKm)
      .sort((a, b) => {
        // Trier par distance puis par note
        const distanceDiff = a.distance - b.distance;
        if (Math.abs(distanceDiff) < 0.5) {
          // Si distance similaire, préférer meilleure note
          return (b.average_rating || 0) - (a.average_rating || 0);
        }
        return distanceDiff;
      })
      .slice(0, this.MAX_DRIVERS_PER_REQUEST);

    return driversWithDistance;
  }

  /**
   * Notifier les chauffeurs d'une nouvelle demande
   */
  private async notifyDrivers(drivers: NearbyDriver[], booking: any): Promise<number> {
    let notified = 0;

    for (const driver of drivers) {
      try {
        // Envoyer via Pusher (temps réel)
        await this.pusherService.broadcastNewBookingRequest(driver.id, booking);

        // Envoyer push notification via OneSignal
        if (driver.player_id) {
          await this.oneSignalService.notifyNewBookingRequest(driver.player_id, booking);
        }

        notified++;
      } catch (error) {
        this.logger.error(
          `Erreur notification chauffeur #${driver.id}: ${error.message}`,
        );
      }
    }

    return notified;
  }

  /**
   * Créer les entrées de demande chauffeur
   */
  private async createDriverRequests(
    bookingId: number,
    drivers: NearbyDriver[],
  ): Promise<void> {
    const requests = drivers.map((driver) => ({
      booking_id: bookingId,
      driver_id: driver.id,
      status: 'pending',
      distance_km: driver.distance,
      created_at: new Date(),
    }));

    await this.prisma.driverRequest.createMany({
      data: requests,
      skipDuplicates: true,
    });
  }

  /**
   * Calculer la distance entre deux points (formule Haversine)
   */
  private calculateDistance(
    lat1: number,
    lon1: number,
    lat2: number,
    lon2: number,
  ): number {
    const R = 6371; // Rayon de la Terre en km
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLon / 2) *
        Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }
}
