import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

interface PromoValidationResult {
  valid: boolean;
  discount: number;
  message: string;
  promoId?: number;
  discountType?: 'percentage' | 'fixed';
  discountValue?: number;
}

interface PromoCodeData {
  code: string;
  discountType: 'percentage' | 'fixed';
  discountValue: number;
  maxDiscount?: number;
  minOrderValue?: number;
  maxUses?: number;
  maxUsesPerUser?: number;
  startDate: Date;
  endDate: Date;
  vehicleTypeIds?: number[];
  userIds?: number[];
  isFirstRideOnly?: boolean;
  description?: string;
}

@Injectable()
export class PromoService {
  private readonly logger = new Logger(PromoService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Validate and apply a promo code
   */
  async validatePromoCode(
    code: string,
    merchantId: number,
    userId: number,
    subtotal: number,
    vehicleTypeId?: number,
  ): Promise<PromoValidationResult> {
    const normalizedCode = code.toUpperCase().trim();

    // Find promo code
    const promo = await this.prisma.promoCode.findFirst({
      where: {
        code: normalizedCode,
        merchant_id: merchantId,
        is_active: 1,
      },
    });

    if (!promo) {
      return {
        valid: false,
        discount: 0,
        message: 'Code promo invalide',
      };
    }

    // Check date validity
    const now = new Date();
    if (promo.start_date && now < new Date(promo.start_date)) {
      return {
        valid: false,
        discount: 0,
        message: 'Ce code promo n\'est pas encore actif',
      };
    }

    if (promo.end_date && now > new Date(promo.end_date)) {
      return {
        valid: false,
        discount: 0,
        message: 'Ce code promo a expire',
      };
    }

    // Check global usage limit
    if (promo.max_uses && promo.used_count >= promo.max_uses) {
      return {
        valid: false,
        discount: 0,
        message: 'Ce code promo a atteint sa limite d\'utilisation',
      };
    }

    // Check per-user usage limit
    if (promo.max_uses_per_user) {
      const userUsageCount = await this.prisma.promoCodeUsage.count({
        where: {
          promo_code_id: promo.id,
          user_id: userId,
        },
      });

      if (userUsageCount >= promo.max_uses_per_user) {
        return {
          valid: false,
          discount: 0,
          message: 'Vous avez deja utilise ce code promo',
        };
      }
    }

    // Check minimum order value
    if (promo.min_order_value && subtotal < Number(promo.min_order_value)) {
      return {
        valid: false,
        discount: 0,
        message: `Montant minimum requis: ${promo.min_order_value} XOF`,
      };
    }

    // Check if first ride only
    if (promo.is_first_ride_only) {
      const previousRides = await this.prisma.booking.count({
        where: {
          user_id: userId,
          booking_status: 'completed',
        },
      });

      if (previousRides > 0) {
        return {
          valid: false,
          discount: 0,
          message: 'Ce code est reserve aux nouvelles courses',
        };
      }
    }

    // Check vehicle type restriction
    if (promo.vehicle_type_ids && vehicleTypeId) {
      const allowedTypes = JSON.parse(promo.vehicle_type_ids);
      if (Array.isArray(allowedTypes) && !allowedTypes.includes(vehicleTypeId)) {
        return {
          valid: false,
          discount: 0,
          message: 'Ce code n\'est pas valide pour ce type de vehicule',
        };
      }
    }

    // Check user restriction
    if (promo.user_ids) {
      const allowedUsers = JSON.parse(promo.user_ids);
      if (Array.isArray(allowedUsers) && !allowedUsers.includes(userId)) {
        return {
          valid: false,
          discount: 0,
          message: 'Ce code n\'est pas valide pour votre compte',
        };
      }
    }

    // Calculate discount
    let discount = 0;
    const discountType = promo.discount_type as 'percentage' | 'fixed';
    const discountValue = Number(promo.discount_value);

    if (discountType === 'percentage') {
      discount = Math.round(subtotal * (discountValue / 100));
      // Apply max discount cap
      if (promo.max_discount && discount > Number(promo.max_discount)) {
        discount = Number(promo.max_discount);
      }
    } else {
      discount = discountValue;
    }

    // Ensure discount doesn't exceed subtotal
    discount = Math.min(discount, subtotal);

    return {
      valid: true,
      discount,
      message: `Reduction de ${discount} XOF appliquee`,
      promoId: promo.id,
      discountType,
      discountValue,
    };
  }

  /**
   * Record promo code usage after booking completion
   */
  async recordUsage(
    promoCodeId: number,
    userId: number,
    bookingId: number,
    discountAmount: number,
  ): Promise<void> {
    await this.prisma.$transaction([
      // Create usage record
      this.prisma.promoCodeUsage.create({
        data: {
          promo_code_id: promoCodeId,
          user_id: userId,
          booking_id: bookingId,
          discount_amount: discountAmount,
          used_at: new Date(),
        },
      }),
      // Increment usage counter
      this.prisma.promoCode.update({
        where: { id: promoCodeId },
        data: {
          used_count: { increment: 1 },
        },
      }),
    ]);

    this.logger.log(`Promo code #${promoCodeId} used by user #${userId} for booking #${bookingId}`);
  }

  /**
   * Create a new promo code
   */
  async createPromoCode(
    merchantId: number,
    data: PromoCodeData,
  ): Promise<any> {
    // Check if code already exists
    const existing = await this.prisma.promoCode.findFirst({
      where: {
        code: data.code.toUpperCase(),
        merchant_id: merchantId,
      },
    });

    if (existing) {
      throw new BadRequestException('Ce code promo existe deja');
    }

    return this.prisma.promoCode.create({
      data: {
        merchant_id: merchantId,
        code: data.code.toUpperCase(),
        discount_type: data.discountType,
        discount_value: data.discountValue,
        max_discount: data.maxDiscount,
        min_order_value: data.minOrderValue,
        max_uses: data.maxUses,
        max_uses_per_user: data.maxUsesPerUser || 1,
        start_date: data.startDate,
        end_date: data.endDate,
        vehicle_type_ids: data.vehicleTypeIds ? JSON.stringify(data.vehicleTypeIds) : null,
        user_ids: data.userIds ? JSON.stringify(data.userIds) : null,
        is_first_ride_only: data.isFirstRideOnly ? 1 : 0,
        description: data.description,
        is_active: 1,
        used_count: 0,
        created_at: new Date(),
      },
    });
  }

  /**
   * Update a promo code
   */
  async updatePromoCode(
    promoCodeId: number,
    merchantId: number,
    data: Partial<PromoCodeData>,
  ): Promise<any> {
    const promo = await this.prisma.promoCode.findFirst({
      where: {
        id: promoCodeId,
        merchant_id: merchantId,
      },
    });

    if (!promo) {
      throw new BadRequestException('Code promo non trouve');
    }

    return this.prisma.promoCode.update({
      where: { id: promoCodeId },
      data: {
        ...(data.discountType && { discount_type: data.discountType }),
        ...(data.discountValue !== undefined && { discount_value: data.discountValue }),
        ...(data.maxDiscount !== undefined && { max_discount: data.maxDiscount }),
        ...(data.minOrderValue !== undefined && { min_order_value: data.minOrderValue }),
        ...(data.maxUses !== undefined && { max_uses: data.maxUses }),
        ...(data.maxUsesPerUser !== undefined && { max_uses_per_user: data.maxUsesPerUser }),
        ...(data.startDate && { start_date: data.startDate }),
        ...(data.endDate && { end_date: data.endDate }),
        ...(data.vehicleTypeIds && { vehicle_type_ids: JSON.stringify(data.vehicleTypeIds) }),
        ...(data.userIds && { user_ids: JSON.stringify(data.userIds) }),
        ...(data.isFirstRideOnly !== undefined && { is_first_ride_only: data.isFirstRideOnly ? 1 : 0 }),
        ...(data.description !== undefined && { description: data.description }),
        updated_at: new Date(),
      },
    });
  }

  /**
   * Deactivate a promo code
   */
  async deactivatePromoCode(promoCodeId: number, merchantId: number): Promise<void> {
    await this.prisma.promoCode.updateMany({
      where: {
        id: promoCodeId,
        merchant_id: merchantId,
      },
      data: {
        is_active: 0,
        updated_at: new Date(),
      },
    });
  }

  /**
   * Get all promo codes for a merchant
   */
  async getPromoCodes(
    merchantId: number,
    options?: {
      activeOnly?: boolean;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };
    if (options?.activeOnly) {
      where.is_active = 1;
      where.end_date = { gte: new Date() };
    }

    const [data, total] = await Promise.all([
      this.prisma.promoCode.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.promoCode.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Get promo code statistics
   */
  async getPromoCodeStats(promoCodeId: number, merchantId: number): Promise<any> {
    const promo = await this.prisma.promoCode.findFirst({
      where: {
        id: promoCodeId,
        merchant_id: merchantId,
      },
    });

    if (!promo) {
      throw new BadRequestException('Code promo non trouve');
    }

    const usages = await this.prisma.promoCodeUsage.aggregate({
      where: { promo_code_id: promoCodeId },
      _count: true,
      _sum: { discount_amount: true },
    });

    const uniqueUsers = await this.prisma.promoCodeUsage.groupBy({
      by: ['user_id'],
      where: { promo_code_id: promoCodeId },
    });

    return {
      promo,
      stats: {
        totalUses: usages._count || 0,
        totalDiscount: usages._sum?.discount_amount || 0,
        uniqueUsers: uniqueUsers.length,
        remainingUses: promo.max_uses ? promo.max_uses - promo.used_count : 'unlimited',
      },
    };
  }

  /**
   * Generate a unique promo code
   */
  async generateUniqueCode(merchantId: number, prefix?: string): Promise<string> {
    const characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789'; // Removed confusing chars (0, O, 1, I)
    const codeLength = 8;

    let code: string;
    let isUnique = false;

    while (!isUnique) {
      code = prefix ? prefix.toUpperCase() : '';

      for (let i = 0; i < codeLength - (prefix?.length || 0); i++) {
        code += characters.charAt(Math.floor(Math.random() * characters.length));
      }

      const existing = await this.prisma.promoCode.findFirst({
        where: {
          code,
          merchant_id: merchantId,
        },
      });

      if (!existing) {
        isUnique = true;
      }
    }

    return code;
  }

  /**
   * Get user's available promo codes
   */
  async getUserPromoCodes(userId: number, merchantId: number): Promise<any[]> {
    const now = new Date();

    // Get all active promos for this merchant
    const promos = await this.prisma.promoCode.findMany({
      where: {
        merchant_id: merchantId,
        is_active: 1,
        start_date: { lte: now },
        end_date: { gte: now },
      },
    });

    // Filter based on user restrictions and usage
    const availablePromos = [];

    for (const promo of promos) {
      // Check user restriction
      if (promo.user_ids) {
        const allowedUsers = JSON.parse(promo.user_ids);
        if (Array.isArray(allowedUsers) && !allowedUsers.includes(userId)) {
          continue;
        }
      }

      // Check global limit
      if (promo.max_uses && promo.used_count >= promo.max_uses) {
        continue;
      }

      // Check per-user limit
      if (promo.max_uses_per_user) {
        const userUsageCount = await this.prisma.promoCodeUsage.count({
          where: {
            promo_code_id: promo.id,
            user_id: userId,
          },
        });

        if (userUsageCount >= promo.max_uses_per_user) {
          continue;
        }
      }

      // Check first ride only
      if (promo.is_first_ride_only) {
        const previousRides = await this.prisma.booking.count({
          where: {
            user_id: userId,
            booking_status: 'completed',
          },
        });

        if (previousRides > 0) {
          continue;
        }
      }

      availablePromos.push({
        code: promo.code,
        description: promo.description,
        discountType: promo.discount_type,
        discountValue: promo.discount_value,
        maxDiscount: promo.max_discount,
        minOrderValue: promo.min_order_value,
        endDate: promo.end_date,
      });
    }

    return availablePromos;
  }
}
