import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { PromoService } from './promo.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';
import {
  ValidatePromoDto,
  CreatePromoCodeDto,
  UpdatePromoCodeDto,
} from './dto';

@ApiTags('Promo')
@Controller('api/promo')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class PromoController {
  constructor(private promoService: PromoService) {}

  // ============================================================================
  // USER ENDPOINTS
  // ============================================================================

  @Post('validate')
  @ApiOperation({ summary: 'Validate a promo code' })
  async validatePromoCode(
    @Body() dto: ValidatePromoDto,
    @MerchantId() merchantId: number,
    @CurrentUser() user: any,
  ) {
    return this.promoService.validatePromoCode(
      dto.code,
      merchantId,
      user.id,
      dto.subtotal,
      dto.vehicleTypeId,
    );
  }

  @Get('available')
  @ApiOperation({ summary: 'Get available promo codes for user' })
  async getAvailablePromoCodes(
    @MerchantId() merchantId: number,
    @CurrentUser() user: any,
  ) {
    return this.promoService.getUserPromoCodes(user.id, merchantId);
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  @Get('admin/list')
  @ApiOperation({ summary: 'List all promo codes (admin)' })
  async listPromoCodes(
    @MerchantId() merchantId: number,
    @Query('active_only') activeOnly?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.promoService.getPromoCodes(merchantId, {
      activeOnly: activeOnly === 'true',
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('admin/create')
  @ApiOperation({ summary: 'Create a new promo code (admin)' })
  async createPromoCode(
    @Body() dto: CreatePromoCodeDto,
    @MerchantId() merchantId: number,
  ) {
    return this.promoService.createPromoCode(merchantId, {
      code: dto.code,
      discountType: dto.discountType,
      discountValue: dto.discountValue,
      maxDiscount: dto.maxDiscount,
      minOrderValue: dto.minOrderValue,
      maxUses: dto.maxUses,
      maxUsesPerUser: dto.maxUsesPerUser,
      startDate: new Date(dto.startDate),
      endDate: new Date(dto.endDate),
      vehicleTypeIds: dto.vehicleTypeIds,
      userIds: dto.userIds,
      isFirstRideOnly: dto.isFirstRideOnly,
      description: dto.description,
    });
  }

  @Put('admin/:id')
  @ApiOperation({ summary: 'Update a promo code (admin)' })
  async updatePromoCode(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdatePromoCodeDto,
    @MerchantId() merchantId: number,
  ) {
    return this.promoService.updatePromoCode(id, merchantId, {
      discountType: dto.discountType,
      discountValue: dto.discountValue,
      maxDiscount: dto.maxDiscount,
      minOrderValue: dto.minOrderValue,
      maxUses: dto.maxUses,
      maxUsesPerUser: dto.maxUsesPerUser,
      startDate: dto.startDate ? new Date(dto.startDate) : undefined,
      endDate: dto.endDate ? new Date(dto.endDate) : undefined,
      vehicleTypeIds: dto.vehicleTypeIds,
      userIds: dto.userIds,
      isFirstRideOnly: dto.isFirstRideOnly,
      description: dto.description,
    });
  }

  @Delete('admin/:id')
  @ApiOperation({ summary: 'Deactivate a promo code (admin)' })
  async deactivatePromoCode(
    @Param('id', ParseIntPipe) id: number,
    @MerchantId() merchantId: number,
  ) {
    await this.promoService.deactivatePromoCode(id, merchantId);
    return { message: 'Code promo desactive' };
  }

  @Get('admin/:id/stats')
  @ApiOperation({ summary: 'Get promo code statistics (admin)' })
  async getPromoCodeStats(
    @Param('id', ParseIntPipe) id: number,
    @MerchantId() merchantId: number,
  ) {
    return this.promoService.getPromoCodeStats(id, merchantId);
  }

  @Post('admin/generate-code')
  @ApiOperation({ summary: 'Generate a unique promo code (admin)' })
  async generateCode(
    @MerchantId() merchantId: number,
    @Body('prefix') prefix?: string,
  ) {
    const code = await this.promoService.generateUniqueCode(merchantId, prefix);
    return { code };
  }
}
