// =============================================================================
// SurgePricingService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { SurgePricingService } from './surge-pricing.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  createMockMerchant,
  createMockSurgeZone,
  createMockDriver,
  createMockBooking,
  resetAllMocks,
} from '../../../test/setup';

describe('SurgePricingService', () => {
  let service: SurgePricingService;
  let prisma: typeof mockPrismaService;

  beforeEach(async () => {
    resetAllMocks();

    // Ajouter les mocks manquants
    mockPrismaService.surgeZone = {
      findFirst: jest.fn(),
      findMany: jest.fn(),
      create: jest.fn(),
      update: jest.fn(),
      updateMany: jest.fn(),
      count: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        SurgePricingService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<SurgePricingService>(SurgePricingService);
    prisma = mockPrismaService;
  });

  afterEach(() => {
    jest.clearAllMocks();
    service.clearCache();
  });

  // ===========================================================================
  // CALCULATE SURGE MULTIPLIER
  // ===========================================================================

  describe('calculateSurgeMultiplier', () => {
    it('devrait retourner 1.0 quand surge désactivé pour le merchant', async () => {
      const mockMerchant = createMockMerchant({
        surge_pricing_enabled: false,
      });
      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      expect(result.multiplier).toBe(1.0);
      expect(result.isActive).toBe(false);
    });

    it('devrait calculer le surge basé sur le ratio demande/offre', async () => {
      const mockMerchant = createMockMerchant({
        surge_pricing_enabled: true,
        surge_max_multiplier: 3.0,
        surge_trigger_threshold: 0.5,
      });

      // Beaucoup de demandes, peu de chauffeurs
      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(50); // High demand
      prisma.driver.count.mockResolvedValue(5);   // Low supply
      prisma.surgeZone.findFirst.mockResolvedValue(null);

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      expect(result.multiplier).toBeGreaterThan(1.0);
      expect(result.isActive).toBe(true);
      expect(result.reason).toContain('demand');
    });

    it('devrait retourner le surge maximum quand aucun chauffeur disponible', async () => {
      const mockMerchant = createMockMerchant({
        surge_pricing_enabled: true,
        surge_max_multiplier: 3.0,
      });

      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(20);
      prisma.driver.count.mockResolvedValue(0); // No drivers
      prisma.surgeZone.findFirst.mockResolvedValue(null);

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      expect(result.multiplier).toBe(3.0);
      expect(result.reason).toContain('Very high demand');
    });

    it('devrait utiliser le cache pour les requêtes répétées', async () => {
      const mockMerchant = createMockMerchant({ surge_pricing_enabled: true });

      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(10);
      prisma.driver.count.mockResolvedValue(20);
      prisma.surgeZone.findFirst.mockResolvedValue(null);

      // Premier appel
      await service.calculateSurgeMultiplier(1, 6.1319, 1.2228, 1);

      // Deuxième appel - devrait utiliser le cache
      await service.calculateSurgeMultiplier(1, 6.1319, 1.2228, 1);

      // Prisma ne devrait être appelé qu'une fois
      expect(prisma.merchant.findUnique).toHaveBeenCalledTimes(1);
    });

    it('devrait appliquer le surge d\'événement', async () => {
      const mockMerchant = createMockMerchant({ surge_pricing_enabled: true });
      const mockSurgeZone = createMockSurgeZone({
        multiplier: 2.0,
        reason: 'Concert event',
      });

      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(5);
      prisma.driver.count.mockResolvedValue(20);
      prisma.surgeZone.findFirst.mockResolvedValue(mockSurgeZone);

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      expect(result.multiplier).toBe(2.0);
      expect(result.reason).toContain('Concert');
    });
  });

  // ===========================================================================
  // TIME-BASED SURGE
  // ===========================================================================

  describe('getTimeBasedSurge', () => {
    it('devrait appliquer le surge heure de pointe matin (7-9h)', () => {
      // Mock la date pour simuler 8h du matin un mardi
      const mockDate = new Date('2024-01-09T08:00:00'); // Mardi 8h
      jest.useFakeTimers().setSystemTime(mockDate);

      const result = (service as any).getTimeBasedSurge();

      expect(result.multiplier).toBeGreaterThan(1.0);
      expect(result.reason).toContain('Morning');

      jest.useRealTimers();
    });

    it('devrait appliquer le surge heure de pointe soir (17-20h)', () => {
      const mockDate = new Date('2024-01-09T18:30:00'); // Mardi 18h30
      jest.useFakeTimers().setSystemTime(mockDate);

      const result = (service as any).getTimeBasedSurge();

      expect(result.multiplier).toBeGreaterThan(1.0);
      expect(result.reason).toContain('Evening');

      jest.useRealTimers();
    });

    it('devrait appliquer le surge weekend nuit', () => {
      const mockDate = new Date('2024-01-13T23:30:00'); // Samedi 23h30
      jest.useFakeTimers().setSystemTime(mockDate);

      const result = (service as any).getTimeBasedSurge();

      expect(result.multiplier).toBeGreaterThan(1.0);
      expect(result.reason).toContain('Weekend');

      jest.useRealTimers();
    });

    it('devrait retourner 1.0 en temps normal', () => {
      const mockDate = new Date('2024-01-09T14:00:00'); // Mardi 14h
      jest.useFakeTimers().setSystemTime(mockDate);

      const result = (service as any).getTimeBasedSurge();

      expect(result.multiplier).toBe(1.0);

      jest.useRealTimers();
    });
  });

  // ===========================================================================
  // APPLY MULTIPLIER
  // ===========================================================================

  describe('applyMultiplier', () => {
    it('devrait appliquer le multiplicateur correctement', () => {
      const baseFare = 5000;
      const multiplier = 1.5;

      const result = service.applyMultiplier(baseFare, multiplier);

      expect(result).toBe(7500);
    });

    it('devrait arrondir le résultat', () => {
      const baseFare = 5000;
      const multiplier = 1.33;

      const result = service.applyMultiplier(baseFare, multiplier);

      expect(result).toBe(6650); // 5000 * 1.33 = 6650
    });

    it('devrait gérer un multiplicateur de 1.0', () => {
      const baseFare = 5000;
      const multiplier = 1.0;

      const result = service.applyMultiplier(baseFare, multiplier);

      expect(result).toBe(5000);
    });
  });

  // ===========================================================================
  // SURGE ZONE MANAGEMENT
  // ===========================================================================

  describe('createSurgeZone', () => {
    it('devrait créer une zone de surge', async () => {
      const surgeData = {
        latitude: 6.1319,
        longitude: 1.2228,
        radiusKm: 3,
        multiplier: 1.8,
        reason: 'Football match',
        startTime: new Date(),
        endTime: new Date(Date.now() + 4 * 60 * 60 * 1000),
      };

      prisma.surgeZone.create.mockResolvedValue({
        id: 1,
        merchant_id: 1,
        ...surgeData,
        is_active: true,
      });

      const result = await service.createSurgeZone(1, surgeData);

      expect(result).toHaveProperty('id');
      expect(prisma.surgeZone.create).toHaveBeenCalledWith(
        expect.objectContaining({
          data: expect.objectContaining({
            merchant_id: 1,
            multiplier: 1.8,
            is_active: true,
          }),
        }),
      );
    });
  });

  describe('deactivateSurgeZone', () => {
    it('devrait désactiver une zone de surge', async () => {
      prisma.surgeZone.update.mockResolvedValue({
        id: 1,
        is_active: false,
      });

      await service.deactivateSurgeZone(1);

      expect(prisma.surgeZone.update).toHaveBeenCalledWith({
        where: { id: 1 },
        data: { is_active: false },
      });
    });
  });

  describe('getActiveSurgeZones', () => {
    it('devrait retourner les zones actives', async () => {
      const mockZones = [
        createMockSurgeZone({ id: 1 }),
        createMockSurgeZone({ id: 2 }),
      ];

      prisma.surgeZone.findMany.mockResolvedValue(mockZones);

      const result = await service.getActiveSurgeZones(1);

      expect(result).toHaveLength(2);
      expect(prisma.surgeZone.findMany).toHaveBeenCalledWith(
        expect.objectContaining({
          where: expect.objectContaining({
            merchant_id: 1,
            is_active: true,
          }),
        }),
      );
    });
  });

  // ===========================================================================
  // CACHE MANAGEMENT
  // ===========================================================================

  describe('clearCache', () => {
    it('devrait vider le cache', async () => {
      const mockMerchant = createMockMerchant({ surge_pricing_enabled: true });
      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(10);
      prisma.driver.count.mockResolvedValue(20);
      prisma.surgeZone.findFirst.mockResolvedValue(null);

      // Remplir le cache
      await service.calculateSurgeMultiplier(1, 6.1319, 1.2228, 1);

      // Vider le cache
      service.clearCache();

      // Recalculer devrait rappeler prisma
      await service.calculateSurgeMultiplier(1, 6.1319, 1.2228, 1);

      expect(prisma.merchant.findUnique).toHaveBeenCalledTimes(2);
    });
  });

  // ===========================================================================
  // EDGE CASES
  // ===========================================================================

  describe('edge cases', () => {
    it('devrait respecter le multiplicateur maximum', async () => {
      const mockMerchant = createMockMerchant({
        surge_pricing_enabled: true,
        surge_max_multiplier: 2.5,
      });

      // Conditions extrêmes
      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(200);
      prisma.driver.count.mockResolvedValue(0);
      prisma.surgeZone.findFirst.mockResolvedValue(
        createMockSurgeZone({ multiplier: 5.0 }),
      );

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      expect(result.multiplier).toBeLessThanOrEqual(2.5);
    });

    it('devrait arrondir le multiplicateur à 1 décimale', async () => {
      const mockMerchant = createMockMerchant({ surge_pricing_enabled: true });

      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);
      prisma.booking.count.mockResolvedValue(15);
      prisma.driver.count.mockResolvedValue(10);
      prisma.surgeZone.findFirst.mockResolvedValue(null);

      const result = await service.calculateSurgeMultiplier(
        1, 6.1319, 1.2228, 1,
      );

      // Le multiplicateur doit avoir au max 1 décimale
      const decimalPlaces = (result.multiplier.toString().split('.')[1] || '').length;
      expect(decimalPlaces).toBeLessThanOrEqual(1);
    });
  });
});
