import {
  Injectable,
  NotFoundException,
  BadRequestException,
  Logger,
} from '@nestjs/common';
import { PrismaService } from '../../shared/database/prisma.service';
import { StripeService } from './stripe.service';
import { AddCardDto } from './dto/add-card.dto';
import { ProcessPaymentDto } from './dto/process-payment.dto';
import { AddWalletFundsDto } from './dto/add-wallet-funds.dto';
import {
  PAYMENT_METHOD,
  PAYMENT_STATUS,
  BOOKING_STATUS,
} from '../booking/booking.constants';

interface PaymentResult {
  success: boolean;
  transaction_id?: string;
  message: string;
  error?: string;
}

interface EarningsBreakdown {
  totalFare: number;
  companyCommission: number;
  driverEarning: number;
  taxAmount: number;
  surgeAmount: number;
  tipAmount: number;
}

@Injectable()
export class PaymentService {
  private readonly logger = new Logger(PaymentService.name);

  constructor(
    private prisma: PrismaService,
    private stripeService: StripeService,
  ) {}

  // =============================================================================
  // PAYMENT METHODS
  // =============================================================================

  async getMethods(merchantId: number) {
    // Get merchant-enabled payment methods
    const merchantMethods = await this.prisma.merchantPaymentMethod.findMany({
      where: {
        merchant_id: merchantId,
        status: 1,
      },
      include: {
        paymentMethod: true,
      },
    });

    // If no merchant-specific methods, return defaults
    if (merchantMethods.length === 0) {
      return {
        message: 'Payment methods',
        data: [
          { id: PAYMENT_METHOD.CASH, name: 'Espèces', code: 'cash', icon: 'cash-icon.png' },
          { id: PAYMENT_METHOD.CARD, name: 'Carte bancaire', code: 'card', icon: 'card-icon.png' },
          { id: PAYMENT_METHOD.WALLET, name: 'Portefeuille', code: 'wallet', icon: 'wallet-icon.png' },
          { id: PAYMENT_METHOD.MOBILE_MONEY, name: 'Mobile Money', code: 'mobile_money', icon: 'momo-icon.png' },
        ],
      };
    }

    return {
      message: 'Payment methods',
      data: merchantMethods.map(m => ({
        id: m.paymentMethod.id,
        name: m.paymentMethod.method_name,
        code: m.paymentMethod.method_type,
        icon: m.paymentMethod.method_icon,
      })),
    };
  }

  // =============================================================================
  // SAVED CARDS
  // =============================================================================

  async getCards(userId: number, userType: string) {
    const cards = await this.prisma.userCard.findMany({
      where: {
        user_id: userId,
        is_deleted: 0,
      },
      orderBy: { is_default: 'desc' },
    });

    return {
      message: 'Saved cards',
      data: cards.map(card => ({
        id: card.id,
        last4: card.last_four,
        brand: card.card_brand,
        exp_month: card.exp_month,
        exp_year: card.exp_year,
        is_default: card.is_default === 1,
      })),
    };
  }

  async addCard(userId: number, userType: string, dto: AddCardDto) {
    try {
      // Get or create Stripe customer
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { stripe_customer_id: true, email: true, first_name: true },
      });

      if (!user) {
        throw new NotFoundException('User not found');
      }

      let stripeCustomerId = user.stripe_customer_id;

      // Create Stripe customer if doesn't exist
      if (!stripeCustomerId) {
        const customer = await this.stripeService.createCustomer({
          email: user.email || '',
          name: user.first_name || '',
          metadata: { user_id: userId.toString() },
        });
        stripeCustomerId = customer.id;

        await this.prisma.user.update({
          where: { id: userId },
          data: { stripe_customer_id: stripeCustomerId },
        });
      }

      // Add payment method to Stripe
      const paymentMethod = await this.stripeService.attachPaymentMethod(
        dto.payment_method_id || dto.token,
        stripeCustomerId,
      );

      // Check if this is the first card (make it default)
      const existingCards = await this.prisma.userCard.count({
        where: { user_id: userId, is_deleted: 0 },
      });

      // Save card in database
      const card = await this.prisma.userCard.create({
        data: {
          user_id: userId,
          stripe_payment_method_id: paymentMethod.id,
          last_four: paymentMethod.card?.last4 || dto.token.slice(-4),
          card_brand: paymentMethod.card?.brand || 'unknown',
          exp_month: paymentMethod.card?.exp_month,
          exp_year: paymentMethod.card?.exp_year,
          is_default: existingCards === 0 ? 1 : 0,
        },
      });

      return {
        message: 'Card added successfully',
        data: {
          id: card.id,
          last4: card.last_four,
          brand: card.card_brand,
          exp_month: card.exp_month,
          exp_year: card.exp_year,
          is_default: card.is_default === 1,
        },
      };
    } catch (error) {
      this.logger.error(`Failed to add card: ${error.message}`);
      throw new BadRequestException('Failed to add card: ' + error.message);
    }
  }

  async setDefaultCard(userId: number, cardId: number) {
    // Remove default from all cards
    await this.prisma.userCard.updateMany({
      where: { user_id: userId },
      data: { is_default: 0 },
    });

    // Set new default
    await this.prisma.userCard.update({
      where: { id: cardId, user_id: userId },
      data: { is_default: 1 },
    });

    return { message: 'Default card updated' };
  }

  async deleteCard(userId: number, userType: string, cardId: number) {
    const card = await this.prisma.userCard.findFirst({
      where: { id: cardId, user_id: userId },
    });

    if (!card) {
      throw new NotFoundException('Card not found');
    }

    // Detach from Stripe
    if (card.stripe_payment_method_id) {
      await this.stripeService.detachPaymentMethod(card.stripe_payment_method_id);
    }

    // Soft delete in database
    await this.prisma.userCard.update({
      where: { id: cardId },
      data: { is_deleted: 1 },
    });

    return { message: 'Card deleted successfully' };
  }

  // =============================================================================
  // PAYMENT PROCESSING
  // =============================================================================

  async processPayment(userId: number, userType: string, dto: ProcessPaymentDto) {
    const booking = await this.prisma.booking.findUnique({
      where: { id: dto.booking_id },
      include: { transaction: true },
    });

    if (!booking) {
      throw new NotFoundException('Booking not found');
    }

    if (booking.payment_status === PAYMENT_STATUS.COMPLETED) {
      throw new BadRequestException('Payment already processed');
    }

    // Verify booking belongs to user
    if (userType === 'user' && booking.user_id !== userId) {
      throw new BadRequestException('Access denied');
    }

    const amount = dto.amount || booking.final_amount || booking.estimate_amount || 0;

    let result: PaymentResult;

    switch (dto.payment_method) {
      case 'cash':
      case PAYMENT_METHOD.CASH:
        result = await this.processCashPayment(booking, amount, dto);
        break;
      case 'card':
      case PAYMENT_METHOD.CARD:
        result = await this.processCardPayment(userId, booking, amount, dto);
        break;
      case 'wallet':
      case PAYMENT_METHOD.WALLET:
        result = await this.processWalletPayment(userId, userType, booking, amount);
        break;
      case 'mobile_money':
      case PAYMENT_METHOD.MOBILE_MONEY:
        result = await this.processMobileMoneyPayment(userId, booking, amount, dto);
        break;
      case 'corporate':
      case PAYMENT_METHOD.CORPORATE:
        result = await this.processCorporatePayment(booking, amount);
        break;
      default:
        throw new BadRequestException('Invalid payment method');
    }

    if (result.success) {
      // Calculate and distribute earnings
      await this.distributeEarnings(booking);

      // Update booking status if ride is completed
      if (booking.booking_status === BOOKING_STATUS.RIDE_COMPLETED) {
        await this.prisma.booking.update({
          where: { id: booking.id },
          data: { booking_status: BOOKING_STATUS.PAYMENT_COMPLETED },
        });
      }
    }

    return {
      message: result.message,
      data: {
        success: result.success,
        transaction_id: result.transaction_id,
      },
    };
  }

  private async processCashPayment(
    booking: any,
    amount: number,
    dto: ProcessPaymentDto,
  ): Promise<PaymentResult> {
    const transactionId = `CASH_${booking.id}_${Date.now()}`;

    await this.prisma.booking.update({
      where: { id: booking.id },
      data: { payment_status: PAYMENT_STATUS.COMPLETED },
    });

    // Update or create transaction record
    if (booking.transaction) {
      await this.prisma.bookingTransaction.update({
        where: { id: booking.transaction.id },
        data: {
          cash_payment: amount.toString(),
          customer_paid_amount: (dto.customer_paid_amount || amount).toString(),
          change_returned: dto.change_returned,
          change_returned_at: dto.change_returned ? new Date() : null,
          transaction_id: transactionId,
        },
      });
    }

    return {
      success: true,
      transaction_id: transactionId,
      message: 'Cash payment recorded',
    };
  }

  private async processCardPayment(
    userId: number,
    booking: any,
    amount: number,
    dto: ProcessPaymentDto,
  ): Promise<PaymentResult> {
    try {
      // Get user's card
      const card = await this.prisma.userCard.findFirst({
        where: {
          id: dto.card_id,
          user_id: userId,
          is_deleted: 0,
        },
      });

      if (!card) {
        throw new BadRequestException('Card not found');
      }

      // Charge via Stripe
      const charge = await this.stripeService.createPaymentIntent({
        amount: Math.round(amount * 100), // Convert to cents
        currency: 'xof',
        payment_method: card.stripe_payment_method_id || '',
        confirm: true,
        metadata: {
          booking_id: booking.id.toString(),
          user_id: userId.toString(),
        },
      });

      // Update booking
      await this.prisma.booking.update({
        where: { id: booking.id },
        data: { payment_status: PAYMENT_STATUS.COMPLETED },
      });

      // Update transaction
      if (booking.transaction) {
        await this.prisma.bookingTransaction.update({
          where: { id: booking.transaction.id },
          data: {
            online_payment: amount.toString(),
            transaction_id: charge.id,
          },
        });
      }

      return {
        success: true,
        transaction_id: charge.id,
        message: 'Card payment successful',
      };
    } catch (error) {
      this.logger.error(`Card payment failed: ${error.message}`);

      await this.prisma.booking.update({
        where: { id: booking.id },
        data: { payment_status: PAYMENT_STATUS.FAILED },
      });

      return {
        success: false,
        message: 'Card payment failed',
        error: error.message,
      };
    }
  }

  private async processWalletPayment(
    userId: number,
    userType: string,
    booking: any,
    amount: number,
  ): Promise<PaymentResult> {
    // Get wallet balance
    const user = await this.prisma.user.findUnique({
      where: { id: userId },
      select: { wallet_balance: true },
    });

    const balance = parseFloat(user?.wallet_balance || '0');

    if (balance < amount) {
      throw new BadRequestException(
        `Insufficient wallet balance. Required: ${amount}, Available: ${balance}`,
      );
    }

    const transactionId = `WALLET_${booking.id}_${Date.now()}`;

    // Deduct from wallet
    await this.prisma.userWalletTransaction.create({
      data: {
        user_id: userId,
        merchant_id: booking.merchant_id,
        type: 'debit',
        amount: -amount,
        description: `Payment for ride #${booking.merchant_booking_id}`,
        reference: transactionId,
        booking_id: booking.id,
        balance_after: (balance - amount).toString(),
      },
    });

    // Update user wallet balance
    await this.prisma.user.update({
      where: { id: userId },
      data: {
        wallet_balance: (balance - amount).toString(),
      },
    });

    // Update booking
    await this.prisma.booking.update({
      where: { id: booking.id },
      data: { payment_status: PAYMENT_STATUS.COMPLETED },
    });

    // Update transaction
    if (booking.transaction) {
      await this.prisma.bookingTransaction.update({
        where: { id: booking.transaction.id },
        data: {
          online_payment: amount.toString(),
          transaction_id: transactionId,
        },
      });
    }

    return {
      success: true,
      transaction_id: transactionId,
      message: 'Wallet payment successful',
    };
  }

  private async processMobileMoneyPayment(
    userId: number,
    booking: any,
    amount: number,
    dto: ProcessPaymentDto,
  ): Promise<PaymentResult> {
    // TODO: Integrate with Mobile Money providers (M-Pesa, MTN, Orange Money, etc.)
    // This would involve:
    // 1. Initiating a payment request to the provider
    // 2. Waiting for user confirmation on their phone
    // 3. Receiving webhook callback with payment status

    const transactionId = `MOMO_${booking.id}_${Date.now()}`;

    // For now, simulate successful payment
    await this.prisma.booking.update({
      where: { id: booking.id },
      data: { payment_status: PAYMENT_STATUS.COMPLETED },
    });

    if (booking.transaction) {
      await this.prisma.bookingTransaction.update({
        where: { id: booking.transaction.id },
        data: {
          online_payment: amount.toString(),
          transaction_id: transactionId,
        },
      });
    }

    return {
      success: true,
      transaction_id: transactionId,
      message: 'Mobile money payment successful',
    };
  }

  private async processCorporatePayment(
    booking: any,
    amount: number,
  ): Promise<PaymentResult> {
    if (!booking.corporate_id) {
      throw new BadRequestException('No corporate account linked');
    }

    // Get corporate account
    const corporate = await this.prisma.corporate.findUnique({
      where: { id: booking.corporate_id },
    });

    if (!corporate) {
      throw new BadRequestException('Corporate account not found');
    }

    const balance = parseFloat(corporate.wallet_balance || '0');

    if (balance < amount) {
      throw new BadRequestException('Insufficient corporate balance');
    }

    const transactionId = `CORP_${booking.id}_${Date.now()}`;

    // Deduct from corporate wallet
    await this.prisma.corporateWalletTransaction.create({
      data: {
        corporate_id: booking.corporate_id,
        type: 'debit',
        amount: (-amount).toString(),
        narration: `Payment for ride #${booking.merchant_booking_id}`,
        balance_after: (balance - amount).toString(),
      },
    });

    // Update corporate balance
    await this.prisma.corporate.update({
      where: { id: booking.corporate_id },
      data: { wallet_balance: (balance - amount).toString() },
    });

    // Update booking
    await this.prisma.booking.update({
      where: { id: booking.id },
      data: { payment_status: PAYMENT_STATUS.COMPLETED },
    });

    return {
      success: true,
      transaction_id: transactionId,
      message: 'Corporate payment successful',
    };
  }

  // =============================================================================
  // EARNINGS DISTRIBUTION
  // =============================================================================

  private async distributeEarnings(booking: any): Promise<void> {
    if (!booking.driver_id) {
      return;
    }

    const amount = booking.final_amount || booking.estimate_amount || 0;

    // Get commission configuration
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: booking.merchant_id },
      select: { tax: true },
    });

    // Get price card for commission percentage
    const priceCard = await this.prisma.priceCardDetail.findFirst({
      where: {
        merchant_id: booking.merchant_id,
        vehicle_type_id: booking.vehicle_type_id,
        status: 1,
      },
    });

    const commissionPercent = parseFloat(priceCard?.commission_percentage || '20');
    const taxPercent = parseFloat(merchant?.tax || '0');

    // Calculate breakdown
    const taxAmount = amount * (taxPercent / 100);
    const amountBeforeTax = amount - taxAmount;
    const companyCommission = amountBeforeTax * (commissionPercent / 100);
    const driverEarning = amountBeforeTax - companyCommission;
    const tipAmount = parseFloat(booking.transaction?.tip || '0');
    const totalDriverEarning = driverEarning + tipAmount;

    // Update transaction with earnings
    if (booking.transaction) {
      await this.prisma.bookingTransaction.update({
        where: { id: booking.transaction.id },
        data: {
          company_earning: companyCommission.toString(),
          driver_earning: driverEarning.toString(),
          driver_total_payout_amount: totalDriverEarning.toString(),
          tax_amount: taxAmount.toString(),
        },
      });
    }

    // Credit driver wallet
    const driver = await this.prisma.driver.findUnique({
      where: { id: booking.driver_id },
      select: { wallet_balance: true },
    });

    const currentBalance = parseFloat(driver?.wallet_balance || '0');
    const newBalance = currentBalance + totalDriverEarning;

    await this.prisma.driver.update({
      where: { id: booking.driver_id },
      data: { wallet_balance: newBalance.toString() },
    });

    // Create wallet transaction
    await this.prisma.driverWalletTransaction.create({
      data: {
        driver_id: booking.driver_id,
        merchant_id: booking.merchant_id,
        booking_id: booking.id,
        type: 'credit',
        amount: totalDriverEarning,
        description: `Earnings from ride #${booking.merchant_booking_id}`,
        balance_after: newBalance.toString(),
      },
    });

    // Create driver earning record
    await this.prisma.driverEarning.create({
      data: {
        driver_id: booking.driver_id,
        merchant_id: booking.merchant_id,
        booking_id: booking.id,
        total_fare: amount,
        commission_amount: companyCommission,
        net_earning: totalDriverEarning,
        tip_amount: tipAmount,
        date: new Date(),
      },
    });

    this.logger.log(
      `Earnings distributed for booking ${booking.id}: Driver=${totalDriverEarning}, Company=${companyCommission}`,
    );
  }

  // =============================================================================
  // WALLET OPERATIONS
  // =============================================================================

  async getWalletBalance(userId: number, userType: string) {
    if (userType === 'user') {
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });
      return {
        message: 'Wallet balance',
        data: { balance: parseFloat(user?.wallet_balance || '0') },
      };
    } else {
      const driver = await this.prisma.driver.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });
      return {
        message: 'Wallet balance',
        data: { balance: parseFloat(driver?.wallet_balance || '0') },
      };
    }
  }

  async addWalletFunds(userId: number, userType: string, dto: AddWalletFundsDto) {
    // TODO: Process payment via card/mobile money first
    const transactionId = `TOPUP_${Date.now()}`;

    if (userType === 'user') {
      const user = await this.prisma.user.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });

      const currentBalance = parseFloat(user?.wallet_balance || '0');
      const newBalance = currentBalance + dto.amount;

      await this.prisma.userWalletTransaction.create({
        data: {
          user_id: userId,
          type: 'credit',
          amount: dto.amount,
          description: 'Wallet top-up',
          reference: transactionId,
          balance_after: newBalance.toString(),
        },
      });

      await this.prisma.user.update({
        where: { id: userId },
        data: { wallet_balance: newBalance.toString() },
      });
    } else {
      const driver = await this.prisma.driver.findUnique({
        where: { id: userId },
        select: { wallet_balance: true },
      });

      const currentBalance = parseFloat(driver?.wallet_balance || '0');
      const newBalance = currentBalance + dto.amount;

      await this.prisma.driverWalletTransaction.create({
        data: {
          driver_id: userId,
          type: 'credit',
          amount: dto.amount,
          description: 'Wallet top-up',
          balance_after: newBalance.toString(),
        },
      });

      await this.prisma.driver.update({
        where: { id: userId },
        data: { wallet_balance: newBalance.toString() },
      });
    }

    return {
      message: 'Funds added to wallet',
      data: { amount_added: dto.amount, transaction_id: transactionId },
    };
  }

  async getWalletTransactions(
    userId: number,
    userType: string,
    page = 1,
    limit = 20,
  ) {
    const skip = (page - 1) * limit;

    if (userType === 'user') {
      const [transactions, total] = await Promise.all([
        this.prisma.userWalletTransaction.findMany({
          where: { user_id: userId },
          orderBy: { created_at: 'desc' },
          skip,
          take: limit,
        }),
        this.prisma.userWalletTransaction.count({ where: { user_id: userId } }),
      ]);

      return {
        message: 'Wallet transactions',
        data: {
          transactions,
          pagination: {
            page,
            limit,
            total,
            total_pages: Math.ceil(total / limit),
          },
        },
      };
    } else {
      const [transactions, total] = await Promise.all([
        this.prisma.driverWalletTransaction.findMany({
          where: { driver_id: userId },
          orderBy: { created_at: 'desc' },
          skip,
          take: limit,
        }),
        this.prisma.driverWalletTransaction.count({ where: { driver_id: userId } }),
      ]);

      return {
        message: 'Wallet transactions',
        data: {
          transactions,
          pagination: {
            page,
            limit,
            total,
            total_pages: Math.ceil(total / limit),
          },
        },
      };
    }
  }

  // =============================================================================
  // WEBHOOKS
  // =============================================================================

  async handleStripeWebhook(body: any, signature: string) {
    try {
      const event = await this.stripeService.constructWebhookEvent(body, signature);

      this.logger.log(`Stripe webhook received: ${event.type}`);

      switch (event.type) {
        case 'payment_intent.succeeded':
          await this.handlePaymentSucceeded(event.data.object);
          break;
        case 'payment_intent.payment_failed':
          await this.handlePaymentFailed(event.data.object);
          break;
        default:
          this.logger.log(`Unhandled event type: ${event.type}`);
      }

      return { received: true };
    } catch (error) {
      this.logger.error(`Webhook error: ${error.message}`);
      throw new BadRequestException('Webhook verification failed');
    }
  }

  private async handlePaymentSucceeded(paymentIntent: any) {
    const bookingId = paymentIntent.metadata?.booking_id;
    if (bookingId) {
      await this.prisma.booking.update({
        where: { id: parseInt(bookingId) },
        data: { payment_status: PAYMENT_STATUS.COMPLETED },
      });
    }
  }

  private async handlePaymentFailed(paymentIntent: any) {
    const bookingId = paymentIntent.metadata?.booking_id;
    if (bookingId) {
      await this.prisma.booking.update({
        where: { id: parseInt(bookingId) },
        data: { payment_status: PAYMENT_STATUS.FAILED },
      });
    }
  }
}
