// =============================================================================
// PaymentService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { NotFoundException, BadRequestException } from '@nestjs/common';
import { PaymentService } from './payment.service';
import { StripeService } from './stripe.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  mockStripe,
  createMockUser,
  createMockDriver,
  createMockBooking,
  createMockMerchant,
  createMockPaymentMethod,
  createMockTransaction,
  resetAllMocks,
} from '../../../test/setup';
import { PAYMENT_STATUS } from '../booking/booking.constants';

// Mock Stripe Service
const mockStripeService = {
  createCustomer: jest.fn(() => Promise.resolve({ id: 'cus_123' })),
  attachPaymentMethod: jest.fn(() => Promise.resolve({
    id: 'pm_123',
    card: { last4: '4242', brand: 'visa', exp_month: 12, exp_year: 2025 },
  })),
  detachPaymentMethod: jest.fn(() => Promise.resolve(true)),
  createPaymentIntent: jest.fn(() => Promise.resolve({ id: 'pi_123', status: 'succeeded' })),
  constructWebhookEvent: jest.fn(),
};

describe('PaymentService', () => {
  let service: PaymentService;
  let prisma: typeof mockPrismaService;
  let stripe: typeof mockStripeService;

  beforeEach(async () => {
    resetAllMocks();

    // Add missing prisma mocks
    mockPrismaService.userCard = {
      findMany: jest.fn(),
      findFirst: jest.fn(),
      create: jest.fn(),
      update: jest.fn(),
      updateMany: jest.fn(),
      count: jest.fn(),
    };
    mockPrismaService.merchantPaymentMethod = {
      findMany: jest.fn(),
    };
    mockPrismaService.userWalletTransaction = {
      create: jest.fn(),
      findMany: jest.fn(),
      count: jest.fn(),
    };
    mockPrismaService.driverWalletTransaction = {
      create: jest.fn(),
      findMany: jest.fn(),
      count: jest.fn(),
    };
    mockPrismaService.driverEarning = {
      create: jest.fn(),
    };
    mockPrismaService.bookingTransaction = {
      findFirst: jest.fn(),
      update: jest.fn(),
    };
    mockPrismaService.priceCardDetail = {
      findFirst: jest.fn(),
    };
    mockPrismaService.corporate = {
      findUnique: jest.fn(),
      update: jest.fn(),
    };
    mockPrismaService.corporateWalletTransaction = {
      create: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        PaymentService,
        { provide: PrismaService, useValue: mockPrismaService },
        { provide: StripeService, useValue: mockStripeService },
      ],
    }).compile();

    service = module.get<PaymentService>(PaymentService);
    prisma = mockPrismaService;
    stripe = mockStripeService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // GET PAYMENT METHODS
  // ===========================================================================

  describe('getMethods', () => {
    it('devrait retourner les méthodes de paiement par défaut', async () => {
      prisma.merchantPaymentMethod.findMany.mockResolvedValue([]);

      const result = await service.getMethods(1);

      expect(result.data).toHaveLength(4);
      expect(result.data.map(m => m.code)).toContain('cash');
      expect(result.data.map(m => m.code)).toContain('card');
      expect(result.data.map(m => m.code)).toContain('wallet');
      expect(result.data.map(m => m.code)).toContain('mobile_money');
    });

    it('devrait retourner les méthodes configurées par le merchant', async () => {
      prisma.merchantPaymentMethod.findMany.mockResolvedValue([
        {
          paymentMethod: { id: 1, method_name: 'Cash', method_type: 'cash', method_icon: 'cash.png' },
        },
        {
          paymentMethod: { id: 2, method_name: 'Card', method_type: 'card', method_icon: 'card.png' },
        },
      ]);

      const result = await service.getMethods(1);

      expect(result.data).toHaveLength(2);
    });
  });

  // ===========================================================================
  // SAVED CARDS
  // ===========================================================================

  describe('getCards', () => {
    it('devrait retourner les cartes enregistrées de l\'utilisateur', async () => {
      prisma.userCard.findMany.mockResolvedValue([
        {
          id: 1,
          last_four: '4242',
          card_brand: 'visa',
          exp_month: 12,
          exp_year: 2025,
          is_default: 1,
        },
        {
          id: 2,
          last_four: '5555',
          card_brand: 'mastercard',
          exp_month: 6,
          exp_year: 2026,
          is_default: 0,
        },
      ]);

      const result = await service.getCards(1, 'user');

      expect(result.data).toHaveLength(2);
      expect(result.data[0].is_default).toBe(true);
    });

    it('devrait retourner une liste vide si pas de cartes', async () => {
      prisma.userCard.findMany.mockResolvedValue([]);

      const result = await service.getCards(1, 'user');

      expect(result.data).toHaveLength(0);
    });
  });

  describe('addCard', () => {
    it('devrait ajouter une nouvelle carte avec succès', async () => {
      const mockUser = createMockUser({ stripe_customer_id: 'cus_123' });
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userCard.count.mockResolvedValue(0);
      prisma.userCard.create.mockResolvedValue({
        id: 1,
        last_four: '4242',
        card_brand: 'visa',
        is_default: 1,
      });

      const result = await service.addCard(1, 'user', {
        payment_method_id: 'pm_123',
        token: 'tok_visa',
      });

      expect(result.message).toBe('Card added successfully');
      expect(result.data.last4).toBe('4242');
    });

    it('devrait créer un customer Stripe si inexistant', async () => {
      const mockUser = createMockUser({ stripe_customer_id: null });
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.user.update.mockResolvedValue(mockUser);
      prisma.userCard.count.mockResolvedValue(0);
      prisma.userCard.create.mockResolvedValue({ id: 1 });

      await service.addCard(1, 'user', { token: 'tok_visa' });

      expect(stripe.createCustomer).toHaveBeenCalled();
    });

    it('devrait lever une erreur si utilisateur non trouvé', async () => {
      prisma.user.findUnique.mockResolvedValue(null);

      await expect(
        service.addCard(1, 'user', { token: 'tok_visa' }),
      ).rejects.toThrow(NotFoundException);
    });
  });

  describe('setDefaultCard', () => {
    it('devrait définir une carte par défaut', async () => {
      prisma.userCard.updateMany.mockResolvedValue({ count: 2 });
      prisma.userCard.update.mockResolvedValue({ id: 1, is_default: 1 });

      const result = await service.setDefaultCard(1, 1);

      expect(result.message).toBe('Default card updated');
      expect(prisma.userCard.updateMany).toHaveBeenCalled();
    });
  });

  describe('deleteCard', () => {
    it('devrait supprimer une carte (soft delete)', async () => {
      prisma.userCard.findFirst.mockResolvedValue({
        id: 1,
        stripe_payment_method_id: 'pm_123',
      });
      prisma.userCard.update.mockResolvedValue({ id: 1, is_deleted: 1 });

      const result = await service.deleteCard(1, 'user', 1);

      expect(result.message).toBe('Card deleted successfully');
      expect(stripe.detachPaymentMethod).toHaveBeenCalledWith('pm_123');
    });

    it('devrait lever une erreur si carte non trouvée', async () => {
      prisma.userCard.findFirst.mockResolvedValue(null);

      await expect(service.deleteCard(1, 'user', 999)).rejects.toThrow(
        NotFoundException,
      );
    });
  });

  // ===========================================================================
  // PROCESS PAYMENT
  // ===========================================================================

  describe('processPayment', () => {
    const mockBooking = createMockBooking({
      id: 1,
      user_id: 1,
      final_amount: 5000,
      payment_status: PAYMENT_STATUS.PENDING,
      transaction: { id: 1 },
    });

    beforeEach(() => {
      prisma.booking.findUnique.mockResolvedValue(mockBooking);
      prisma.booking.update.mockResolvedValue(mockBooking);
      prisma.bookingTransaction.update.mockResolvedValue({ id: 1 });
      prisma.merchant.findUnique.mockResolvedValue(createMockMerchant());
      prisma.priceCardDetail.findFirst.mockResolvedValue({ commission_percentage: '20' });
      prisma.driver.findUnique.mockResolvedValue(createMockDriver());
      prisma.driver.update.mockResolvedValue(createMockDriver());
      prisma.driverWalletTransaction.create.mockResolvedValue({});
      prisma.driverEarning.create.mockResolvedValue({});
    });

    it('devrait traiter un paiement en espèces', async () => {
      const result = await service.processPayment(1, 'user', {
        booking_id: 1,
        payment_method: 'cash',
        customer_paid_amount: 5000,
      });

      expect(result.data.success).toBe(true);
      expect(result.data.transaction_id).toContain('CASH_');
    });

    it('devrait traiter un paiement par carte', async () => {
      prisma.userCard.findFirst.mockResolvedValue({
        id: 1,
        stripe_payment_method_id: 'pm_123',
      });

      const result = await service.processPayment(1, 'user', {
        booking_id: 1,
        payment_method: 'card',
        card_id: 1,
      });

      expect(result.data.success).toBe(true);
      expect(stripe.createPaymentIntent).toHaveBeenCalled();
    });

    it('devrait traiter un paiement par portefeuille', async () => {
      prisma.user.findUnique.mockResolvedValue(
        createMockUser({ wallet_balance: '10000' }),
      );
      prisma.user.update.mockResolvedValue(createMockUser());
      prisma.userWalletTransaction.create.mockResolvedValue({});

      const result = await service.processPayment(1, 'user', {
        booking_id: 1,
        payment_method: 'wallet',
      });

      expect(result.data.success).toBe(true);
      expect(result.data.transaction_id).toContain('WALLET_');
    });

    it('devrait lever une erreur si solde insuffisant', async () => {
      prisma.user.findUnique.mockResolvedValue(
        createMockUser({ wallet_balance: '1000' }),
      );

      await expect(
        service.processPayment(1, 'user', {
          booking_id: 1,
          payment_method: 'wallet',
        }),
      ).rejects.toThrow(BadRequestException);
    });

    it('devrait traiter un paiement mobile money', async () => {
      const result = await service.processPayment(1, 'user', {
        booking_id: 1,
        payment_method: 'mobile_money',
      });

      expect(result.data.success).toBe(true);
      expect(result.data.transaction_id).toContain('MOMO_');
    });

    it('devrait lever une erreur si réservation non trouvée', async () => {
      prisma.booking.findUnique.mockResolvedValue(null);

      await expect(
        service.processPayment(1, 'user', {
          booking_id: 999,
          payment_method: 'cash',
        }),
      ).rejects.toThrow(NotFoundException);
    });

    it('devrait lever une erreur si paiement déjà effectué', async () => {
      prisma.booking.findUnique.mockResolvedValue({
        ...mockBooking,
        payment_status: PAYMENT_STATUS.COMPLETED,
      });

      await expect(
        service.processPayment(1, 'user', {
          booking_id: 1,
          payment_method: 'cash',
        }),
      ).rejects.toThrow(BadRequestException);
    });

    it('devrait distribuer les gains au chauffeur après paiement', async () => {
      await service.processPayment(1, 'user', {
        booking_id: 1,
        payment_method: 'cash',
      });

      expect(prisma.driverWalletTransaction.create).toHaveBeenCalled();
      expect(prisma.driverEarning.create).toHaveBeenCalled();
    });
  });

  // ===========================================================================
  // WALLET OPERATIONS
  // ===========================================================================

  describe('getWalletBalance', () => {
    it('devrait retourner le solde du portefeuille utilisateur', async () => {
      prisma.user.findUnique.mockResolvedValue(
        createMockUser({ wallet_balance: '15000' }),
      );

      const result = await service.getWalletBalance(1, 'user');

      expect(result.data.balance).toBe(15000);
    });

    it('devrait retourner le solde du portefeuille chauffeur', async () => {
      prisma.driver.findUnique.mockResolvedValue(
        createMockDriver({ wallet_balance: '50000' }),
      );

      const result = await service.getWalletBalance(1, 'driver');

      expect(result.data.balance).toBe(50000);
    });

    it('devrait retourner 0 si pas de solde', async () => {
      prisma.user.findUnique.mockResolvedValue(
        createMockUser({ wallet_balance: null }),
      );

      const result = await service.getWalletBalance(1, 'user');

      expect(result.data.balance).toBe(0);
    });
  });

  describe('addWalletFunds', () => {
    it('devrait ajouter des fonds au portefeuille utilisateur', async () => {
      prisma.user.findUnique.mockResolvedValue(
        createMockUser({ wallet_balance: '5000' }),
      );
      prisma.user.update.mockResolvedValue(createMockUser());
      prisma.userWalletTransaction.create.mockResolvedValue({});

      const result = await service.addWalletFunds(1, 'user', { amount: 10000 });

      expect(result.message).toBe('Funds added to wallet');
      expect(result.data.amount_added).toBe(10000);
    });

    it('devrait ajouter des fonds au portefeuille chauffeur', async () => {
      prisma.driver.findUnique.mockResolvedValue(
        createMockDriver({ wallet_balance: '20000' }),
      );
      prisma.driver.update.mockResolvedValue(createMockDriver());
      prisma.driverWalletTransaction.create.mockResolvedValue({});

      const result = await service.addWalletFunds(1, 'driver', { amount: 5000 });

      expect(result.data.amount_added).toBe(5000);
    });
  });

  describe('getWalletTransactions', () => {
    it('devrait retourner les transactions utilisateur avec pagination', async () => {
      prisma.userWalletTransaction.findMany.mockResolvedValue([
        { id: 1, amount: 10000, type: 'credit' },
        { id: 2, amount: -5000, type: 'debit' },
      ]);
      prisma.userWalletTransaction.count.mockResolvedValue(2);

      const result = await service.getWalletTransactions(1, 'user', 1, 10);

      expect(result.data.transactions).toHaveLength(2);
      expect(result.data.pagination.total).toBe(2);
    });

    it('devrait retourner les transactions chauffeur', async () => {
      prisma.driverWalletTransaction.findMany.mockResolvedValue([
        { id: 1, amount: 5000, type: 'credit' },
      ]);
      prisma.driverWalletTransaction.count.mockResolvedValue(1);

      const result = await service.getWalletTransactions(1, 'driver', 1, 10);

      expect(result.data.transactions).toHaveLength(1);
    });
  });

  // ===========================================================================
  // WEBHOOKS
  // ===========================================================================

  describe('handleStripeWebhook', () => {
    it('devrait traiter payment_intent.succeeded', async () => {
      stripe.constructWebhookEvent.mockResolvedValue({
        type: 'payment_intent.succeeded',
        data: { object: { metadata: { booking_id: '1' } } },
      });
      prisma.booking.update.mockResolvedValue(createMockBooking());

      const result = await service.handleStripeWebhook({}, 'sig_123');

      expect(result.received).toBe(true);
      expect(prisma.booking.update).toHaveBeenCalledWith(
        expect.objectContaining({
          data: { payment_status: PAYMENT_STATUS.COMPLETED },
        }),
      );
    });

    it('devrait traiter payment_intent.payment_failed', async () => {
      stripe.constructWebhookEvent.mockResolvedValue({
        type: 'payment_intent.payment_failed',
        data: { object: { metadata: { booking_id: '1' } } },
      });
      prisma.booking.update.mockResolvedValue(createMockBooking());

      await service.handleStripeWebhook({}, 'sig_123');

      expect(prisma.booking.update).toHaveBeenCalledWith(
        expect.objectContaining({
          data: { payment_status: PAYMENT_STATUS.FAILED },
        }),
      );
    });

    it('devrait lever une erreur si signature invalide', async () => {
      stripe.constructWebhookEvent.mockRejectedValue(new Error('Invalid signature'));

      await expect(
        service.handleStripeWebhook({}, 'invalid_sig'),
      ).rejects.toThrow(BadRequestException);
    });
  });
});
