import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class PayDunyaGateway implements PaymentGateway {
  readonly name = 'paydunya';
  readonly displayName = 'PayDunya';
  readonly supportedCurrencies = ['XOF'];
  readonly supportedCountries = ['SN', 'CI', 'BJ', 'TG', 'BF', 'ML'];

  private readonly logger = new Logger(PayDunyaGateway.name);
  private readonly baseUrl: string;
  private masterKey: string;
  private privateKey: string;
  private publicKey: string;
  private token: string;

  constructor(private configService: ConfigService) {
    const isLive = this.configService.get<string>('PAYDUNYA_MODE') === 'live';
    this.baseUrl = isLive
      ? 'https://app.paydunya.com/api/v1'
      : 'https://app.paydunya.com/sandbox-api/v1';

    this.masterKey = this.configService.get<string>('PAYDUNYA_MASTER_KEY');
    this.privateKey = this.configService.get<string>('PAYDUNYA_PRIVATE_KEY');
    this.publicKey = this.configService.get<string>('PAYDUNYA_PUBLIC_KEY');
    this.token = this.configService.get<string>('PAYDUNYA_TOKEN');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  private getHeaders() {
    return {
      'Content-Type': 'application/json',
      'PAYDUNYA-MASTER-KEY': this.masterKey,
      'PAYDUNYA-PRIVATE-KEY': this.privateKey,
      'PAYDUNYA-TOKEN': this.token,
    };
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/checkout-invoice/create`,
        {
          invoice: {
            total_amount: Math.round(params.amount),
            description: `Paiement - Ref: ${params.reference}`,
          },
          store: {
            name: this.configService.get('APP_NAME', 'MonkAPI'),
            tagline: 'Paiement securise',
            postal_address: '',
            phone: '',
            logo_url: '',
            website_url: this.configService.get('APP_URL'),
          },
          custom_data: {
            reference: params.reference,
            ...params.metadata,
          },
          actions: {
            cancel_url: `${params.callbackUrl}?status=cancelled&reference=${params.reference}`,
            return_url: `${params.callbackUrl}?status=success&reference=${params.reference}`,
            callback_url: `${this.configService.get('APP_URL')}/api/payment/webhook/paydunya`,
          },
        },
        { headers: this.getHeaders() },
      );

      if (response.data.response_code === '00') {
        return {
          success: true,
          paymentUrl: response.data.response_text,
          reference: params.reference,
          transactionId: response.data.token,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.response_text,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`PayDunya initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.response_text || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      // PayDunya uses token for verification
      const response = await axios.get(
        `${this.baseUrl}/checkout-invoice/confirm/${reference}`,
        { headers: this.getHeaders() },
      );

      if (response.data.response_code === '00' && response.data.status === 'completed') {
        return {
          success: true,
          status: 'success',
          amount: response.data.invoice?.total_amount,
          currency: 'XOF',
          reference: response.data.custom_data?.reference,
          transactionId: response.data.token,
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        completed: 'success',
        pending: 'pending',
        cancelled: 'cancelled',
        failed: 'failed',
      };

      return {
        success: false,
        status: statusMap[response.data.status] || 'pending',
        message: response.data.response_text,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`PayDunya verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.response_text || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    // PayDunya doesn't have automated refunds
    return {
      success: false,
      message: 'Les remboursements PayDunya doivent etre effectues manuellement',
    };
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      const data = payload.data || payload;

      if (data.status === 'completed') {
        return {
          success: true,
          event: 'payment_completed',
          reference: data.custom_data?.reference,
          status: 'success',
          amount: data.invoice?.total_amount,
          data,
        };
      }

      return {
        success: true,
        event: `payment_${data.status}`,
        reference: data.custom_data?.reference,
        status: data.status === 'cancelled' ? 'cancelled' : 'failed',
        data,
      };
    } catch (error) {
      this.logger.error(`PayDunya webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }

  // ============================================================================
  // DIRECT MOBILE MONEY
  // ============================================================================

  async initializeDirectPay(
    params: InitializePaymentParams & {
      phone: string;
      provider: 'orange_money_senegal' | 'free_money_senegal' | 'wizall_senegal' | 'wave_senegal'
    },
  ): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/direct-pay/credit-account`,
        {
          account_alias: params.phone,
          amount: Math.round(params.amount),
          withdraw_mode: params.provider,
        },
        { headers: this.getHeaders() },
      );

      if (response.data.response_code === '00') {
        return {
          success: true,
          reference: params.reference,
          transactionId: response.data.transaction_id,
          message: response.data.response_text,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.response_text,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`PayDunya directPay error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.response_text || error.message,
      };
    }
  }
}
