import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class OrangeMoneyGateway implements PaymentGateway {
  readonly name = 'orange_money';
  readonly displayName = 'Orange Money';
  readonly supportedCurrencies = ['XOF', 'XAF', 'GNF', 'EUR'];
  readonly supportedCountries = ['SN', 'CI', 'ML', 'BF', 'NE', 'GN', 'CM', 'MG', 'CD'];

  private readonly logger = new Logger(OrangeMoneyGateway.name);
  private readonly baseUrl: string;
  private clientId: string;
  private clientSecret: string;
  private merchantKey: string;
  private accessToken: string;
  private tokenExpiry: number = 0;

  constructor(private configService: ConfigService) {
    const isLive = this.configService.get<string>('ORANGE_MONEY_MODE') === 'live';
    this.baseUrl = isLive
      ? 'https://api.orange.com'
      : 'https://api.orange.com/orange-money-webpay';

    this.clientId = this.configService.get<string>('ORANGE_MONEY_CLIENT_ID');
    this.clientSecret = this.configService.get<string>('ORANGE_MONEY_CLIENT_SECRET');
    this.merchantKey = this.configService.get<string>('ORANGE_MONEY_MERCHANT_KEY');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  private async getAccessToken(): Promise<string> {
    if (this.accessToken && Date.now() < this.tokenExpiry) {
      return this.accessToken;
    }

    const credentials = Buffer.from(`${this.clientId}:${this.clientSecret}`).toString('base64');

    const response = await axios.post(
      `${this.baseUrl}/oauth/v3/token`,
      'grant_type=client_credentials',
      {
        headers: {
          Authorization: `Basic ${credentials}`,
          'Content-Type': 'application/x-www-form-urlencoded',
        },
      },
    );

    this.accessToken = response.data.access_token;
    this.tokenExpiry = Date.now() + (response.data.expires_in - 60) * 1000;

    return this.accessToken;
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const token = await this.getAccessToken();

      const response = await axios.post(
        `${this.baseUrl}/orange-money-webpay/dev/v1/webpayment`,
        {
          merchant_key: this.merchantKey,
          currency: params.currency,
          order_id: params.reference,
          amount: Math.round(params.amount),
          return_url: params.callbackUrl,
          cancel_url: `${params.callbackUrl}?status=cancelled`,
          notif_url: `${this.configService.get('APP_URL')}/api/payment/webhook/orange-money`,
          lang: 'fr',
          reference: params.reference,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 201 || response.data.payment_url) {
        return {
          success: true,
          paymentUrl: response.data.payment_url,
          reference: params.reference,
          transactionId: response.data.pay_token,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.message || 'Failed to initialize payment',
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Orange Money initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      const token = await this.getAccessToken();

      const response = await axios.post(
        `${this.baseUrl}/orange-money-webpay/dev/v1/transactionstatus`,
        {
          order_id: reference,
          amount: 0, // Amount is optional for status check
          pay_token: reference,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const data = response.data;

      if (data.status === 'SUCCESS' || data.status === 'SUCCESSFULL') {
        return {
          success: true,
          status: 'success',
          amount: parseFloat(data.amount),
          currency: data.currency,
          reference: data.order_id,
          transactionId: data.txnid,
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        SUCCESS: 'success',
        SUCCESSFULL: 'success',
        FAILED: 'failed',
        CANCELLED: 'cancelled',
        PENDING: 'pending',
        INITIATED: 'pending',
      };

      return {
        success: false,
        status: statusMap[data.status] || 'pending',
        message: data.message || `Status: ${data.status}`,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Orange Money verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    // Orange Money refunds need to be processed manually
    return {
      success: false,
      message: 'Les remboursements Orange Money doivent etre effectues manuellement',
    };
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      const status = payload.status;
      const orderId = payload.order_id || payload.orderId;

      if (status === 'SUCCESS' || status === 'SUCCESSFULL') {
        return {
          success: true,
          event: 'payment_success',
          reference: orderId,
          status: 'success',
          amount: parseFloat(payload.amount),
          data: payload,
        };
      }

      if (status === 'FAILED') {
        return {
          success: true,
          event: 'payment_failed',
          reference: orderId,
          status: 'failed',
          data: payload,
        };
      }

      return {
        success: true,
        event: `payment_${status?.toLowerCase()}`,
        reference: orderId,
        status: 'pending',
        data: payload,
      };
    } catch (error) {
      this.logger.error(`Orange Money webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }

  // ============================================================================
  // USSD PUSH
  // ============================================================================

  async initiateUssdPush(
    phoneNumber: string,
    amount: number,
    currency: string,
    reference: string,
    description?: string,
  ): Promise<{ success: boolean; transactionId?: string; message?: string }> {
    try {
      const token = await this.getAccessToken();

      // Format phone number (remove +, ensure country code)
      const formattedPhone = phoneNumber.replace(/[^0-9]/g, '');

      const response = await axios.post(
        `${this.baseUrl}/orange-money-webpay/dev/v1/mp`,
        {
          merchant_key: this.merchantKey,
          currency,
          order_id: reference,
          amount: Math.round(amount),
          subscriber_msisdn: formattedPhone,
          description: description || `Paiement ${reference}`,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 201 || response.data.data?.inittxnstatus === '200') {
        return {
          success: true,
          transactionId: response.data.data?.txnid || response.data.txnid,
          message: 'Veuillez valider le paiement sur votre telephone',
        };
      }

      return {
        success: false,
        message: response.data.message || 'USSD push failed',
      };
    } catch (error) {
      this.logger.error(`Orange Money USSD push error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  // ============================================================================
  // CASHOUT (B2C)
  // ============================================================================

  async cashout(
    phoneNumber: string,
    amount: number,
    currency: string,
    reference: string,
  ): Promise<{ success: boolean; transactionId?: string; message?: string }> {
    try {
      const token = await this.getAccessToken();

      const formattedPhone = phoneNumber.replace(/[^0-9]/g, '');

      const response = await axios.post(
        `${this.baseUrl}/orange-money-webpay/dev/v1/cashout`,
        {
          merchant_key: this.merchantKey,
          currency,
          order_id: reference,
          amount: Math.round(amount),
          receiver_msisdn: formattedPhone,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 201) {
        return {
          success: true,
          transactionId: response.data.txnid,
        };
      }

      return {
        success: false,
        message: response.data.message || 'Cashout failed',
      };
    } catch (error) {
      this.logger.error(`Orange Money cashout error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }
}
