import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class MpesaGateway implements PaymentGateway {
  readonly name = 'mpesa';
  readonly displayName = 'M-Pesa';
  readonly supportedCurrencies = ['KES', 'TZS', 'UGX'];
  readonly supportedCountries = ['KE', 'TZ', 'UG'];

  private readonly logger = new Logger(MpesaGateway.name);
  private consumerKey: string;
  private consumerSecret: string;
  private passkey: string;
  private shortcode: string;
  private baseUrl: string;
  private accessToken: string;
  private tokenExpiry: number = 0;

  constructor(private configService: ConfigService) {
    const isLive = this.configService.get<string>('MPESA_MODE') === 'live';
    this.baseUrl = isLive
      ? 'https://api.safaricom.co.ke'
      : 'https://sandbox.safaricom.co.ke';

    this.consumerKey = this.configService.get<string>('MPESA_CONSUMER_KEY');
    this.consumerSecret = this.configService.get<string>('MPESA_CONSUMER_SECRET');
    this.passkey = this.configService.get<string>('MPESA_PASSKEY');
    this.shortcode = this.configService.get<string>('MPESA_SHORTCODE');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  private async getAccessToken(): Promise<string> {
    if (this.accessToken && Date.now() < this.tokenExpiry) {
      return this.accessToken;
    }

    const auth = Buffer.from(`${this.consumerKey}:${this.consumerSecret}`).toString('base64');

    const response = await axios.get(
      `${this.baseUrl}/oauth/v1/generate?grant_type=client_credentials`,
      {
        headers: {
          Authorization: `Basic ${auth}`,
        },
      },
    );

    this.accessToken = response.data.access_token;
    this.tokenExpiry = Date.now() + (response.data.expires_in - 60) * 1000;

    return this.accessToken;
  }

  private generatePassword(): { password: string; timestamp: string } {
    const timestamp = new Date()
      .toISOString()
      .replace(/[^0-9]/g, '')
      .slice(0, 14);
    const password = Buffer.from(`${this.shortcode}${this.passkey}${timestamp}`).toString('base64');
    return { password, timestamp };
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      if (!params.phone) {
        return { success: false, message: 'Phone number is required for M-Pesa' };
      }

      const token = await this.getAccessToken();
      const { password, timestamp } = this.generatePassword();

      // Format phone number (remove leading 0 or +, ensure starts with 254)
      let phoneNumber = params.phone.replace(/[^0-9]/g, '');
      if (phoneNumber.startsWith('0')) {
        phoneNumber = '254' + phoneNumber.substring(1);
      } else if (!phoneNumber.startsWith('254')) {
        phoneNumber = '254' + phoneNumber;
      }

      const callbackUrl = `${this.configService.get('APP_URL')}/api/payment/webhook/mpesa`;

      const response = await axios.post(
        `${this.baseUrl}/mpesa/stkpush/v1/processrequest`,
        {
          BusinessShortCode: this.shortcode,
          Password: password,
          Timestamp: timestamp,
          TransactionType: 'CustomerPayBillOnline',
          Amount: Math.round(params.amount),
          PartyA: phoneNumber,
          PartyB: this.shortcode,
          PhoneNumber: phoneNumber,
          CallBackURL: callbackUrl,
          AccountReference: params.reference,
          TransactionDesc: `Payment for ${params.reference}`,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.ResponseCode === '0') {
        return {
          success: true,
          reference: params.reference,
          transactionId: response.data.CheckoutRequestID,
          message: 'STK Push sent. Please check your phone to complete payment.',
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.ResponseDescription || 'Failed to initiate payment',
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`M-Pesa initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.errorMessage || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      const token = await this.getAccessToken();
      const { password, timestamp } = this.generatePassword();

      const response = await axios.post(
        `${this.baseUrl}/mpesa/stkpushquery/v1/query`,
        {
          BusinessShortCode: this.shortcode,
          Password: password,
          Timestamp: timestamp,
          CheckoutRequestID: reference,
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.ResultCode === '0') {
        return {
          success: true,
          status: 'success',
          reference,
          transactionId: response.data.CheckoutRequestID,
          message: response.data.ResultDesc,
          rawResponse: response.data,
        };
      }

      // Result codes: 0=Success, 1032=Cancelled, 1037=Timeout
      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        '0': 'success',
        '1032': 'cancelled',
        '1037': 'failed',
      };

      return {
        success: false,
        status: statusMap[response.data.ResultCode] || 'failed',
        message: response.data.ResultDesc,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`M-Pesa verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.errorMessage || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    try {
      const token = await this.getAccessToken();

      const response = await axios.post(
        `${this.baseUrl}/mpesa/reversal/v1/request`,
        {
          Initiator: this.configService.get('MPESA_INITIATOR'),
          SecurityCredential: this.configService.get('MPESA_SECURITY_CREDENTIAL'),
          CommandID: 'TransactionReversal',
          TransactionID: reference,
          Amount: amount,
          ReceiverParty: this.shortcode,
          RecieverIdentifierType: '11',
          ResultURL: `${this.configService.get('APP_URL')}/api/payment/webhook/mpesa/reversal`,
          QueueTimeOutURL: `${this.configService.get('APP_URL')}/api/payment/webhook/mpesa/timeout`,
          Remarks: 'Refund',
          Occasion: 'Refund',
        },
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.ResponseCode === '0') {
        return {
          success: true,
          refundId: response.data.ConversationID,
          message: response.data.ResponseDescription,
        };
      }

      return {
        success: false,
        message: response.data.ResponseDescription,
      };
    } catch (error) {
      this.logger.error(`M-Pesa refundPayment error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      const body = payload.Body?.stkCallback || payload;

      const resultCode = body.ResultCode;
      const checkoutRequestID = body.CheckoutRequestID;
      const merchantRequestID = body.MerchantRequestID;

      if (resultCode === 0) {
        // Parse callback metadata
        const callbackMetadata = body.CallbackMetadata?.Item || [];
        const amount = callbackMetadata.find((i: any) => i.Name === 'Amount')?.Value;
        const mpesaReceiptNumber = callbackMetadata.find(
          (i: any) => i.Name === 'MpesaReceiptNumber',
        )?.Value;

        return {
          success: true,
          event: 'payment_success',
          reference: checkoutRequestID,
          status: 'success',
          amount,
          data: {
            merchantRequestID,
            mpesaReceiptNumber,
            ...body,
          },
        };
      }

      return {
        success: true,
        event: 'payment_failed',
        reference: checkoutRequestID,
        status: 'failed',
        data: body,
      };
    } catch (error) {
      this.logger.error(`M-Pesa webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }
}
