import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { PaymentGateway, PaymentGatewayType } from './gateway.interface';
import { StripeGateway } from './stripe.gateway';
import { FlutterwaveGateway } from './flutterwave.gateway';
import { CinetPayGateway } from './cinetpay.gateway';
import { PayDunyaGateway } from './paydunya.gateway';
import { MpesaGateway } from './mpesa.gateway';
import { PaystackGateway } from './paystack.gateway';
import { WaveGateway } from './wave.gateway';
import { OrangeMoneyGateway } from './orange-money.gateway';

@Injectable()
export class GatewayFactory implements OnModuleInit {
  private readonly logger = new Logger(GatewayFactory.name);
  private gateways: Map<string, PaymentGateway> = new Map();

  constructor(
    private stripeGateway: StripeGateway,
    private flutterwaveGateway: FlutterwaveGateway,
    private cinetPayGateway: CinetPayGateway,
    private payDunyaGateway: PayDunyaGateway,
    private mpesaGateway: MpesaGateway,
    private paystackGateway: PaystackGateway,
    private waveGateway: WaveGateway,
    private orangeMoneyGateway: OrangeMoneyGateway,
  ) {}

  onModuleInit() {
    // Register all gateways
    this.registerGateway(this.stripeGateway);
    this.registerGateway(this.flutterwaveGateway);
    this.registerGateway(this.cinetPayGateway);
    this.registerGateway(this.payDunyaGateway);
    this.registerGateway(this.mpesaGateway);
    this.registerGateway(this.paystackGateway);
    this.registerGateway(this.waveGateway);
    this.registerGateway(this.orangeMoneyGateway);

    this.logger.log(`Registered ${this.gateways.size} payment gateways`);
  }

  /**
   * Register a gateway
   */
  registerGateway(gateway: PaymentGateway): void {
    this.gateways.set(gateway.name, gateway);
    this.logger.debug(`Registered gateway: ${gateway.name}`);
  }

  /**
   * Get gateway by name
   */
  getGateway(name: string): PaymentGateway | undefined {
    return this.gateways.get(name);
  }

  /**
   * Get all available gateways
   */
  getAllGateways(): PaymentGateway[] {
    return Array.from(this.gateways.values());
  }

  /**
   * Get gateways that support a specific currency and country
   */
  getGatewaysForCurrency(currency: string, country?: string): PaymentGateway[] {
    return this.getAllGateways().filter(gateway =>
      gateway.isSupported(currency, country),
    );
  }

  /**
   * Get the best gateway for a currency/country combination
   * Priority: Local gateways first, then international
   */
  getBestGateway(currency: string, country?: string): PaymentGateway | undefined {
    const supported = this.getGatewaysForCurrency(currency, country);

    if (supported.length === 0) return undefined;

    // Priority order for different regions
    const priorityMap: Record<string, string[]> = {
      // West Africa (XOF)
      XOF: ['cinetpay', 'paydunya', 'flutterwave', 'stripe'],
      // Central Africa (XAF)
      XAF: ['cinetpay', 'flutterwave', 'stripe'],
      // Nigeria
      NGN: ['flutterwave', 'stripe'],
      // Ghana
      GHS: ['flutterwave', 'stripe'],
      // Kenya
      KES: ['mpesa', 'flutterwave', 'stripe'],
      // Tanzania
      TZS: ['mpesa', 'flutterwave'],
      // Default
      USD: ['stripe', 'flutterwave'],
      EUR: ['stripe', 'flutterwave'],
    };

    const priorities = priorityMap[currency.toUpperCase()] || ['stripe', 'flutterwave'];

    for (const gatewayName of priorities) {
      const gateway = supported.find(g => g.name === gatewayName);
      if (gateway) return gateway;
    }

    return supported[0];
  }

  /**
   * Get gateway information for display
   */
  getGatewayInfo(): Array<{
    name: string;
    displayName: string;
    currencies: string[];
    countries: string[];
  }> {
    return this.getAllGateways().map(gateway => ({
      name: gateway.name,
      displayName: gateway.displayName,
      currencies: gateway.supportedCurrencies,
      countries: gateway.supportedCountries,
    }));
  }

  /**
   * Get available payment methods for a specific context
   */
  getPaymentMethods(
    currency: string,
    country?: string,
  ): Array<{
    gateway: string;
    displayName: string;
    type: 'card' | 'mobile_money' | 'bank_transfer' | 'wallet';
    icon?: string;
  }> {
    const methods: Array<{
      gateway: string;
      displayName: string;
      type: 'card' | 'mobile_money' | 'bank_transfer' | 'wallet';
      icon?: string;
    }> = [];

    const gateways = this.getGatewaysForCurrency(currency, country);

    for (const gateway of gateways) {
      switch (gateway.name) {
        case 'stripe':
          methods.push({
            gateway: 'stripe',
            displayName: 'Carte bancaire',
            type: 'card',
            icon: 'credit-card',
          });
          break;

        case 'flutterwave':
          methods.push({
            gateway: 'flutterwave',
            displayName: 'Carte bancaire',
            type: 'card',
            icon: 'credit-card',
          });
          if (['NGN', 'GHS', 'KES', 'UGX', 'TZS', 'XOF', 'XAF'].includes(currency)) {
            methods.push({
              gateway: 'flutterwave_momo',
              displayName: 'Mobile Money',
              type: 'mobile_money',
              icon: 'smartphone',
            });
          }
          break;

        case 'cinetpay':
          methods.push({
            gateway: 'cinetpay',
            displayName: 'CinetPay',
            type: 'mobile_money',
            icon: 'smartphone',
          });
          break;

        case 'paydunya':
          methods.push({
            gateway: 'paydunya',
            displayName: 'PayDunya',
            type: 'mobile_money',
            icon: 'smartphone',
          });
          if (country === 'SN') {
            methods.push(
              {
                gateway: 'paydunya_orange',
                displayName: 'Orange Money',
                type: 'mobile_money',
                icon: 'orange-money',
              },
              {
                gateway: 'paydunya_wave',
                displayName: 'Wave',
                type: 'mobile_money',
                icon: 'wave',
              },
              {
                gateway: 'paydunya_free',
                displayName: 'Free Money',
                type: 'mobile_money',
                icon: 'free-money',
              },
            );
          }
          break;

        case 'mpesa':
          methods.push({
            gateway: 'mpesa',
            displayName: 'M-Pesa',
            type: 'mobile_money',
            icon: 'mpesa',
          });
          break;
      }
    }

    // Always add wallet if available
    methods.push({
      gateway: 'wallet',
      displayName: 'Portefeuille',
      type: 'wallet',
      icon: 'wallet',
    });

    // Always add cash
    methods.push({
      gateway: 'cash',
      displayName: 'Especes',
      type: 'card', // Using card type for display purposes
      icon: 'cash',
    });

    return methods;
  }
}
