import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
} from '@nestjs/common';
import {
  PartnerService,
  PartnerType,
  PartnerStatus,
  CreatePartnerDto,
} from './partner.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class AddLocationDto {
  name: string;
  address: string;
  latitude: number;
  longitude: number;
}

class CreatePartnerBookingDto {
  guestName: string;
  guestPhone: string;
  guestEmail?: string;
  pickupLocationId?: number;
  dropoffAddress: string;
  dropoffLat: number;
  dropoffLng: number;
  vehicleTypeId: number;
  scheduledAt?: string;
  notes?: string;
  partnerReference?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('partners')
@UseGuards(JwtAuthGuard)
export class PartnerController {
  constructor(private readonly partnerService: PartnerService) {}

  // ==========================================================================
  // PARTNER CRUD
  // ==========================================================================

  /**
   * Lister les partenaires
   */
  @Get()
  async listPartners(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('type') type?: PartnerType,
    @Query('status') status?: PartnerStatus,
    @Query('search') search?: string,
  ) {
    return this.partnerService.listPartners(merchantId, {
      page,
      limit,
      type,
      status,
      search,
    });
  }

  /**
   * Créer un partenaire
   */
  @Post()
  async createPartner(
    @MerchantId() merchantId: number,
    @Body() dto: CreatePartnerDto,
  ) {
    const partner = await this.partnerService.createPartner(merchantId, dto);
    return {
      success: true,
      partner,
      message: 'Partenaire créé. En attente d\'activation.',
    };
  }

  /**
   * Obtenir un partenaire
   */
  @Get(':id')
  async getPartner(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const partner = await this.partnerService.getPartner(id, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return partner;
  }

  /**
   * Mettre à jour un partenaire
   */
  @Put(':id')
  async updatePartner(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: Partial<CreatePartnerDto & { status: PartnerStatus }>,
  ) {
    const partner = await this.partnerService.updatePartner(id, merchantId, dto);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return {
      success: true,
      partner,
    };
  }

  /**
   * Activer un partenaire
   */
  @Post(':id/activate')
  async activatePartner(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const partner = await this.partnerService.activatePartner(id, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return {
      success: true,
      partner,
      message: 'Partenaire activé',
    };
  }

  /**
   * Suspendre un partenaire
   */
  @Post(':id/suspend')
  async suspendPartner(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const partner = await this.partnerService.suspendPartner(id, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return {
      success: true,
      partner,
      message: 'Partenaire suspendu',
    };
  }

  /**
   * Régénérer la clé API
   */
  @Post(':id/regenerate-key')
  async regenerateApiKey(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const result = await this.partnerService.regenerateApiKey(id, merchantId);
    if (!result) {
      return { error: 'Partenaire non trouvé' };
    }
    return {
      success: true,
      apiKey: result.apiKey,
      message: 'Clé API régénérée. Sauvegardez-la, elle ne sera plus affichée.',
    };
  }

  // ==========================================================================
  // PARTNER LOCATIONS
  // ==========================================================================

  /**
   * Lister les localisations d'un partenaire
   */
  @Get(':id/locations')
  async getLocations(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
  ) {
    const partner = await this.partnerService.getPartner(partnerId, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return this.partnerService.getLocations(partnerId);
  }

  /**
   * Ajouter une localisation
   */
  @Post(':id/locations')
  async addLocation(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Body() dto: AddLocationDto,
  ) {
    const location = await this.partnerService.addLocation(partnerId, merchantId, dto);
    return {
      success: true,
      location,
    };
  }

  /**
   * Mettre à jour une localisation
   */
  @Put(':id/locations/:locationId')
  async updateLocation(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Param('locationId', ParseIntPipe) locationId: number,
    @Body() dto: Partial<AddLocationDto & { active: boolean }>,
  ) {
    const partner = await this.partnerService.getPartner(partnerId, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }

    const location = await this.partnerService.updateLocation(locationId, partnerId, dto);
    if (!location) {
      return { error: 'Localisation non trouvée' };
    }
    return {
      success: true,
      location,
    };
  }

  /**
   * Définir la localisation par défaut
   */
  @Post(':id/locations/:locationId/default')
  async setDefaultLocation(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Param('locationId', ParseIntPipe) locationId: number,
  ) {
    const partner = await this.partnerService.getPartner(partnerId, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }

    const success = await this.partnerService.setDefaultLocation(locationId, partnerId);
    return {
      success,
      message: success ? 'Localisation par défaut mise à jour' : 'Localisation non trouvée',
    };
  }

  // ==========================================================================
  // PARTNER BOOKINGS
  // ==========================================================================

  /**
   * Créer une réservation pour un partenaire
   */
  @Post(':id/bookings')
  async createPartnerBooking(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Body() dto: CreatePartnerBookingDto,
  ) {
    const booking = await this.partnerService.createPartnerBooking(partnerId, merchantId, {
      ...dto,
      scheduledAt: dto.scheduledAt ? new Date(dto.scheduledAt) : undefined,
    });

    return {
      success: true,
      booking,
      message: 'Réservation créée',
    };
  }

  /**
   * Lister les réservations d'un partenaire
   */
  @Get(':id/bookings')
  async getPartnerBookings(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('status') status?: string,
    @Query('dateFrom') dateFrom?: string,
    @Query('dateTo') dateTo?: string,
  ) {
    return this.partnerService.getPartnerBookings(partnerId, merchantId, {
      page,
      limit,
      status,
      dateFrom: dateFrom ? new Date(dateFrom) : undefined,
      dateTo: dateTo ? new Date(dateTo) : undefined,
    });
  }

  // ==========================================================================
  // PARTNER REPORTS
  // ==========================================================================

  /**
   * Statistiques d'un partenaire
   */
  @Get(':id/stats')
  async getPartnerStats(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
  ) {
    const partner = await this.partnerService.getPartner(partnerId, merchantId);
    if (!partner) {
      return { error: 'Partenaire non trouvé' };
    }
    return partner.stats;
  }

  /**
   * Rapport d'un partenaire
   */
  @Get(':id/report')
  async getPartnerReport(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) partnerId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    const start = startDate ? new Date(startDate) : new Date(Date.now() - 30 * 24 * 60 * 60 * 1000);
    const end = endDate ? new Date(endDate) : new Date();

    return this.partnerService.getPartnerReport(partnerId, merchantId, start, end);
  }

  // ==========================================================================
  // METADATA
  // ==========================================================================

  /**
   * Types de partenaires
   */
  @Get('meta/types')
  getPartnerTypes() {
    return {
      types: [
        { type: 'hotel', name: 'Hôtel', description: 'Hôtels et hébergements' },
        { type: 'restaurant', name: 'Restaurant', description: 'Restaurants et bars' },
        { type: 'airport', name: 'Aéroport', description: 'Aéroports et gares' },
        { type: 'mall', name: 'Centre commercial', description: 'Centres commerciaux' },
        { type: 'hospital', name: 'Hôpital', description: 'Hôpitaux et cliniques' },
        { type: 'corporate', name: 'Entreprise', description: 'Entreprises et bureaux' },
        { type: 'event_venue', name: 'Lieu événementiel', description: 'Salles de spectacle, stades' },
        { type: 'other', name: 'Autre', description: 'Autre type de partenaire' },
      ],
    };
  }
}
