import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios, { AxiosInstance } from 'axios';

interface WhatsAppResult {
  success: boolean;
  messageId?: string;
  error?: string;
}

interface WhatsAppTemplate {
  name: string;
  language: string;
  components?: Array<{
    type: string;
    parameters: Array<{
      type: string;
      text?: string;
      image?: { link: string };
      document?: { link: string; filename: string };
    }>;
  }>;
}

@Injectable()
export class WhatsAppService implements OnModuleInit {
  private readonly logger = new Logger(WhatsAppService.name);
  private client: AxiosInstance | null = null;
  private phoneNumberId: string;
  private businessAccountId: string;

  constructor(private configService: ConfigService) {}

  async onModuleInit() {
    this.initializeWhatsApp();
  }

  private initializeWhatsApp() {
    try {
      const accessToken = this.configService.get<string>('WHATSAPP_ACCESS_TOKEN');
      this.phoneNumberId = this.configService.get<string>('WHATSAPP_PHONE_NUMBER_ID') || '';
      this.businessAccountId = this.configService.get<string>('WHATSAPP_BUSINESS_ACCOUNT_ID') || '';

      if (!accessToken || !this.phoneNumberId) {
        this.logger.warn('WhatsApp credentials not configured');
        return;
      }

      this.client = axios.create({
        baseURL: 'https://graph.facebook.com/v18.0',
        headers: {
          Authorization: `Bearer ${accessToken}`,
          'Content-Type': 'application/json',
        },
      });

      this.logger.log('WhatsApp Business API initialized');
    } catch (error) {
      this.logger.error(`Failed to initialize WhatsApp: ${error.message}`);
    }
  }

  /**
   * Send WhatsApp text message
   */
  async sendMessage(to: string, message: string): Promise<WhatsAppResult> {
    if (!this.client) {
      this.logger.warn('WhatsApp not initialized, skipping message');
      return { success: false, error: 'WhatsApp service not configured' };
    }

    try {
      const formattedPhone = this.formatPhoneNumber(to);

      const response = await this.client.post(`/${this.phoneNumberId}/messages`, {
        messaging_product: 'whatsapp',
        recipient_type: 'individual',
        to: formattedPhone,
        type: 'text',
        text: {
          preview_url: false,
          body: message,
        },
      });

      const messageId = response.data?.messages?.[0]?.id;
      this.logger.log(`WhatsApp message sent to ${formattedPhone}: ${messageId}`);
      return { success: true, messageId };
    } catch (error) {
      this.logger.error(`Failed to send WhatsApp message: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send WhatsApp template message
   */
  async sendTemplate(to: string, template: WhatsAppTemplate): Promise<WhatsAppResult> {
    if (!this.client) {
      return { success: false, error: 'WhatsApp service not configured' };
    }

    try {
      const formattedPhone = this.formatPhoneNumber(to);

      const response = await this.client.post(`/${this.phoneNumberId}/messages`, {
        messaging_product: 'whatsapp',
        recipient_type: 'individual',
        to: formattedPhone,
        type: 'template',
        template: {
          name: template.name,
          language: {
            code: template.language,
          },
          components: template.components,
        },
      });

      const messageId = response.data?.messages?.[0]?.id;
      this.logger.log(`WhatsApp template sent to ${formattedPhone}: ${messageId}`);
      return { success: true, messageId };
    } catch (error) {
      this.logger.error(`Failed to send WhatsApp template: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send booking confirmation via WhatsApp
   */
  async sendBookingConfirmation(
    to: string,
    driverName: string,
    vehiclePlate: string,
    vehicleModel: string,
    eta: number,
  ): Promise<WhatsAppResult> {
    // Using template for booking confirmation
    return this.sendTemplate(to, {
      name: 'booking_confirmed',
      language: 'fr',
      components: [
        {
          type: 'body',
          parameters: [
            { type: 'text', text: driverName },
            { type: 'text', text: `${vehicleModel} - ${vehiclePlate}` },
            { type: 'text', text: `${eta} min` },
          ],
        },
      ],
    });
  }

  /**
   * Send driver arrived notification via WhatsApp
   */
  async sendDriverArrived(to: string, driverName: string): Promise<WhatsAppResult> {
    const message = `🚗 Votre chauffeur ${driverName} est arrivé!\n\nIl vous attend au point de prise en charge.`;
    return this.sendMessage(to, message);
  }

  /**
   * Send ride completion with receipt via WhatsApp
   */
  async sendRideReceipt(
    to: string,
    bookingId: string,
    fare: {
      baseFare: number;
      distanceFare: number;
      timeFare: number;
      total: number;
      currency: string;
    },
  ): Promise<WhatsAppResult> {
    const message = `🧾 *Reçu de course #${bookingId}*\n\n` +
      `Tarif de base: ${fare.baseFare} ${fare.currency}\n` +
      `Distance: ${fare.distanceFare} ${fare.currency}\n` +
      `Temps: ${fare.timeFare} ${fare.currency}\n` +
      `━━━━━━━━━━━━\n` +
      `*Total: ${fare.total} ${fare.currency}*\n\n` +
      `Merci d'avoir voyagé avec nous! 🙏`;

    return this.sendMessage(to, message);
  }

  /**
   * Send OTP via WhatsApp
   */
  async sendOtp(to: string, otp: string): Promise<WhatsAppResult> {
    return this.sendTemplate(to, {
      name: 'otp_verification',
      language: 'fr',
      components: [
        {
          type: 'body',
          parameters: [
            { type: 'text', text: otp },
          ],
        },
        {
          type: 'button',
          parameters: [
            { type: 'text', text: otp },
          ],
        },
      ],
    });
  }

  /**
   * Send location sharing request
   */
  async sendLocationRequest(to: string): Promise<WhatsAppResult> {
    if (!this.client) {
      return { success: false, error: 'WhatsApp service not configured' };
    }

    try {
      const formattedPhone = this.formatPhoneNumber(to);

      const response = await this.client.post(`/${this.phoneNumberId}/messages`, {
        messaging_product: 'whatsapp',
        recipient_type: 'individual',
        to: formattedPhone,
        type: 'interactive',
        interactive: {
          type: 'location_request_message',
          body: {
            text: 'Partagez votre position pour que nous puissions vous envoyer un chauffeur.',
          },
          action: {
            name: 'send_location',
          },
        },
      });

      const messageId = response.data?.messages?.[0]?.id;
      return { success: true, messageId };
    } catch (error) {
      this.logger.error(`Failed to send location request: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send driver location as WhatsApp location message
   */
  async sendDriverLocation(
    to: string,
    latitude: number,
    longitude: number,
    driverName: string,
  ): Promise<WhatsAppResult> {
    if (!this.client) {
      return { success: false, error: 'WhatsApp service not configured' };
    }

    try {
      const formattedPhone = this.formatPhoneNumber(to);

      const response = await this.client.post(`/${this.phoneNumberId}/messages`, {
        messaging_product: 'whatsapp',
        recipient_type: 'individual',
        to: formattedPhone,
        type: 'location',
        location: {
          latitude,
          longitude,
          name: `Position de ${driverName}`,
          address: 'Votre chauffeur est ici',
        },
      });

      const messageId = response.data?.messages?.[0]?.id;
      return { success: true, messageId };
    } catch (error) {
      this.logger.error(`Failed to send driver location: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Handle incoming webhook from WhatsApp
   */
  async handleWebhook(payload: any): Promise<void> {
    try {
      const entry = payload.entry?.[0];
      const changes = entry?.changes?.[0];
      const value = changes?.value;

      if (value?.messages) {
        for (const message of value.messages) {
          await this.processIncomingMessage(message, value.contacts?.[0]);
        }
      }

      if (value?.statuses) {
        for (const status of value.statuses) {
          this.logger.log(`Message ${status.id} status: ${status.status}`);
        }
      }
    } catch (error) {
      this.logger.error(`Error processing WhatsApp webhook: ${error.message}`);
    }
  }

  private async processIncomingMessage(message: any, contact: any): Promise<void> {
    const from = message.from;
    const type = message.type;
    const timestamp = message.timestamp;

    this.logger.log(`Received ${type} message from ${from}`);

    // Handle different message types
    switch (type) {
      case 'text':
        // Process text message
        this.logger.log(`Text: ${message.text.body}`);
        break;
      case 'location':
        // Process location message
        this.logger.log(`Location: ${message.location.latitude}, ${message.location.longitude}`);
        break;
      case 'interactive':
        // Process interactive response
        if (message.interactive?.type === 'button_reply') {
          this.logger.log(`Button reply: ${message.interactive.button_reply.id}`);
        }
        break;
      default:
        this.logger.log(`Unhandled message type: ${type}`);
    }
  }

  /**
   * Format phone number to international format
   */
  private formatPhoneNumber(phone: string): string {
    let cleaned = phone.replace(/[\s-+]/g, '');

    // Remove leading zeros
    if (cleaned.startsWith('00')) {
      cleaned = cleaned.substring(2);
    } else if (cleaned.startsWith('0')) {
      cleaned = '228' + cleaned.substring(1); // Togo
    }

    return cleaned;
  }
}
