import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

interface PusherResult {
  success: boolean;
  error?: string;
}

interface PusherChannelInfo {
  occupied: boolean;
  subscription_count?: number;
  user_count?: number;
}

@Injectable()
export class PusherService implements OnModuleInit {
  private readonly logger = new Logger(PusherService.name);
  private pusher: any = null;

  constructor(private configService: ConfigService) {}

  async onModuleInit() {
    await this.initializePusher();
  }

  private async initializePusher() {
    try {
      const appId = this.configService.get<string>('PUSHER_APP_ID');
      const key = this.configService.get<string>('PUSHER_KEY');
      const secret = this.configService.get<string>('PUSHER_SECRET');
      const cluster = this.configService.get<string>('PUSHER_CLUSTER') || 'eu';

      if (!appId || !key || !secret) {
        this.logger.warn('Pusher credentials not configured');
        return;
      }

      // Dynamic import
      const Pusher = (await import('pusher')).default;

      this.pusher = new Pusher({
        appId,
        key,
        secret,
        cluster,
        useTLS: true,
      });

      this.logger.log('Pusher service initialized');
    } catch (error) {
      this.logger.error(`Failed to initialize Pusher: ${error.message}`);
    }
  }

  /**
   * Trigger event on a channel
   */
  async trigger(
    channel: string | string[],
    event: string,
    data: any,
  ): Promise<PusherResult> {
    if (!this.pusher) {
      return { success: false, error: 'Pusher not configured' };
    }

    try {
      await this.pusher.trigger(channel, event, data);
      this.logger.debug(`Pusher event ${event} triggered on ${channel}`);
      return { success: true };
    } catch (error) {
      this.logger.error(`Failed to trigger Pusher event: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Trigger event with socket ID exclusion
   */
  async triggerExclude(
    channel: string | string[],
    event: string,
    data: any,
    socketId: string,
  ): Promise<PusherResult> {
    if (!this.pusher) {
      return { success: false, error: 'Pusher not configured' };
    }

    try {
      await this.pusher.trigger(channel, event, data, { socket_id: socketId });
      return { success: true };
    } catch (error) {
      this.logger.error(`Failed to trigger Pusher event: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Trigger batch of events
   */
  async triggerBatch(
    events: Array<{ channel: string; name: string; data: any }>,
  ): Promise<PusherResult> {
    if (!this.pusher) {
      return { success: false, error: 'Pusher not configured' };
    }

    try {
      await this.pusher.triggerBatch(events);
      return { success: true };
    } catch (error) {
      this.logger.error(`Failed to trigger batch: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Authenticate private channel
   */
  authenticateChannel(socketId: string, channel: string): any {
    if (!this.pusher) {
      throw new Error('Pusher not configured');
    }

    return this.pusher.authorizeChannel(socketId, channel);
  }

  /**
   * Authenticate presence channel with user data
   */
  authenticatePresenceChannel(
    socketId: string,
    channel: string,
    userData: { user_id: string; user_info?: any },
  ): any {
    if (!this.pusher) {
      throw new Error('Pusher not configured');
    }

    return this.pusher.authorizeChannel(socketId, channel, userData);
  }

  /**
   * Get channel info
   */
  async getChannelInfo(channel: string): Promise<PusherChannelInfo | null> {
    if (!this.pusher) return null;

    try {
      const response = await this.pusher.get({
        path: `/channels/${channel}`,
        params: { info: 'subscription_count,user_count' },
      });
      return JSON.parse(response.body);
    } catch (error) {
      this.logger.error(`Failed to get channel info: ${error.message}`);
      return null;
    }
  }

  /**
   * Get users in presence channel
   */
  async getPresenceUsers(channel: string): Promise<string[]> {
    if (!this.pusher) return [];

    try {
      const response = await this.pusher.get({
        path: `/channels/${channel}/users`,
      });
      const data = JSON.parse(response.body);
      return data.users?.map((u: any) => u.id) || [];
    } catch (error) {
      this.logger.error(`Failed to get presence users: ${error.message}`);
      return [];
    }
  }

  /**
   * Terminate user connections
   */
  async terminateUserConnections(userId: string): Promise<boolean> {
    if (!this.pusher) return false;

    try {
      await this.pusher.terminateUserConnections(userId);
      return true;
    } catch (error) {
      this.logger.error(`Failed to terminate user connections: ${error.message}`);
      return false;
    }
  }

  // =============================================================================
  // CHANNEL HELPERS
  // =============================================================================

  /**
   * Get user-specific private channel name
   */
  getUserChannel(userId: number): string {
    return `private-user-${userId}`;
  }

  /**
   * Get driver-specific private channel name
   */
  getDriverChannel(driverId: number): string {
    return `private-driver-${driverId}`;
  }

  /**
   * Get booking channel name
   */
  getBookingChannel(bookingId: number): string {
    return `private-booking-${bookingId}`;
  }

  /**
   * Get merchant channel name
   */
  getMerchantChannel(merchantId: number): string {
    return `private-merchant-${merchantId}`;
  }

  /**
   * Get driver presence channel for tracking online drivers
   */
  getDriverPresenceChannel(merchantId: number): string {
    return `presence-drivers-${merchantId}`;
  }

  // =============================================================================
  // BOOKING EVENTS
  // =============================================================================

  /**
   * Broadcast new booking request to driver
   */
  async broadcastNewBookingRequest(driverId: number, booking: any): Promise<PusherResult> {
    return this.trigger(this.getDriverChannel(driverId), 'booking:new_request', {
      id: booking.id,
      merchant_booking_id: booking.merchant_booking_id,
      pickup: {
        address: booking.pickup_address,
        latitude: booking.pickup_latitude,
        longitude: booking.pickup_longitude,
      },
      drop: {
        address: booking.drop_address,
        latitude: booking.drop_latitude,
        longitude: booking.drop_longitude,
      },
      estimate_amount: booking.estimate_amount,
      vehicle_type_id: booking.vehicle_type_id,
      user: booking.user
        ? {
            id: booking.user.id,
            name: `${booking.user.first_name} ${booking.user.last_name}`,
            rating: booking.user.rating,
          }
        : null,
      expires_at: new Date(Date.now() + 30000).toISOString(),
    });
  }

  /**
   * Broadcast booking accepted to user
   */
  async broadcastBookingAccepted(
    userId: number,
    booking: any,
    driver: any,
    vehicle?: any,
  ): Promise<PusherResult> {
    return this.trigger(this.getUserChannel(userId), 'booking:accepted', {
      booking_id: booking.id,
      driver: {
        id: driver.id,
        name: `${driver.first_name} ${driver.last_name}`,
        phone: driver.phoneNumber,
        photo: driver.profile_image,
        rating: driver.average_rating,
        location: {
          latitude: driver.latitude,
          longitude: driver.longitude,
        },
      },
      vehicle: vehicle
        ? {
            make: vehicle.vehicle_make,
            model: vehicle.vehicle_model,
            color: vehicle.vehicle_color,
            plate_number: vehicle.vehicle_plate_number,
          }
        : null,
      eta: booking.eta,
      otp: booking.ride_otp,
    });
  }

  /**
   * Broadcast driver location update
   */
  async broadcastDriverLocation(
    bookingId: number,
    driverId: number,
    location: {
      latitude: number;
      longitude: number;
      heading?: number;
      speed?: number;
    },
  ): Promise<PusherResult> {
    return this.trigger(this.getBookingChannel(bookingId), 'driver:location', {
      driver_id: driverId,
      ...location,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Broadcast booking status change
   */
  async broadcastBookingStatus(
    bookingId: number,
    userId: number,
    driverId: number | null,
    status: number,
    statusLabel: string,
  ): Promise<PusherResult> {
    const channels = [
      this.getBookingChannel(bookingId),
      this.getUserChannel(userId),
    ];

    if (driverId) {
      channels.push(this.getDriverChannel(driverId));
    }

    return this.trigger(channels, 'booking:status_changed', {
      booking_id: bookingId,
      status,
      status_label: statusLabel,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Broadcast booking cancelled
   */
  async broadcastBookingCancelled(
    bookingId: number,
    userId: number,
    driverId: number | null,
    cancelledBy: 'user' | 'driver' | 'system',
    reason?: string,
  ): Promise<PusherResult> {
    const channels = [
      this.getBookingChannel(bookingId),
      this.getUserChannel(userId),
    ];

    if (driverId) {
      channels.push(this.getDriverChannel(driverId));
    }

    return this.trigger(channels, 'booking:cancelled', {
      booking_id: bookingId,
      cancelled_by: cancelledBy,
      reason,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Broadcast chat message
   */
  async broadcastChatMessage(
    bookingId: number,
    message: {
      sender_id: number;
      sender_type: 'user' | 'driver';
      message: string;
      message_type: string;
    },
  ): Promise<PusherResult> {
    return this.trigger(this.getBookingChannel(bookingId), 'chat:message', {
      ...message,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Broadcast to all online drivers in a merchant
   */
  async broadcastToMerchantDrivers(
    merchantId: number,
    event: string,
    data: any,
  ): Promise<PusherResult> {
    return this.trigger(
      this.getDriverPresenceChannel(merchantId),
      event,
      data,
    );
  }
}
