import { Injectable, Logger } from '@nestjs/common';
import { FirebaseService } from './firebase.service';
import { SmsService } from './sms.service';
import { WhatsAppService } from './whatsapp.service';
import { OneSignalService } from './onesignal.service';
import { PusherService } from './pusher.service';
import { PrismaService } from '../../common/prisma/prisma.service';
import {
  BOOKING_STATUS,
  getStatusLabel,
} from '../booking/booking.constants';

export interface PushNotification {
  title: string;
  body: string;
  data?: Record<string, string>;
  image?: string;
  sound?: string;
}

export interface NotificationOptions {
  push?: boolean;
  sms?: boolean;
  whatsapp?: boolean;
  email?: boolean;
  inApp?: boolean;
  persist?: boolean;
}

const DEFAULT_OPTIONS: NotificationOptions = {
  push: true,
  sms: false,
  whatsapp: false,
  email: false,
  inApp: true,
  persist: true,
};

@Injectable()
export class NotificationService {
  private readonly logger = new Logger(NotificationService.name);

  constructor(
    private firebaseService: FirebaseService,
    private smsService: SmsService,
    private whatsappService: WhatsAppService,
    private oneSignalService: OneSignalService,
    private pusherService: PusherService,
    private prisma: PrismaService,
  ) {}

  // =============================================================================
  // CORE NOTIFICATION METHODS
  // =============================================================================

  /**
   * Send notification to a user via multiple channels
   */
  async sendToUser(
    userId: number,
    notification: PushNotification,
    options: NotificationOptions = DEFAULT_OPTIONS,
  ): Promise<{ success: boolean; channels: string[] }> {
    const channels: string[] = [];

    // Get user data
    const user = await this.prisma.user.findUnique({
      where: { id: userId },
      select: {
        id: true,
        UserPhone: true,
        whatsapp_phone: true,
        whatsapp_notifications: true,
        fcm_device_token: true,
        player_id: true,
        email: true,
        merchant_id: true,
      },
    });

    if (!user) {
      this.logger.warn(`User ${userId} not found`);
      return { success: false, channels: [] };
    }

    // Push notification via FCM
    if (options.push && user.fcm_device_token) {
      try {
        await this.firebaseService.sendPushNotification(
          user.fcm_device_token,
          notification.title,
          notification.body,
          notification.data,
        );
        channels.push('fcm');
      } catch (error) {
        this.logger.error(`FCM failed for user ${userId}: ${error.message}`);
      }
    }

    // Push notification via OneSignal
    if (options.push && user.player_id) {
      try {
        await this.oneSignalService.sendToPlayers(
          [user.player_id],
          { title: notification.title, message: notification.body, data: notification.data },
          'user',
        );
        channels.push('onesignal');
      } catch (error) {
        this.logger.error(`OneSignal failed for user ${userId}: ${error.message}`);
      }
    }

    // Real-time via Pusher
    if (options.inApp) {
      try {
        await this.pusherService.trigger(
          this.pusherService.getUserChannel(userId),
          'notification',
          notification,
        );
        channels.push('pusher');
      } catch (error) {
        this.logger.error(`Pusher failed for user ${userId}: ${error.message}`);
      }
    }

    // SMS notification
    if (options.sms && user.UserPhone) {
      try {
        await this.smsService.sendSms(user.UserPhone, notification.body);
        channels.push('sms');
      } catch (error) {
        this.logger.error(`SMS failed for user ${userId}: ${error.message}`);
      }
    }

    // WhatsApp notification
    if (options.whatsapp && user.whatsapp_notifications && user.whatsapp_phone) {
      try {
        await this.whatsappService.sendMessage(user.whatsapp_phone, notification.body);
        channels.push('whatsapp');
      } catch (error) {
        this.logger.error(`WhatsApp failed for user ${userId}: ${error.message}`);
      }
    }

    // Persist notification to database
    if (options.persist) {
      try {
        await this.persistNotification(
          user.merchant_id,
          userId,
          null,
          null,
          notification,
          channels,
        );
      } catch (error) {
        this.logger.error(`Failed to persist notification: ${error.message}`);
      }
    }

    return { success: channels.length > 0, channels };
  }

  /**
   * Send notification to a driver via multiple channels
   */
  async sendToDriver(
    driverId: number,
    notification: PushNotification,
    options: NotificationOptions = DEFAULT_OPTIONS,
  ): Promise<{ success: boolean; channels: string[] }> {
    const channels: string[] = [];

    const driver = await this.prisma.driver.findUnique({
      where: { id: driverId },
      select: {
        id: true,
        phoneNumber: true,
        whatsapp_phone: true,
        whatsapp_notifications: true,
        fcm_device_token: true,
        player_id: true,
        email: true,
        merchant_id: true,
      },
    });

    if (!driver) {
      this.logger.warn(`Driver ${driverId} not found`);
      return { success: false, channels: [] };
    }

    // Push notification via FCM
    if (options.push && driver.fcm_device_token) {
      try {
        await this.firebaseService.sendPushNotification(
          driver.fcm_device_token,
          notification.title,
          notification.body,
          notification.data,
        );
        channels.push('fcm');
      } catch (error) {
        this.logger.error(`FCM failed for driver ${driverId}: ${error.message}`);
      }
    }

    // Push notification via OneSignal
    if (options.push && driver.player_id) {
      try {
        await this.oneSignalService.sendToPlayers(
          [driver.player_id],
          { title: notification.title, message: notification.body, data: notification.data },
          'driver',
        );
        channels.push('onesignal');
      } catch (error) {
        this.logger.error(`OneSignal failed for driver ${driverId}: ${error.message}`);
      }
    }

    // Real-time via Pusher
    if (options.inApp) {
      try {
        await this.pusherService.trigger(
          this.pusherService.getDriverChannel(driverId),
          'notification',
          notification,
        );
        channels.push('pusher');
      } catch (error) {
        this.logger.error(`Pusher failed for driver ${driverId}: ${error.message}`);
      }
    }

    // SMS notification
    if (options.sms && driver.phoneNumber) {
      try {
        await this.smsService.sendSms(driver.phoneNumber, notification.body);
        channels.push('sms');
      } catch (error) {
        this.logger.error(`SMS failed for driver ${driverId}: ${error.message}`);
      }
    }

    // WhatsApp notification
    if (options.whatsapp && driver.whatsapp_notifications && driver.whatsapp_phone) {
      try {
        await this.whatsappService.sendMessage(driver.whatsapp_phone, notification.body);
        channels.push('whatsapp');
      } catch (error) {
        this.logger.error(`WhatsApp failed for driver ${driverId}: ${error.message}`);
      }
    }

    // Persist notification
    if (options.persist) {
      try {
        await this.persistNotification(
          driver.merchant_id || undefined,
          null,
          driverId,
          null,
          notification,
          channels,
        );
      } catch (error) {
        this.logger.error(`Failed to persist notification: ${error.message}`);
      }
    }

    return { success: channels.length > 0, channels };
  }

  /**
   * Persist notification to database
   */
  private async persistNotification(
    merchantId: number | undefined,
    userId: number | null,
    driverId: number | null,
    bookingId: number | null,
    notification: PushNotification,
    sentVia: string[],
  ): Promise<void> {
    await this.prisma.notification.create({
      data: {
        merchant_id: merchantId,
        user_id: userId,
        driver_id: driverId,
        booking_id: bookingId,
        notification_type: notification.data?.type || 'general',
        title: notification.title,
        message: notification.body,
        data: notification.data,
        sent_via: sentVia.join(','),
        sent_at: new Date(),
        is_read: 0,
      },
    });
  }

  // =============================================================================
  // BOOKING NOTIFICATIONS
  // =============================================================================

  /**
   * Notify driver of new booking request
   */
  async notifyNewBookingRequest(
    driverId: number,
    booking: any,
    expiresInSeconds = 30,
  ): Promise<void> {
    const notification: PushNotification = {
      title: '🚗 Nouvelle course',
      body: `De: ${booking.pickup_address || booking.pickup_location}\nVers: ${booking.drop_address || booking.drop_location}`,
      data: {
        type: 'new_booking_request',
        booking_id: String(booking.id),
        expires_at: new Date(Date.now() + expiresInSeconds * 1000).toISOString(),
      },
      sound: 'new_ride',
    };

    await this.sendToDriver(driverId, notification, {
      push: true,
      inApp: true,
      persist: true,
    });

    // Also send via Pusher for real-time
    await this.pusherService.broadcastNewBookingRequest(driverId, booking);
  }

  /**
   * Notify user that booking was accepted
   */
  async notifyBookingAccepted(
    userId: number,
    booking: any,
    driver: any,
    vehicle?: any,
  ): Promise<void> {
    const notification: PushNotification = {
      title: '✅ Course confirmée',
      body: `${driver.first_name} arrive dans ${booking.eta || 5} minutes`,
      data: {
        type: 'booking_accepted',
        booking_id: String(booking.id),
        driver_id: String(driver.id),
      },
      sound: 'booking_accepted',
    };

    // Get user preferences
    const user = await this.prisma.user.findUnique({
      where: { id: userId },
      select: { whatsapp_notifications: true },
    });

    await this.sendToUser(userId, notification, {
      push: true,
      whatsapp: user?.whatsapp_notifications || false,
      inApp: true,
      persist: true,
    });

    // Real-time via Pusher
    await this.pusherService.broadcastBookingAccepted(userId, booking, driver, vehicle);
  }

  /**
   * Notify user that driver is on the way
   */
  async notifyDriverOnTheWay(userId: number, booking: any, eta?: number): Promise<void> {
    const notification: PushNotification = {
      title: '🚗 Chauffeur en route',
      body: eta ? `Arrivée estimée dans ${eta} minutes` : 'Votre chauffeur arrive bientôt',
      data: {
        type: 'driver_on_way',
        booking_id: String(booking.id),
        status: String(BOOKING_STATUS.DRIVER_ON_THE_WAY),
      },
    };

    await this.sendToUser(userId, notification);
  }

  /**
   * Notify user that driver arrived
   */
  async notifyDriverArrived(userId: number, booking: any): Promise<void> {
    const notification: PushNotification = {
      title: '📍 Chauffeur arrivé',
      body: 'Votre chauffeur vous attend au point de prise en charge',
      data: {
        type: 'driver_arrived',
        booking_id: String(booking.id),
        status: String(BOOKING_STATUS.DRIVER_ARRIVED),
      },
      sound: 'driver_arrived',
    };

    const user = await this.prisma.user.findUnique({
      where: { id: userId },
      select: { whatsapp_notifications: true, whatsapp_phone: true },
    });

    await this.sendToUser(userId, notification, {
      push: true,
      sms: true, // Important notification
      whatsapp: user?.whatsapp_notifications || false,
      inApp: true,
      persist: true,
    });
  }

  /**
   * Notify user that ride started
   */
  async notifyRideStarted(userId: number, booking: any): Promise<void> {
    const notification: PushNotification = {
      title: '🚀 Course démarrée',
      body: `En route vers ${booking.drop_address || booking.drop_location}`,
      data: {
        type: 'ride_started',
        booking_id: String(booking.id),
        status: String(BOOKING_STATUS.RIDE_STARTED),
      },
    };

    await this.sendToUser(userId, notification);
  }

  /**
   * Notify user that ride completed
   */
  async notifyRideCompleted(userId: number, booking: any): Promise<void> {
    const notification: PushNotification = {
      title: '🏁 Course terminée',
      body: `Montant: ${booking.final_amount || booking.estimate_amount} XOF`,
      data: {
        type: 'ride_completed',
        booking_id: String(booking.id),
        status: String(BOOKING_STATUS.RIDE_COMPLETED),
        amount: String(booking.final_amount || booking.estimate_amount),
      },
    };

    const user = await this.prisma.user.findUnique({
      where: { id: userId },
      select: { whatsapp_notifications: true, whatsapp_phone: true, UserPhone: true },
    });

    await this.sendToUser(userId, notification, {
      push: true,
      whatsapp: user?.whatsapp_notifications || false,
      inApp: true,
      persist: true,
    });

    // Send receipt via WhatsApp if enabled
    if (user?.whatsapp_notifications && user?.whatsapp_phone) {
      await this.whatsappService.sendRideReceipt(user.whatsapp_phone, booking.merchant_booking_id, {
        baseFare: 0,
        distanceFare: 0,
        timeFare: 0,
        total: booking.final_amount || booking.estimate_amount,
        currency: 'XOF',
      });
    }
  }

  /**
   * Notify about booking cancellation
   */
  async notifyBookingCancelled(
    booking: any,
    cancelledBy: 'user' | 'driver' | 'system',
    reason?: string,
  ): Promise<void> {
    const cancellerLabel = cancelledBy === 'user' ? "l'utilisateur" :
      cancelledBy === 'driver' ? 'le chauffeur' : 'le système';

    // Notify user (if cancelled by driver or system)
    if (cancelledBy !== 'user' && booking.user_id) {
      await this.sendToUser(booking.user_id, {
        title: '❌ Course annulée',
        body: `La course a été annulée par ${cancellerLabel}${reason ? `. Raison: ${reason}` : ''}`,
        data: {
          type: 'booking_cancelled',
          booking_id: String(booking.id),
          cancelled_by: cancelledBy,
        },
      });
    }

    // Notify driver (if cancelled by user or system)
    if (cancelledBy !== 'driver' && booking.driver_id) {
      await this.sendToDriver(booking.driver_id, {
        title: '❌ Course annulée',
        body: `La course a été annulée par ${cancellerLabel}${reason ? `. Raison: ${reason}` : ''}`,
        data: {
          type: 'booking_cancelled',
          booking_id: String(booking.id),
          cancelled_by: cancelledBy,
        },
      });
    }

    // Broadcast via Pusher
    await this.pusherService.broadcastBookingCancelled(
      booking.id,
      booking.user_id,
      booking.driver_id,
      cancelledBy,
      reason,
    );
  }

  /**
   * Notify user that no driver was found
   */
  async notifyNoDriverFound(userId: number, bookingId: number): Promise<void> {
    await this.sendToUser(userId, {
      title: '😔 Aucun chauffeur disponible',
      body: 'Aucun chauffeur disponible pour le moment. Veuillez réessayer.',
      data: {
        type: 'no_driver_found',
        booking_id: String(bookingId),
        status: String(BOOKING_STATUS.NO_DRIVER_FOUND),
      },
    });
  }

  // =============================================================================
  // DRIVER NOTIFICATIONS
  // =============================================================================

  /**
   * Notify driver of new earnings
   */
  async notifyDriverEarnings(
    driverId: number,
    booking: any,
    earnings: number,
  ): Promise<void> {
    await this.sendToDriver(driverId, {
      title: '💰 Nouveau gain',
      body: `+${earnings} XOF pour la course #${booking.merchant_booking_id}`,
      data: {
        type: 'new_earnings',
        booking_id: String(booking.id),
        amount: String(earnings),
      },
    });
  }

  /**
   * Notify driver request expired
   */
  async notifyDriverRequestExpired(driverId: number, bookingId: number): Promise<void> {
    await this.sendToDriver(driverId, {
      title: '⏰ Demande expirée',
      body: 'La demande de course a expiré',
      data: {
        type: 'request_expired',
        booking_id: String(bookingId),
      },
    }, { push: true, inApp: true, persist: false });
  }

  // =============================================================================
  // OTP & VERIFICATION
  // =============================================================================

  /**
   * Send OTP via SMS
   */
  async sendOtpSms(phone: string, otp: string, merchantName = 'MonkAPI'): Promise<boolean> {
    const result = await this.smsService.sendOtp(phone, otp, merchantName);
    return result.success;
  }

  /**
   * Send OTP via WhatsApp
   */
  async sendOtpWhatsApp(phone: string, otp: string): Promise<boolean> {
    const result = await this.whatsappService.sendOtp(phone, otp);
    return result.success;
  }

  // =============================================================================
  // MARKETING & PROMO
  // =============================================================================

  /**
   * Send promotional notification to all users of a merchant
   */
  async sendPromoToMerchantUsers(
    merchantId: number,
    title: string,
    message: string,
  ): Promise<void> {
    // Send via OneSignal segment
    await this.oneSignalService.sendWithFilters(
      [{ field: 'tag', key: 'merchant_id', value: String(merchantId) }],
      { title, message, data: { type: 'promo' } },
      'user',
    );
  }

  /**
   * Send promotional notification to all drivers of a merchant
   */
  async sendPromoToMerchantDrivers(
    merchantId: number,
    title: string,
    message: string,
  ): Promise<void> {
    await this.oneSignalService.sendWithFilters(
      [{ field: 'tag', key: 'merchant_id', value: String(merchantId) }],
      { title, message, data: { type: 'promo' } },
      'driver',
    );
  }

  // =============================================================================
  // USER NOTIFICATIONS MANAGEMENT
  // =============================================================================

  /**
   * Get user's notifications
   */
  async getUserNotifications(
    userId: number,
    page = 1,
    limit = 20,
  ) {
    const skip = (page - 1) * limit;

    const [notifications, total] = await Promise.all([
      this.prisma.notification.findMany({
        where: { user_id: userId },
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
      }),
      this.prisma.notification.count({ where: { user_id: userId } }),
    ]);

    const unreadCount = await this.prisma.notification.count({
      where: { user_id: userId, is_read: 0 },
    });

    return {
      notifications,
      unreadCount,
      pagination: {
        page,
        limit,
        total,
        total_pages: Math.ceil(total / limit),
      },
    };
  }

  /**
   * Mark notification as read
   */
  async markAsRead(notificationId: number, userId: number): Promise<void> {
    await this.prisma.notification.updateMany({
      where: { id: notificationId, user_id: userId },
      data: { is_read: 1, read_at: new Date() },
    });
  }

  /**
   * Mark all notifications as read
   */
  async markAllAsRead(userId: number): Promise<void> {
    await this.prisma.notification.updateMany({
      where: { user_id: userId, is_read: 0 },
      data: { is_read: 1, read_at: new Date() },
    });
  }
}
