// =============================================================================
// NotificationService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { ConfigService } from '@nestjs/config';
import { NotificationService } from './notification.service';
import { FirebaseService } from './firebase.service';
import { SmsService } from './sms.service';
import { WhatsAppService } from './whatsapp.service';
import { OneSignalService } from './onesignal.service';
import { PusherService } from './pusher.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  mockConfigService,
  mockFirebaseService,
  mockSmsService,
  mockPusherService,
  mockOneSignalService,
  createMockUser,
  createMockDriver,
  createMockNotification,
  createMockBooking,
  resetAllMocks,
} from '../../../test/setup';

// Mock WhatsApp service
const mockWhatsAppService = {
  sendMessage: jest.fn(() => Promise.resolve({ success: true, messageId: 'WA123' })),
  sendTemplate: jest.fn(() => Promise.resolve({ success: true })),
  sendBookingConfirmation: jest.fn(() => Promise.resolve({ success: true })),
  sendDriverArrived: jest.fn(() => Promise.resolve({ success: true })),
  sendRideReceipt: jest.fn(() => Promise.resolve({ success: true })),
};

describe('NotificationService', () => {
  let service: NotificationService;
  let prisma: typeof mockPrismaService;

  beforeEach(async () => {
    resetAllMocks();

    // Ajouter les mocks pour userDevice et driverDevice
    mockPrismaService.userDevice = {
      findFirst: jest.fn(),
      findMany: jest.fn(),
    };
    mockPrismaService.driverDevice = {
      findFirst: jest.fn(),
      findMany: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        NotificationService,
        { provide: PrismaService, useValue: mockPrismaService },
        { provide: ConfigService, useValue: mockConfigService },
        { provide: FirebaseService, useValue: mockFirebaseService },
        { provide: SmsService, useValue: mockSmsService },
        { provide: WhatsAppService, useValue: mockWhatsAppService },
        { provide: OneSignalService, useValue: mockOneSignalService },
        { provide: PusherService, useValue: mockPusherService },
      ],
    }).compile();

    service = module.get<NotificationService>(NotificationService);
    prisma = mockPrismaService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // SEND TO USER
  // ===========================================================================

  describe('sendToUser', () => {
    it('devrait envoyer une notification push à un utilisateur', async () => {
      const mockUser = createMockUser();
      const mockDevice = {
        player_id: 'device-token-123',
        device_type: 'android',
      };

      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue(mockDevice);
      prisma.notification.create.mockResolvedValue(createMockNotification());

      await service.sendToUser(
        1,
        {
          title: 'Test',
          body: 'Test notification',
          data: { type: 'test' },
        },
        { push: true },
      );

      expect(mockFirebaseService.sendPushNotification).toHaveBeenCalled();
    });

    it('devrait envoyer une notification via OneSignal', async () => {
      const mockUser = createMockUser();
      const mockDevice = {
        player_id: 'onesignal-player-123',
        device_type: 'ios',
      };

      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue(mockDevice);
      prisma.notification.create.mockResolvedValue(createMockNotification());

      await service.sendToUser(
        1,
        { title: 'Test', body: 'Test' },
        { push: true, useOneSignal: true },
      );

      expect(mockOneSignalService.sendToPlayers).toHaveBeenCalled();
    });

    it('devrait envoyer un SMS si demandé', async () => {
      const mockUser = createMockUser({ phone: '+22890123456' });

      prisma.user.findUnique.mockResolvedValue(mockUser);

      await service.sendToUser(
        1,
        { title: 'OTP', body: 'Votre code: 123456' },
        { sms: true },
      );

      expect(mockSmsService.sendSms).toHaveBeenCalledWith(
        '+22890123456',
        expect.any(String),
      );
    });

    it('devrait sauvegarder la notification en base', async () => {
      const mockUser = createMockUser();
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.notification.create.mockResolvedValue(createMockNotification());

      await service.sendToUser(
        1,
        { title: 'Test', body: 'Test' },
        { inApp: true },
      );

      expect(prisma.notification.create).toHaveBeenCalledWith(
        expect.objectContaining({
          data: expect.objectContaining({
            user_id: 1,
            title: 'Test',
            body: 'Test',
          }),
        }),
      );
    });

    it('ne devrait rien envoyer si utilisateur non trouvé', async () => {
      prisma.user.findUnique.mockResolvedValue(null);

      await service.sendToUser(
        999,
        { title: 'Test', body: 'Test' },
        { push: true },
      );

      expect(mockFirebaseService.sendPushNotification).not.toHaveBeenCalled();
    });
  });

  // ===========================================================================
  // SEND TO DRIVER
  // ===========================================================================

  describe('sendToDriver', () => {
    it('devrait envoyer une notification push à un chauffeur', async () => {
      const mockDriver = createMockDriver();
      const mockDevice = {
        player_id: 'driver-device-token',
        device_type: 'android',
      };

      prisma.driver.findUnique.mockResolvedValue(mockDriver);
      prisma.driverDevice.findFirst.mockResolvedValue(mockDevice);
      prisma.notification.create.mockResolvedValue(createMockNotification());

      await service.sendToDriver(
        1,
        {
          title: 'Nouvelle course',
          body: 'Une course est disponible',
          data: { type: 'new_booking' },
        },
        { push: true },
      );

      expect(mockFirebaseService.sendPushNotification).toHaveBeenCalled();
    });

    it('devrait utiliser OneSignal pour les chauffeurs', async () => {
      const mockDriver = createMockDriver();
      const mockDevice = { player_id: 'driver-onesignal' };

      prisma.driver.findUnique.mockResolvedValue(mockDriver);
      prisma.driverDevice.findFirst.mockResolvedValue(mockDevice);

      await service.sendToDriver(
        1,
        { title: 'Test', body: 'Test' },
        { push: true, useOneSignal: true },
      );

      expect(mockOneSignalService.sendToPlayers).toHaveBeenCalledWith(
        ['driver-onesignal'],
        expect.any(Object),
        'driver',
      );
    });
  });

  // ===========================================================================
  // SEND BULK
  // ===========================================================================

  describe('sendBulk', () => {
    it('devrait envoyer des notifications en masse', async () => {
      const mockUsers = [
        createMockUser({ id: 1 }),
        createMockUser({ id: 2 }),
        createMockUser({ id: 3 }),
      ];

      prisma.user.findMany.mockResolvedValue(mockUsers);
      prisma.userDevice.findMany.mockResolvedValue([
        { user_id: 1, player_id: 'token1' },
        { user_id: 2, player_id: 'token2' },
        { user_id: 3, player_id: 'token3' },
      ]);
      mockFirebaseService.sendMulticast.mockResolvedValue({
        successCount: 3,
        failureCount: 0,
      });

      const result = await service.sendBulk(
        [1, 2, 3],
        'user',
        { title: 'Promo!', body: '-20% sur votre prochaine course' },
      );

      expect(result.sent).toBe(3);
      expect(result.failed).toBe(0);
    });

    it('devrait gérer les échecs partiels', async () => {
      prisma.user.findMany.mockResolvedValue([
        createMockUser({ id: 1 }),
        createMockUser({ id: 2 }),
      ]);
      prisma.userDevice.findMany.mockResolvedValue([
        { user_id: 1, player_id: 'token1' },
        { user_id: 2, player_id: 'token2' },
      ]);
      mockFirebaseService.sendMulticast.mockResolvedValue({
        successCount: 1,
        failureCount: 1,
      });

      const result = await service.sendBulk(
        [1, 2],
        'user',
        { title: 'Test', body: 'Test' },
      );

      expect(result.sent).toBe(1);
      expect(result.failed).toBe(1);
    });
  });

  // ===========================================================================
  // BOOKING NOTIFICATIONS
  // ===========================================================================

  describe('notifyBookingAccepted', () => {
    it('devrait notifier l\'utilisateur quand sa course est acceptée', async () => {
      const mockUser = createMockUser();
      const mockDriver = createMockDriver();
      const mockBooking = createMockBooking();

      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue({ player_id: 'token' });

      await service.notifyBookingAccepted(mockBooking, mockDriver);

      expect(mockPusherService.broadcastBookingAccepted).toHaveBeenCalled();
    });
  });

  describe('notifyNewBookingRequest', () => {
    it('devrait notifier le chauffeur d\'une nouvelle demande', async () => {
      const mockDriver = createMockDriver();
      const mockBooking = createMockBooking();

      prisma.driver.findUnique.mockResolvedValue(mockDriver);
      prisma.driverDevice.findFirst.mockResolvedValue({
        player_id: 'driver-token',
      });

      await service.notifyNewBookingRequest(mockDriver.id, mockBooking);

      expect(mockPusherService.broadcastNewBookingRequest).toHaveBeenCalled();
      expect(mockOneSignalService.notifyNewBookingRequest).toHaveBeenCalled();
    });
  });

  // ===========================================================================
  // GET USER NOTIFICATIONS
  // ===========================================================================

  describe('getUserNotifications', () => {
    it('devrait retourner les notifications d\'un utilisateur', async () => {
      const mockNotifications = [
        createMockNotification({ id: 1 }),
        createMockNotification({ id: 2 }),
      ];

      prisma.notification.findMany.mockResolvedValue(mockNotifications);
      prisma.notification.count.mockResolvedValue(2);

      const result = await service.getUserNotifications(1, 1, {
        page: 1,
        limit: 10,
      });

      expect(result.data).toHaveLength(2);
      expect(result.total).toBe(2);
    });

    it('devrait filtrer les notifications non lues', async () => {
      prisma.notification.findMany.mockResolvedValue([
        createMockNotification({ is_read: 0 }),
      ]);
      prisma.notification.count.mockResolvedValue(1);

      const result = await service.getUserNotifications(1, 1, {
        unreadOnly: true,
      });

      expect(prisma.notification.findMany).toHaveBeenCalledWith(
        expect.objectContaining({
          where: expect.objectContaining({
            is_read: 0,
          }),
        }),
      );
    });
  });

  // ===========================================================================
  // MARK AS READ
  // ===========================================================================

  describe('markAsRead', () => {
    it('devrait marquer une notification comme lue', async () => {
      prisma.notification.update.mockResolvedValue(
        createMockNotification({ is_read: 1 }),
      );

      await service.markAsRead(1, 1);

      expect(prisma.notification.update).toHaveBeenCalledWith({
        where: { id: 1 },
        data: { is_read: 1 },
      });
    });

    it('devrait marquer toutes les notifications comme lues', async () => {
      prisma.notification.updateMany.mockResolvedValue({ count: 5 });

      await service.markAllAsRead(1, 1);

      expect(prisma.notification.updateMany).toHaveBeenCalledWith({
        where: { user_id: 1, merchant_id: 1 },
        data: { is_read: 1 },
      });
    });
  });

  // ===========================================================================
  // EDGE CASES
  // ===========================================================================

  describe('edge cases', () => {
    it('devrait gérer les erreurs Firebase gracieusement', async () => {
      const mockUser = createMockUser();
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue({ player_id: 'token' });
      mockFirebaseService.sendPushNotification.mockRejectedValue(
        new Error('Firebase error'),
      );

      // Ne devrait pas lever d'exception
      await expect(
        service.sendToUser(
          1,
          { title: 'Test', body: 'Test' },
          { push: true },
        ),
      ).resolves.not.toThrow();
    });

    it('devrait gérer l\'absence de device token', async () => {
      const mockUser = createMockUser();
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue(null);

      await service.sendToUser(
        1,
        { title: 'Test', body: 'Test' },
        { push: true },
      );

      expect(mockFirebaseService.sendPushNotification).not.toHaveBeenCalled();
    });

    it('devrait envoyer via plusieurs canaux simultanément', async () => {
      const mockUser = createMockUser({ phone: '+22890123456' });
      prisma.user.findUnique.mockResolvedValue(mockUser);
      prisma.userDevice.findFirst.mockResolvedValue({ player_id: 'token' });
      prisma.notification.create.mockResolvedValue(createMockNotification());

      await service.sendToUser(
        1,
        { title: 'Important', body: 'Message important' },
        { push: true, sms: true, inApp: true },
      );

      expect(mockFirebaseService.sendPushNotification).toHaveBeenCalled();
      expect(mockSmsService.sendSms).toHaveBeenCalled();
      expect(prisma.notification.create).toHaveBeenCalled();
    });
  });
});
