import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { I18nService } from './i18n.service';

export interface TranslationEntity {
  id: number;
  key: string;
  language: string;
  value: string;
  namespace?: string;
  merchantId?: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface TranslationInput {
  key: string;
  language: string;
  value: string;
  namespace?: string;
}

@Injectable()
export class TranslationService {
  private readonly logger = new Logger(TranslationService.name);
  private cache: Map<string, string> = new Map();
  private cacheLoaded: boolean = false;

  constructor(
    private prisma: PrismaService,
    private i18nService: I18nService,
  ) {}

  /**
   * Get translation from database (with fallback to files)
   */
  async getTranslation(
    key: string,
    language: string,
    merchantId?: number,
  ): Promise<string> {
    const cacheKey = this.getCacheKey(key, language, merchantId);

    // Check cache
    if (this.cache.has(cacheKey)) {
      return this.cache.get(cacheKey)!;
    }

    // Check database
    try {
      const where: any = { key, language };
      if (merchantId) {
        where.merchant_id = merchantId;
      }

      const translation = await this.prisma.translation.findFirst({
        where,
        orderBy: {
          merchant_id: merchantId ? 'desc' : 'asc', // Prefer merchant-specific
        },
      });

      if (translation) {
        this.cache.set(cacheKey, translation.value);
        return translation.value;
      }
    } catch (error) {
      this.logger.debug(`DB translation not found for ${key}, using file`);
    }

    // Fallback to file-based translations
    return this.i18nService.translate(key, language);
  }

  /**
   * Translate with parameters
   */
  async translate(
    key: string,
    language: string,
    params?: Record<string, string | number>,
    merchantId?: number,
  ): Promise<string> {
    let translation = await this.getTranslation(key, language, merchantId);

    // Replace parameters
    if (params) {
      translation = translation.replace(/\{(\w+)\}/g, (match, paramKey) => {
        return params[paramKey]?.toString() ?? match;
      });
    }

    return translation;
  }

  /**
   * Alias for translate
   */
  async t(
    key: string,
    language: string,
    params?: Record<string, string | number>,
    merchantId?: number,
  ): Promise<string> {
    return this.translate(key, language, params, merchantId);
  }

  /**
   * Create or update translation
   */
  async upsertTranslation(
    input: TranslationInput,
    merchantId?: number,
  ): Promise<TranslationEntity> {
    const where: any = {
      key: input.key,
      language: input.language,
      merchant_id: merchantId || null,
    };

    const data = {
      key: input.key,
      language: input.language,
      value: input.value,
      namespace: input.namespace,
      merchant_id: merchantId,
      updated_at: new Date(),
    };

    const existing = await this.prisma.translation.findFirst({ where });

    let result;
    if (existing) {
      result = await this.prisma.translation.update({
        where: { id: existing.id },
        data,
      });
    } else {
      result = await this.prisma.translation.create({
        data: {
          ...data,
          created_at: new Date(),
        },
      });
    }

    // Clear cache for this key
    this.clearCacheForKey(input.key, input.language, merchantId);

    return result as unknown as TranslationEntity;
  }

  /**
   * Bulk upsert translations
   */
  async bulkUpsertTranslations(
    translations: TranslationInput[],
    merchantId?: number,
  ): Promise<{ created: number; updated: number }> {
    let created = 0;
    let updated = 0;

    for (const input of translations) {
      const where: any = {
        key: input.key,
        language: input.language,
        merchant_id: merchantId || null,
      };

      const existing = await this.prisma.translation.findFirst({ where });

      if (existing) {
        await this.prisma.translation.update({
          where: { id: existing.id },
          data: {
            value: input.value,
            namespace: input.namespace,
            updated_at: new Date(),
          },
        });
        updated++;
      } else {
        await this.prisma.translation.create({
          data: {
            key: input.key,
            language: input.language,
            value: input.value,
            namespace: input.namespace,
            merchant_id: merchantId,
            created_at: new Date(),
            updated_at: new Date(),
          },
        });
        created++;
      }
    }

    // Clear cache
    this.cache.clear();

    return { created, updated };
  }

  /**
   * Delete translation
   */
  async deleteTranslation(
    key: string,
    language: string,
    merchantId?: number,
  ): Promise<boolean> {
    try {
      await this.prisma.translation.deleteMany({
        where: {
          key,
          language,
          merchant_id: merchantId || null,
        },
      });

      this.clearCacheForKey(key, language, merchantId);
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Get all translations for a language
   */
  async getTranslationsForLanguage(
    language: string,
    merchantId?: number,
    namespace?: string,
  ): Promise<Record<string, string>> {
    const where: any = { language };
    if (merchantId) {
      where.OR = [{ merchant_id: merchantId }, { merchant_id: null }];
    }
    if (namespace) {
      where.namespace = namespace;
    }

    const translations = await this.prisma.translation.findMany({
      where,
      orderBy: { merchant_id: 'asc' }, // General first, then merchant-specific
    });

    const result: Record<string, string> = {};
    for (const t of translations) {
      // Merchant-specific overrides general
      result[t.key] = t.value;
    }

    return result;
  }

  /**
   * Get missing translations for a language
   */
  async getMissingTranslations(
    sourceLanguage: string,
    targetLanguage: string,
    merchantId?: number,
  ): Promise<string[]> {
    const sourceWhere: any = { language: sourceLanguage };
    const targetWhere: any = { language: targetLanguage };

    if (merchantId) {
      sourceWhere.OR = [{ merchant_id: merchantId }, { merchant_id: null }];
      targetWhere.OR = [{ merchant_id: merchantId }, { merchant_id: null }];
    }

    const [sourceKeys, targetKeys] = await Promise.all([
      this.prisma.translation.findMany({
        where: sourceWhere,
        select: { key: true },
      }),
      this.prisma.translation.findMany({
        where: targetWhere,
        select: { key: true },
      }),
    ]);

    const sourceSet = new Set(sourceKeys.map((t) => t.key));
    const targetSet = new Set(targetKeys.map((t) => t.key));

    return Array.from(sourceSet).filter((key) => !targetSet.has(key));
  }

  /**
   * Export translations for a language
   */
  async exportTranslations(
    language: string,
    merchantId?: number,
    format: 'json' | 'csv' = 'json',
  ): Promise<string> {
    const translations = await this.getTranslationsForLanguage(
      language,
      merchantId,
    );

    if (format === 'csv') {
      let csv = 'key,value\n';
      for (const [key, value] of Object.entries(translations)) {
        csv += `"${key}","${value.replace(/"/g, '""')}"\n`;
      }
      return csv;
    }

    return JSON.stringify(translations, null, 2);
  }

  /**
   * Import translations from JSON
   */
  async importTranslations(
    language: string,
    translations: Record<string, string>,
    merchantId?: number,
    namespace?: string,
  ): Promise<{ created: number; updated: number }> {
    const inputs: TranslationInput[] = Object.entries(translations).map(
      ([key, value]) => ({
        key,
        language,
        value,
        namespace,
      }),
    );

    return this.bulkUpsertTranslations(inputs, merchantId);
  }

  /**
   * Search translations
   */
  async searchTranslations(
    query: string,
    language?: string,
    merchantId?: number,
  ): Promise<TranslationEntity[]> {
    const where: any = {
      OR: [
        { key: { contains: query } },
        { value: { contains: query } },
      ],
    };

    if (language) {
      where.language = language;
    }

    if (merchantId) {
      where.OR = [
        { ...where.OR, merchant_id: merchantId },
        { ...where.OR, merchant_id: null },
      ];
    }

    return this.prisma.translation.findMany({
      where,
      orderBy: { key: 'asc' },
      take: 100,
    }) as unknown as TranslationEntity[];
  }

  /**
   * Preload translations into cache
   */
  async preloadCache(
    languages: string[] = ['en', 'fr', 'ar'],
    merchantId?: number,
  ): Promise<void> {
    this.logger.log('Preloading translation cache...');

    for (const language of languages) {
      const translations = await this.getTranslationsForLanguage(
        language,
        merchantId,
      );

      for (const [key, value] of Object.entries(translations)) {
        const cacheKey = this.getCacheKey(key, language, merchantId);
        this.cache.set(cacheKey, value);
      }
    }

    this.cacheLoaded = true;
    this.logger.log(`Loaded ${this.cache.size} translations into cache`);
  }

  /**
   * Clear cache
   */
  clearCache(): void {
    this.cache.clear();
    this.cacheLoaded = false;
  }

  private getCacheKey(
    key: string,
    language: string,
    merchantId?: number,
  ): string {
    return `${merchantId || 'global'}:${language}:${key}`;
  }

  private clearCacheForKey(
    key: string,
    language: string,
    merchantId?: number,
  ): void {
    const cacheKey = this.getCacheKey(key, language, merchantId);
    this.cache.delete(cacheKey);
    // Also clear global if merchant-specific
    if (merchantId) {
      const globalKey = this.getCacheKey(key, language);
      this.cache.delete(globalKey);
    }
  }
}
