import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as fs from 'fs';
import * as path from 'path';

export interface TranslationFile {
  [key: string]: string | TranslationFile;
}

export interface SupportedLanguage {
  code: string;
  name: string;
  nativeName: string;
  direction: 'ltr' | 'rtl';
  default?: boolean;
}

@Injectable()
export class I18nService implements OnModuleInit {
  private readonly logger = new Logger(I18nService.name);
  private translations: Map<string, TranslationFile> = new Map();
  private defaultLanguage: string;
  private fallbackLanguage: string;
  private supportedLanguages: SupportedLanguage[];
  private translationsPath: string;

  constructor(private configService: ConfigService) {
    this.defaultLanguage = this.configService.get<string>('DEFAULT_LANGUAGE', 'en');
    this.fallbackLanguage = this.configService.get<string>('FALLBACK_LANGUAGE', 'en');
    this.translationsPath = this.configService.get<string>(
      'TRANSLATIONS_PATH',
      './src/modules/i18n/translations',
    );

    this.supportedLanguages = [
      { code: 'en', name: 'English', nativeName: 'English', direction: 'ltr', default: true },
      { code: 'fr', name: 'French', nativeName: 'Français', direction: 'ltr' },
      { code: 'ar', name: 'Arabic', nativeName: 'العربية', direction: 'rtl' },
      { code: 'pt', name: 'Portuguese', nativeName: 'Português', direction: 'ltr' },
      { code: 'es', name: 'Spanish', nativeName: 'Español', direction: 'ltr' },
      { code: 'sw', name: 'Swahili', nativeName: 'Kiswahili', direction: 'ltr' },
      { code: 'am', name: 'Amharic', nativeName: 'አማርኛ', direction: 'ltr' },
      { code: 'yo', name: 'Yoruba', nativeName: 'Yorùbá', direction: 'ltr' },
      { code: 'ha', name: 'Hausa', nativeName: 'Hausa', direction: 'ltr' },
      { code: 'ig', name: 'Igbo', nativeName: 'Igbo', direction: 'ltr' },
    ];
  }

  async onModuleInit() {
    await this.loadTranslations();
  }

  /**
   * Load all translation files
   */
  private async loadTranslations(): Promise<void> {
    const translationsDir = path.resolve(this.translationsPath);

    if (!fs.existsSync(translationsDir)) {
      this.logger.warn(`Translations directory not found: ${translationsDir}`);
      // Create default translations in memory
      this.createDefaultTranslations();
      return;
    }

    try {
      const files = fs.readdirSync(translationsDir);

      for (const file of files) {
        if (file.endsWith('.json')) {
          const langCode = file.replace('.json', '');
          const filePath = path.join(translationsDir, file);
          const content = fs.readFileSync(filePath, 'utf-8');
          this.translations.set(langCode, JSON.parse(content));
          this.logger.log(`Loaded translations for: ${langCode}`);
        }
      }
    } catch (error) {
      this.logger.error(`Error loading translations: ${error.message}`);
      this.createDefaultTranslations();
    }
  }

  /**
   * Create default translations in memory
   */
  private createDefaultTranslations(): void {
    // English (default)
    this.translations.set('en', this.getDefaultEnglishTranslations());
    // French
    this.translations.set('fr', this.getDefaultFrenchTranslations());
    // Arabic
    this.translations.set('ar', this.getDefaultArabicTranslations());

    this.logger.log('Created default in-memory translations');
  }

  /**
   * Get translation for a key
   */
  translate(
    key: string,
    language?: string,
    params?: Record<string, string | number>,
  ): string {
    const lang = language || this.defaultLanguage;
    let translation = this.getNestedValue(
      this.translations.get(lang) || {},
      key,
    );

    // Fallback to default language
    if (!translation && lang !== this.fallbackLanguage) {
      translation = this.getNestedValue(
        this.translations.get(this.fallbackLanguage) || {},
        key,
      );
    }

    // Return key if no translation found
    if (!translation) {
      this.logger.debug(`Translation not found: ${key} (${lang})`);
      return key;
    }

    // Replace parameters
    if (params) {
      return this.interpolate(translation, params);
    }

    return translation;
  }

  /**
   * Alias for translate
   */
  t(key: string, language?: string, params?: Record<string, string | number>): string {
    return this.translate(key, language, params);
  }

  /**
   * Check if language is supported
   */
  isSupported(language: string): boolean {
    return this.supportedLanguages.some((l) => l.code === language);
  }

  /**
   * Get supported languages
   */
  getSupportedLanguages(): SupportedLanguage[] {
    return this.supportedLanguages;
  }

  /**
   * Get default language
   */
  getDefaultLanguage(): string {
    return this.defaultLanguage;
  }

  /**
   * Get all translations for a language
   */
  getTranslations(language: string): TranslationFile | undefined {
    return this.translations.get(language);
  }

  /**
   * Get translations for a specific namespace
   */
  getNamespace(namespace: string, language?: string): TranslationFile | undefined {
    const lang = language || this.defaultLanguage;
    const translations = this.translations.get(lang);
    return translations?.[namespace] as TranslationFile | undefined;
  }

  /**
   * Add or update a translation
   */
  setTranslation(language: string, key: string, value: string): void {
    if (!this.translations.has(language)) {
      this.translations.set(language, {});
    }

    const translations = this.translations.get(language)!;
    this.setNestedValue(translations, key, value);
  }

  /**
   * Get language direction (ltr/rtl)
   */
  getDirection(language: string): 'ltr' | 'rtl' {
    const lang = this.supportedLanguages.find((l) => l.code === language);
    return lang?.direction || 'ltr';
  }

  /**
   * Detect language from Accept-Language header
   */
  detectLanguage(acceptLanguage: string): string {
    if (!acceptLanguage) {
      return this.defaultLanguage;
    }

    // Parse Accept-Language header
    const languages = acceptLanguage
      .split(',')
      .map((lang) => {
        const [code, quality] = lang.trim().split(';q=');
        return {
          code: code.split('-')[0].toLowerCase(),
          quality: quality ? parseFloat(quality) : 1.0,
        };
      })
      .sort((a, b) => b.quality - a.quality);

    // Find first supported language
    for (const lang of languages) {
      if (this.isSupported(lang.code)) {
        return lang.code;
      }
    }

    return this.defaultLanguage;
  }

  /**
   * Get nested value from object using dot notation
   */
  private getNestedValue(obj: TranslationFile, key: string): string | undefined {
    const keys = key.split('.');
    let current: any = obj;

    for (const k of keys) {
      if (current && typeof current === 'object' && k in current) {
        current = current[k];
      } else {
        return undefined;
      }
    }

    return typeof current === 'string' ? current : undefined;
  }

  /**
   * Set nested value in object using dot notation
   */
  private setNestedValue(obj: TranslationFile, key: string, value: string): void {
    const keys = key.split('.');
    let current: any = obj;

    for (let i = 0; i < keys.length - 1; i++) {
      const k = keys[i];
      if (!(k in current) || typeof current[k] !== 'object') {
        current[k] = {};
      }
      current = current[k];
    }

    current[keys[keys.length - 1]] = value;
  }

  /**
   * Replace placeholders in translation string
   */
  private interpolate(
    text: string,
    params: Record<string, string | number>,
  ): string {
    return text.replace(/\{(\w+)\}/g, (match, key) => {
      return params[key]?.toString() ?? match;
    });
  }

  // ============================================================================
  // DEFAULT TRANSLATIONS
  // ============================================================================

  private getDefaultEnglishTranslations(): TranslationFile {
    return {
      common: {
        success: 'Success',
        error: 'Error',
        loading: 'Loading...',
        retry: 'Retry',
        cancel: 'Cancel',
        confirm: 'Confirm',
        save: 'Save',
        delete: 'Delete',
        edit: 'Edit',
        close: 'Close',
        back: 'Back',
        next: 'Next',
        yes: 'Yes',
        no: 'No',
        ok: 'OK',
      },
      auth: {
        login: 'Login',
        logout: 'Logout',
        signup: 'Sign Up',
        forgotPassword: 'Forgot Password?',
        resetPassword: 'Reset Password',
        verifyOTP: 'Verify OTP',
        otpSent: 'OTP sent to {phone}',
        invalidCredentials: 'Invalid credentials',
        accountLocked: 'Account is locked',
        sessionExpired: 'Session expired, please login again',
      },
      booking: {
        newBooking: 'New Booking',
        pickupLocation: 'Pickup Location',
        dropoffLocation: 'Dropoff Location',
        searchingDriver: 'Searching for driver...',
        driverAssigned: 'Driver assigned',
        driverArriving: 'Driver is arriving',
        tripStarted: 'Trip started',
        tripCompleted: 'Trip completed',
        tripCancelled: 'Trip cancelled',
        estimatedFare: 'Estimated Fare',
        actualFare: 'Actual Fare',
        paymentMethod: 'Payment Method',
        rateYourTrip: 'Rate your trip',
        thankYou: 'Thank you for riding with us!',
      },
      driver: {
        online: 'Online',
        offline: 'Offline',
        newRideRequest: 'New ride request',
        acceptRide: 'Accept',
        declineRide: 'Decline',
        arrivedAtPickup: 'Arrived at pickup',
        startTrip: 'Start Trip',
        completeTrip: 'Complete Trip',
        earnings: 'Earnings',
        todayEarnings: "Today's Earnings",
        weeklyEarnings: 'Weekly Earnings',
      },
      payment: {
        cash: 'Cash',
        card: 'Card',
        wallet: 'Wallet',
        paymentSuccessful: 'Payment successful',
        paymentFailed: 'Payment failed',
        insufficientFunds: 'Insufficient funds',
        addMoney: 'Add Money',
        withdraw: 'Withdraw',
      },
      notifications: {
        rideAccepted: 'Your ride has been accepted',
        driverArrived: 'Your driver has arrived',
        tripStarted: 'Your trip has started',
        tripEnded: 'Your trip has ended',
        paymentReceived: 'Payment received',
        promotionAlert: 'New promotion available!',
      },
      errors: {
        networkError: 'Network error. Please check your connection.',
        serverError: 'Server error. Please try again later.',
        notFound: 'Not found',
        unauthorized: 'Unauthorized access',
        validationError: 'Validation error',
        unknownError: 'An unknown error occurred',
      },
    };
  }

  private getDefaultFrenchTranslations(): TranslationFile {
    return {
      common: {
        success: 'Succès',
        error: 'Erreur',
        loading: 'Chargement...',
        retry: 'Réessayer',
        cancel: 'Annuler',
        confirm: 'Confirmer',
        save: 'Enregistrer',
        delete: 'Supprimer',
        edit: 'Modifier',
        close: 'Fermer',
        back: 'Retour',
        next: 'Suivant',
        yes: 'Oui',
        no: 'Non',
        ok: 'OK',
      },
      auth: {
        login: 'Connexion',
        logout: 'Déconnexion',
        signup: "S'inscrire",
        forgotPassword: 'Mot de passe oublié ?',
        resetPassword: 'Réinitialiser le mot de passe',
        verifyOTP: 'Vérifier OTP',
        otpSent: 'OTP envoyé à {phone}',
        invalidCredentials: 'Identifiants invalides',
        accountLocked: 'Compte verrouillé',
        sessionExpired: 'Session expirée, veuillez vous reconnecter',
      },
      booking: {
        newBooking: 'Nouvelle réservation',
        pickupLocation: 'Lieu de prise en charge',
        dropoffLocation: 'Lieu de dépose',
        searchingDriver: 'Recherche de chauffeur...',
        driverAssigned: 'Chauffeur assigné',
        driverArriving: 'Le chauffeur arrive',
        tripStarted: 'Course démarrée',
        tripCompleted: 'Course terminée',
        tripCancelled: 'Course annulée',
        estimatedFare: 'Tarif estimé',
        actualFare: 'Tarif réel',
        paymentMethod: 'Mode de paiement',
        rateYourTrip: 'Notez votre course',
        thankYou: 'Merci de voyager avec nous !',
      },
      driver: {
        online: 'En ligne',
        offline: 'Hors ligne',
        newRideRequest: 'Nouvelle demande de course',
        acceptRide: 'Accepter',
        declineRide: 'Refuser',
        arrivedAtPickup: 'Arrivé au point de prise en charge',
        startTrip: 'Démarrer la course',
        completeTrip: 'Terminer la course',
        earnings: 'Gains',
        todayEarnings: "Gains d'aujourd'hui",
        weeklyEarnings: 'Gains de la semaine',
      },
      payment: {
        cash: 'Espèces',
        card: 'Carte',
        wallet: 'Portefeuille',
        paymentSuccessful: 'Paiement réussi',
        paymentFailed: 'Paiement échoué',
        insufficientFunds: 'Fonds insuffisants',
        addMoney: 'Ajouter des fonds',
        withdraw: 'Retirer',
      },
      notifications: {
        rideAccepted: 'Votre course a été acceptée',
        driverArrived: 'Votre chauffeur est arrivé',
        tripStarted: 'Votre course a commencé',
        tripEnded: 'Votre course est terminée',
        paymentReceived: 'Paiement reçu',
        promotionAlert: 'Nouvelle promotion disponible !',
      },
      errors: {
        networkError: 'Erreur réseau. Vérifiez votre connexion.',
        serverError: 'Erreur serveur. Réessayez plus tard.',
        notFound: 'Non trouvé',
        unauthorized: 'Accès non autorisé',
        validationError: 'Erreur de validation',
        unknownError: 'Une erreur inconnue est survenue',
      },
    };
  }

  private getDefaultArabicTranslations(): TranslationFile {
    return {
      common: {
        success: 'نجاح',
        error: 'خطأ',
        loading: 'جاري التحميل...',
        retry: 'إعادة المحاولة',
        cancel: 'إلغاء',
        confirm: 'تأكيد',
        save: 'حفظ',
        delete: 'حذف',
        edit: 'تعديل',
        close: 'إغلاق',
        back: 'رجوع',
        next: 'التالي',
        yes: 'نعم',
        no: 'لا',
        ok: 'موافق',
      },
      auth: {
        login: 'تسجيل الدخول',
        logout: 'تسجيل الخروج',
        signup: 'إنشاء حساب',
        forgotPassword: 'نسيت كلمة المرور؟',
        resetPassword: 'إعادة تعيين كلمة المرور',
        verifyOTP: 'تحقق من الرمز',
        otpSent: 'تم إرسال الرمز إلى {phone}',
        invalidCredentials: 'بيانات اعتماد غير صالحة',
        accountLocked: 'الحساب مقفل',
        sessionExpired: 'انتهت الجلسة، يرجى تسجيل الدخول مرة أخرى',
      },
      booking: {
        newBooking: 'حجز جديد',
        pickupLocation: 'موقع الالتقاط',
        dropoffLocation: 'موقع التوصيل',
        searchingDriver: 'جاري البحث عن سائق...',
        driverAssigned: 'تم تعيين السائق',
        driverArriving: 'السائق في الطريق',
        tripStarted: 'بدأت الرحلة',
        tripCompleted: 'اكتملت الرحلة',
        tripCancelled: 'تم إلغاء الرحلة',
        estimatedFare: 'السعر التقديري',
        actualFare: 'السعر الفعلي',
        paymentMethod: 'طريقة الدفع',
        rateYourTrip: 'قيم رحلتك',
        thankYou: 'شكرا لاستخدامك خدمتنا!',
      },
      driver: {
        online: 'متصل',
        offline: 'غير متصل',
        newRideRequest: 'طلب رحلة جديد',
        acceptRide: 'قبول',
        declineRide: 'رفض',
        arrivedAtPickup: 'وصل إلى موقع الالتقاط',
        startTrip: 'بدء الرحلة',
        completeTrip: 'إنهاء الرحلة',
        earnings: 'الأرباح',
        todayEarnings: 'أرباح اليوم',
        weeklyEarnings: 'أرباح الأسبوع',
      },
      payment: {
        cash: 'نقدا',
        card: 'بطاقة',
        wallet: 'المحفظة',
        paymentSuccessful: 'تمت عملية الدفع بنجاح',
        paymentFailed: 'فشلت عملية الدفع',
        insufficientFunds: 'رصيد غير كافي',
        addMoney: 'إضافة رصيد',
        withdraw: 'سحب',
      },
      notifications: {
        rideAccepted: 'تم قبول رحلتك',
        driverArrived: 'وصل السائق',
        tripStarted: 'بدأت رحلتك',
        tripEnded: 'انتهت رحلتك',
        paymentReceived: 'تم استلام الدفع',
        promotionAlert: 'عرض جديد متاح!',
      },
      errors: {
        networkError: 'خطأ في الشبكة. تحقق من اتصالك.',
        serverError: 'خطأ في الخادم. حاول مرة أخرى لاحقا.',
        notFound: 'غير موجود',
        unauthorized: 'وصول غير مصرح',
        validationError: 'خطأ في التحقق',
        unknownError: 'حدث خطأ غير معروف',
      },
    };
  }
}
