import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  Headers,
  UseGuards,
  Res,
} from '@nestjs/common';
import { Response } from 'express';
import { I18nService, SupportedLanguage } from './i18n.service';
import { TranslationService, TranslationInput } from './translation.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { Public } from '../../common/decorators/public.decorator';
import { CurrentUser } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class TranslateDto {
  key: string;
  params?: Record<string, string | number>;
}

class BulkTranslateDto {
  keys: string[];
}

class UpsertTranslationDto {
  key: string;
  language: string;
  value: string;
  namespace?: string;
}

class BulkUpsertTranslationsDto {
  translations: TranslationInput[];
}

class ImportTranslationsDto {
  language: string;
  translations: Record<string, string>;
  namespace?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('i18n')
export class I18nController {
  constructor(
    private readonly i18nService: I18nService,
    private readonly translationService: TranslationService,
  ) {}

  // ============================================================================
  // PUBLIC ENDPOINTS
  // ============================================================================

  /**
   * Get supported languages
   */
  @Get('languages')
  @Public()
  getSupportedLanguages(): SupportedLanguage[] {
    return this.i18nService.getSupportedLanguages();
  }

  /**
   * Get default language
   */
  @Get('default')
  @Public()
  getDefaultLanguage() {
    return {
      language: this.i18nService.getDefaultLanguage(),
    };
  }

  /**
   * Detect language from Accept-Language header
   */
  @Get('detect')
  @Public()
  detectLanguage(@Headers('accept-language') acceptLanguage: string) {
    const detected = this.i18nService.detectLanguage(acceptLanguage);
    const direction = this.i18nService.getDirection(detected);

    return {
      language: detected,
      direction,
    };
  }

  /**
   * Get all translations for a language (file-based)
   */
  @Get('file/:language')
  @Public()
  getFileTranslations(@Param('language') language: string) {
    if (!this.i18nService.isSupported(language)) {
      return { error: `Language '${language}' is not supported` };
    }

    return {
      language,
      direction: this.i18nService.getDirection(language),
      translations: this.i18nService.getTranslations(language),
    };
  }

  /**
   * Get translations namespace (file-based)
   */
  @Get('file/:language/:namespace')
  @Public()
  getFileNamespace(
    @Param('language') language: string,
    @Param('namespace') namespace: string,
  ) {
    return {
      language,
      namespace,
      translations: this.i18nService.getNamespace(namespace, language),
    };
  }

  /**
   * Translate a single key
   */
  @Post('translate/:language')
  @Public()
  async translate(
    @Param('language') language: string,
    @Body() dto: TranslateDto,
    @Query('merchantId') merchantId?: string,
  ) {
    const translation = await this.translationService.translate(
      dto.key,
      language,
      dto.params,
      merchantId ? parseInt(merchantId) : undefined,
    );

    return {
      key: dto.key,
      language,
      value: translation,
    };
  }

  /**
   * Translate multiple keys
   */
  @Post('translate/:language/bulk')
  @Public()
  async bulkTranslate(
    @Param('language') language: string,
    @Body() dto: BulkTranslateDto,
    @Query('merchantId') merchantId?: string,
  ) {
    const translations: Record<string, string> = {};
    const mId = merchantId ? parseInt(merchantId) : undefined;

    for (const key of dto.keys) {
      translations[key] = await this.translationService.translate(
        key,
        language,
        undefined,
        mId,
      );
    }

    return {
      language,
      translations,
    };
  }

  // ============================================================================
  // PROTECTED ENDPOINTS (Admin)
  // ============================================================================

  /**
   * Get all translations for a language from database
   */
  @Get('translations/:language')
  @UseGuards(JwtAuthGuard)
  async getTranslations(
    @Param('language') language: string,
    @CurrentUser() userId: number,
    @Query('namespace') namespace?: string,
  ) {
    const translations = await this.translationService.getTranslationsForLanguage(
      language,
      undefined, // TODO: Get merchantId from user
      namespace,
    );

    return {
      language,
      namespace,
      count: Object.keys(translations).length,
      translations,
    };
  }

  /**
   * Create or update translation
   */
  @Post('translations')
  @UseGuards(JwtAuthGuard)
  async upsertTranslation(
    @Body() dto: UpsertTranslationDto,
    @CurrentUser() userId: number,
  ) {
    const translation = await this.translationService.upsertTranslation(
      dto,
      undefined, // TODO: Get merchantId from user
    );

    return {
      success: true,
      translation,
    };
  }

  /**
   * Bulk create or update translations
   */
  @Post('translations/bulk')
  @UseGuards(JwtAuthGuard)
  async bulkUpsertTranslations(
    @Body() dto: BulkUpsertTranslationsDto,
    @CurrentUser() userId: number,
  ) {
    const result = await this.translationService.bulkUpsertTranslations(
      dto.translations,
      undefined, // TODO: Get merchantId from user
    );

    return {
      success: true,
      ...result,
    };
  }

  /**
   * Delete translation
   */
  @Delete('translations/:language/:key')
  @UseGuards(JwtAuthGuard)
  async deleteTranslation(
    @Param('language') language: string,
    @Param('key') key: string,
    @CurrentUser() userId: number,
  ) {
    const deleted = await this.translationService.deleteTranslation(
      key,
      language,
      undefined, // TODO: Get merchantId from user
    );

    return { success: deleted };
  }

  /**
   * Search translations
   */
  @Get('search')
  @UseGuards(JwtAuthGuard)
  async searchTranslations(
    @Query('q') query: string,
    @Query('language') language?: string,
    @CurrentUser() userId?: number,
  ) {
    const results = await this.translationService.searchTranslations(
      query,
      language,
      undefined, // TODO: Get merchantId from user
    );

    return {
      query,
      language,
      count: results.length,
      results,
    };
  }

  /**
   * Get missing translations
   */
  @Get('missing/:source/:target')
  @UseGuards(JwtAuthGuard)
  async getMissingTranslations(
    @Param('source') sourceLanguage: string,
    @Param('target') targetLanguage: string,
    @CurrentUser() userId: number,
  ) {
    const missing = await this.translationService.getMissingTranslations(
      sourceLanguage,
      targetLanguage,
      undefined, // TODO: Get merchantId from user
    );

    return {
      sourceLanguage,
      targetLanguage,
      count: missing.length,
      missingKeys: missing,
    };
  }

  /**
   * Export translations
   */
  @Get('export/:language')
  @UseGuards(JwtAuthGuard)
  async exportTranslations(
    @Param('language') language: string,
    @Query('format') format: 'json' | 'csv' = 'json',
    @Res() res: Response,
    @CurrentUser() userId: number,
  ) {
    const content = await this.translationService.exportTranslations(
      language,
      undefined, // TODO: Get merchantId from user
      format,
    );

    const filename = `translations_${language}.${format}`;
    const contentType =
      format === 'csv' ? 'text/csv' : 'application/json';

    res.setHeader('Content-Type', contentType);
    res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
    res.send(content);
  }

  /**
   * Import translations
   */
  @Post('import')
  @UseGuards(JwtAuthGuard)
  async importTranslations(
    @Body() dto: ImportTranslationsDto,
    @CurrentUser() userId: number,
  ) {
    const result = await this.translationService.importTranslations(
      dto.language,
      dto.translations,
      undefined, // TODO: Get merchantId from user
      dto.namespace,
    );

    return {
      success: true,
      language: dto.language,
      ...result,
    };
  }

  /**
   * Preload translation cache
   */
  @Post('cache/preload')
  @UseGuards(JwtAuthGuard)
  async preloadCache(
    @Body() body: { languages?: string[] },
    @CurrentUser() userId: number,
  ) {
    await this.translationService.preloadCache(
      body.languages,
      undefined, // TODO: Get merchantId from user
    );

    return { success: true, message: 'Cache preloaded' };
  }

  /**
   * Clear translation cache
   */
  @Post('cache/clear')
  @UseGuards(JwtAuthGuard)
  clearCache(@CurrentUser() userId: number) {
    this.translationService.clearCache();
    return { success: true, message: 'Cache cleared' };
  }
}
