import { Injectable } from '@nestjs/common';
import {
  HealthIndicator,
  HealthIndicatorResult,
  HealthCheckError,
} from '@nestjs/terminus';
import { ConfigService } from '@nestjs/config';
import { S3Client, HeadBucketCommand } from '@aws-sdk/client-s3';
import * as fs from 'fs';
import * as path from 'path';

@Injectable()
export class StorageHealthIndicator extends HealthIndicator {
  private s3Client: S3Client;
  private bucket: string;
  private localStoragePath: string;

  constructor(private configService: ConfigService) {
    super();

    this.s3Client = new S3Client({
      region: this.configService.get<string>('AWS_REGION', 'us-east-1'),
      credentials: {
        accessKeyId: this.configService.get<string>('AWS_ACCESS_KEY_ID', ''),
        secretAccessKey: this.configService.get<string>('AWS_SECRET_ACCESS_KEY', ''),
      },
    });

    this.bucket = this.configService.get<string>('AWS_S3_BUCKET', '');
    this.localStoragePath = this.configService.get<string>(
      'LOCAL_STORAGE_PATH',
      './storage',
    );
  }

  async isHealthy(key: string): Promise<HealthIndicatorResult> {
    const results: Record<string, any> = {};

    // Check local storage
    try {
      const localHealth = await this.checkLocalStorage();
      results.localStorage = localHealth;
    } catch (error) {
      results.localStorage = { status: 'error', message: error.message };
    }

    // Check S3 storage
    try {
      const s3Health = await this.checkS3Storage();
      results.s3Storage = s3Health;
    } catch (error) {
      results.s3Storage = { status: 'error', message: error.message };
    }

    const allHealthy =
      results.localStorage?.status === 'ok' ||
      results.s3Storage?.status === 'ok';

    if (!allHealthy) {
      const result = this.getStatus(key, false, results);
      throw new HealthCheckError('Storage check failed', result);
    }

    return this.getStatus(key, true, results);
  }

  private async checkLocalStorage(): Promise<{
    status: string;
    path?: string;
    writable?: boolean;
    freeSpace?: string;
  }> {
    const storagePath = path.resolve(this.localStoragePath);

    // Check if directory exists
    if (!fs.existsSync(storagePath)) {
      try {
        fs.mkdirSync(storagePath, { recursive: true });
      } catch {
        return { status: 'error', path: storagePath, writable: false };
      }
    }

    // Check if writable
    const testFile = path.join(storagePath, '.health-check');
    try {
      fs.writeFileSync(testFile, 'health check');
      fs.unlinkSync(testFile);
    } catch {
      return { status: 'error', path: storagePath, writable: false };
    }

    // Get free space (platform dependent)
    let freeSpace = 'unknown';
    try {
      const stats = fs.statfsSync(storagePath);
      const freeBytes = stats.bfree * stats.bsize;
      freeSpace = this.formatBytes(freeBytes);
    } catch {
      // statfsSync not available on all platforms
    }

    return {
      status: 'ok',
      path: storagePath,
      writable: true,
      freeSpace,
    };
  }

  private async checkS3Storage(): Promise<{
    status: string;
    bucket?: string;
    accessible?: boolean;
    responseTime?: string;
  }> {
    if (!this.bucket) {
      return { status: 'not_configured' };
    }

    const startTime = Date.now();

    try {
      await this.s3Client.send(
        new HeadBucketCommand({ Bucket: this.bucket }),
      );

      const responseTime = Date.now() - startTime;

      return {
        status: 'ok',
        bucket: this.bucket,
        accessible: true,
        responseTime: `${responseTime}ms`,
      };
    } catch (error) {
      return {
        status: 'error',
        bucket: this.bucket,
        accessible: false,
      };
    }
  }

  async getStorageStats(): Promise<{
    local: {
      path: string;
      totalFiles: number;
      totalSize: string;
      freeSpace: string;
    };
    s3: {
      bucket: string;
      configured: boolean;
    };
  }> {
    const storagePath = path.resolve(this.localStoragePath);
    let totalFiles = 0;
    let totalSize = 0;

    if (fs.existsSync(storagePath)) {
      const files = this.getFilesRecursively(storagePath);
      totalFiles = files.length;
      totalSize = files.reduce((acc, file) => {
        try {
          return acc + fs.statSync(file).size;
        } catch {
          return acc;
        }
      }, 0);
    }

    let freeSpace = 'unknown';
    try {
      const stats = fs.statfsSync(storagePath);
      freeSpace = this.formatBytes(stats.bfree * stats.bsize);
    } catch {
      // Not available on all platforms
    }

    return {
      local: {
        path: storagePath,
        totalFiles,
        totalSize: this.formatBytes(totalSize),
        freeSpace,
      },
      s3: {
        bucket: this.bucket || 'not_configured',
        configured: !!this.bucket,
      },
    };
  }

  private getFilesRecursively(dir: string): string[] {
    const files: string[] = [];

    try {
      const entries = fs.readdirSync(dir, { withFileTypes: true });
      for (const entry of entries) {
        const fullPath = path.join(dir, entry.name);
        if (entry.isDirectory()) {
          files.push(...this.getFilesRecursively(fullPath));
        } else {
          files.push(fullPath);
        }
      }
    } catch {
      // Ignore permission errors
    }

    return files;
  }

  private formatBytes(bytes: number): string {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
  }
}
