import { Injectable } from '@nestjs/common';
import {
  HealthIndicator,
  HealthIndicatorResult,
  HealthCheckError,
} from '@nestjs/terminus';
import { ConfigService } from '@nestjs/config';
import Redis from 'ioredis';

@Injectable()
export class RedisHealthIndicator extends HealthIndicator {
  private redis: Redis;

  constructor(private configService: ConfigService) {
    super();
    this.redis = new Redis({
      host: this.configService.get<string>('REDIS_HOST', 'localhost'),
      port: this.configService.get<number>('REDIS_PORT', 6379),
      password: this.configService.get<string>('REDIS_PASSWORD'),
      db: this.configService.get<number>('REDIS_DB', 0),
      lazyConnect: true,
    });
  }

  async isHealthy(key: string): Promise<HealthIndicatorResult> {
    const startTime = Date.now();

    try {
      await this.redis.ping();
      const responseTime = Date.now() - startTime;

      const info = await this.getRedisInfo();

      return this.getStatus(key, true, {
        responseTime: `${responseTime}ms`,
        status: 'connected',
        ...info,
      });
    } catch (error) {
      const result = this.getStatus(key, false, {
        message: error.message,
        status: 'disconnected',
      });
      throw new HealthCheckError('Redis check failed', result);
    }
  }

  private async getRedisInfo(): Promise<Record<string, any>> {
    try {
      const info = await this.redis.info('memory');
      const keyspace = await this.redis.info('keyspace');

      const memoryMatch = info.match(/used_memory_human:(\S+)/);
      const peakMemoryMatch = info.match(/used_memory_peak_human:(\S+)/);
      const dbMatch = keyspace.match(/db0:keys=(\d+)/);

      return {
        usedMemory: memoryMatch ? memoryMatch[1] : 'unknown',
        peakMemory: peakMemoryMatch ? peakMemoryMatch[1] : 'unknown',
        keys: dbMatch ? parseInt(dbMatch[1]) : 0,
      };
    } catch {
      return {};
    }
  }

  async getDetailedInfo(): Promise<{
    connected: boolean;
    memory: {
      used: string;
      peak: string;
      fragmentation: number;
    };
    clients: {
      connected: number;
      blocked: number;
    };
    stats: {
      totalConnections: number;
      totalCommands: number;
      opsPerSec: number;
    };
    replication: {
      role: string;
      connectedSlaves: number;
    };
  }> {
    try {
      const [serverInfo, memoryInfo, clientsInfo, statsInfo, replicationInfo] =
        await Promise.all([
          this.redis.info('server'),
          this.redis.info('memory'),
          this.redis.info('clients'),
          this.redis.info('stats'),
          this.redis.info('replication'),
        ]);

      return {
        connected: true,
        memory: {
          used: this.extractValue(memoryInfo, 'used_memory_human') || '0',
          peak: this.extractValue(memoryInfo, 'used_memory_peak_human') || '0',
          fragmentation: parseFloat(
            this.extractValue(memoryInfo, 'mem_fragmentation_ratio') || '0',
          ),
        },
        clients: {
          connected: parseInt(
            this.extractValue(clientsInfo, 'connected_clients') || '0',
          ),
          blocked: parseInt(
            this.extractValue(clientsInfo, 'blocked_clients') || '0',
          ),
        },
        stats: {
          totalConnections: parseInt(
            this.extractValue(statsInfo, 'total_connections_received') || '0',
          ),
          totalCommands: parseInt(
            this.extractValue(statsInfo, 'total_commands_processed') || '0',
          ),
          opsPerSec: parseInt(
            this.extractValue(statsInfo, 'instantaneous_ops_per_sec') || '0',
          ),
        },
        replication: {
          role: this.extractValue(replicationInfo, 'role') || 'master',
          connectedSlaves: parseInt(
            this.extractValue(replicationInfo, 'connected_slaves') || '0',
          ),
        },
      };
    } catch {
      return {
        connected: false,
        memory: { used: '0', peak: '0', fragmentation: 0 },
        clients: { connected: 0, blocked: 0 },
        stats: { totalConnections: 0, totalCommands: 0, opsPerSec: 0 },
        replication: { role: 'unknown', connectedSlaves: 0 },
      };
    }
  }

  private extractValue(info: string, key: string): string | null {
    const regex = new RegExp(`${key}:(.+)`);
    const match = info.match(regex);
    return match ? match[1].trim() : null;
  }
}
