import { Injectable } from '@nestjs/common';
import {
  HealthIndicator,
  HealthIndicatorResult,
  HealthCheckError,
} from '@nestjs/terminus';
import { ConfigService } from '@nestjs/config';
import { HttpService } from '@nestjs/axios';
import { firstValueFrom, timeout, catchError } from 'rxjs';
import { of } from 'rxjs';

interface ServiceHealth {
  name: string;
  status: 'ok' | 'degraded' | 'down' | 'not_configured';
  responseTime?: string;
  message?: string;
}

@Injectable()
export class ExternalServicesHealthIndicator extends HealthIndicator {
  private readonly services: Array<{
    name: string;
    url: string;
    configKey: string;
    timeout: number;
  }>;

  constructor(
    private configService: ConfigService,
    private httpService: HttpService,
  ) {
    super();

    this.services = [
      {
        name: 'google_maps',
        url: 'https://maps.googleapis.com/maps/api/geocode/json?address=test&key=',
        configKey: 'GOOGLE_MAPS_API_KEY',
        timeout: 5000,
      },
      {
        name: 'stripe',
        url: 'https://api.stripe.com/v1/balance',
        configKey: 'STRIPE_SECRET_KEY',
        timeout: 5000,
      },
      {
        name: 'firebase',
        url: 'https://fcm.googleapis.com/fcm/send',
        configKey: 'FIREBASE_SERVER_KEY',
        timeout: 5000,
      },
      {
        name: 'twilio',
        url: 'https://api.twilio.com/2010-04-01.json',
        configKey: 'TWILIO_ACCOUNT_SID',
        timeout: 5000,
      },
      {
        name: 'sendgrid',
        url: 'https://api.sendgrid.com/v3/user/credits',
        configKey: 'SENDGRID_API_KEY',
        timeout: 5000,
      },
    ];
  }

  async isHealthy(key: string): Promise<HealthIndicatorResult> {
    const results = await this.checkAllServices();

    const allHealthy = results.every(
      (r) => r.status === 'ok' || r.status === 'not_configured',
    );
    const anyDown = results.some((r) => r.status === 'down');

    const servicesMap = results.reduce(
      (acc, service) => {
        acc[service.name] = {
          status: service.status,
          responseTime: service.responseTime,
          message: service.message,
        };
        return acc;
      },
      {} as Record<string, any>,
    );

    if (anyDown) {
      const result = this.getStatus(key, false, servicesMap);
      throw new HealthCheckError('External services check failed', result);
    }

    return this.getStatus(key, allHealthy, servicesMap);
  }

  async checkAllServices(): Promise<ServiceHealth[]> {
    const results = await Promise.all(
      this.services.map((service) => this.checkService(service)),
    );
    return results;
  }

  async checkService(service: {
    name: string;
    url: string;
    configKey: string;
    timeout: number;
  }): Promise<ServiceHealth> {
    const apiKey = this.configService.get<string>(service.configKey);

    if (!apiKey) {
      return {
        name: service.name,
        status: 'not_configured',
      };
    }

    const startTime = Date.now();

    try {
      let url = service.url;
      let headers: Record<string, string> = {};

      // Configure authentication based on service
      switch (service.name) {
        case 'google_maps':
          url += apiKey;
          break;
        case 'stripe':
          headers['Authorization'] = `Bearer ${apiKey}`;
          break;
        case 'firebase':
          headers['Authorization'] = `key=${apiKey}`;
          break;
        case 'twilio':
          const authToken = this.configService.get<string>('TWILIO_AUTH_TOKEN');
          headers['Authorization'] =
            'Basic ' +
            Buffer.from(`${apiKey}:${authToken}`).toString('base64');
          break;
        case 'sendgrid':
          headers['Authorization'] = `Bearer ${apiKey}`;
          break;
      }

      const response = await firstValueFrom(
        this.httpService.get(url, { headers, timeout: service.timeout }).pipe(
          timeout(service.timeout),
          catchError(() => of({ status: 0 })),
        ),
      );

      const responseTime = Date.now() - startTime;

      // Check for valid response
      if (response.status >= 200 && response.status < 500) {
        return {
          name: service.name,
          status: 'ok',
          responseTime: `${responseTime}ms`,
        };
      }

      return {
        name: service.name,
        status: 'degraded',
        responseTime: `${responseTime}ms`,
        message: `HTTP ${response.status}`,
      };
    } catch (error) {
      return {
        name: service.name,
        status: 'down',
        message: error.message,
      };
    }
  }

  async checkGoogleMaps(): Promise<ServiceHealth> {
    return this.checkService(this.services.find((s) => s.name === 'google_maps')!);
  }

  async checkStripe(): Promise<ServiceHealth> {
    return this.checkService(this.services.find((s) => s.name === 'stripe')!);
  }

  async checkFirebase(): Promise<ServiceHealth> {
    return this.checkService(this.services.find((s) => s.name === 'firebase')!);
  }

  async checkTwilio(): Promise<ServiceHealth> {
    return this.checkService(this.services.find((s) => s.name === 'twilio')!);
  }

  async checkSendGrid(): Promise<ServiceHealth> {
    return this.checkService(this.services.find((s) => s.name === 'sendgrid')!);
  }
}
