import { Injectable } from '@nestjs/common';
import {
  HealthIndicator,
  HealthIndicatorResult,
  HealthCheckError,
} from '@nestjs/terminus';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class DatabaseHealthIndicator extends HealthIndicator {
  constructor(private prisma: PrismaService) {
    super();
  }

  async isHealthy(key: string): Promise<HealthIndicatorResult> {
    const startTime = Date.now();

    try {
      // Simple query to check database connectivity
      await this.prisma.$queryRaw`SELECT 1`;

      const responseTime = Date.now() - startTime;

      return this.getStatus(key, true, {
        responseTime: `${responseTime}ms`,
        status: 'connected',
      });
    } catch (error) {
      const result = this.getStatus(key, false, {
        message: error.message,
        status: 'disconnected',
      });
      throw new HealthCheckError('Database check failed', result);
    }
  }

  async getConnectionPool(): Promise<{
    active: number;
    idle: number;
    waiting: number;
  }> {
    try {
      // Prisma doesn't expose connection pool directly
      // This would need custom implementation with raw SQL
      const result = await this.prisma.$queryRaw<any[]>`
        SHOW STATUS LIKE 'Threads_connected'
      `;

      return {
        active: parseInt(result[0]?.Value || '0'),
        idle: 0,
        waiting: 0,
      };
    } catch {
      return { active: 0, idle: 0, waiting: 0 };
    }
  }

  async getReplicationStatus(): Promise<{
    isMaster: boolean;
    slaveCount?: number;
    replicationLag?: number;
  }> {
    try {
      const result = await this.prisma.$queryRaw<any[]>`
        SHOW SLAVE STATUS
      `;

      if (result.length === 0) {
        return { isMaster: true };
      }

      return {
        isMaster: false,
        replicationLag: parseInt(result[0]?.Seconds_Behind_Master || '0'),
      };
    } catch {
      return { isMaster: true };
    }
  }
}
