import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import {
  HealthCheckService,
  HealthCheckResult,
  MemoryHealthIndicator,
  DiskHealthIndicator,
} from '@nestjs/terminus';
import { DatabaseHealthIndicator } from './indicators/database.indicator';
import { RedisHealthIndicator } from './indicators/redis.indicator';
import { StorageHealthIndicator } from './indicators/storage.indicator';
import { ExternalServicesHealthIndicator } from './indicators/external-services.indicator';
import * as os from 'os';

export interface SystemInfo {
  hostname: string;
  platform: string;
  arch: string;
  nodeVersion: string;
  uptime: number;
  loadAverage: number[];
  cpuCount: number;
  totalMemory: string;
  freeMemory: string;
  usedMemory: string;
  memoryUsagePercent: number;
}

export interface ApplicationInfo {
  name: string;
  version: string;
  environment: string;
  startTime: Date;
  uptime: number;
  pid: number;
}

@Injectable()
export class HealthService {
  private readonly logger = new Logger(HealthService.name);
  private readonly startTime: Date;

  constructor(
    private health: HealthCheckService,
    private memory: MemoryHealthIndicator,
    private disk: DiskHealthIndicator,
    private database: DatabaseHealthIndicator,
    private redis: RedisHealthIndicator,
    private storage: StorageHealthIndicator,
    private externalServices: ExternalServicesHealthIndicator,
    private configService: ConfigService,
  ) {
    this.startTime = new Date();
  }

  /**
   * Basic health check (for load balancers)
   */
  async check(): Promise<HealthCheckResult> {
    return this.health.check([
      () => this.database.isHealthy('database'),
    ]);
  }

  /**
   * Readiness check (is app ready to serve traffic)
   */
  async readiness(): Promise<HealthCheckResult> {
    return this.health.check([
      () => this.database.isHealthy('database'),
      () => this.redis.isHealthy('redis'),
    ]);
  }

  /**
   * Liveness check (is app alive)
   */
  async liveness(): Promise<HealthCheckResult> {
    return this.health.check([
      () => this.memory.checkHeap('memory_heap', 500 * 1024 * 1024), // 500MB
      () => this.memory.checkRSS('memory_rss', 1024 * 1024 * 1024), // 1GB
    ]);
  }

  /**
   * Full health check (all services)
   */
  async full(): Promise<HealthCheckResult> {
    return this.health.check([
      () => this.database.isHealthy('database'),
      () => this.redis.isHealthy('redis'),
      () => this.storage.isHealthy('storage'),
      () => this.memory.checkHeap('memory_heap', 500 * 1024 * 1024),
      () => this.memory.checkRSS('memory_rss', 1024 * 1024 * 1024),
      () =>
        this.disk.checkStorage('disk', {
          path: '/',
          thresholdPercent: 0.9,
        }),
    ]);
  }

  /**
   * Check external services
   */
  async externalServicesHealth(): Promise<HealthCheckResult> {
    return this.health.check([
      () => this.externalServices.isHealthy('external_services'),
    ]);
  }

  /**
   * Get system information
   */
  getSystemInfo(): SystemInfo {
    const totalMemory = os.totalmem();
    const freeMemory = os.freemem();
    const usedMemory = totalMemory - freeMemory;

    return {
      hostname: os.hostname(),
      platform: os.platform(),
      arch: os.arch(),
      nodeVersion: process.version,
      uptime: os.uptime(),
      loadAverage: os.loadavg(),
      cpuCount: os.cpus().length,
      totalMemory: this.formatBytes(totalMemory),
      freeMemory: this.formatBytes(freeMemory),
      usedMemory: this.formatBytes(usedMemory),
      memoryUsagePercent: Math.round((usedMemory / totalMemory) * 100),
    };
  }

  /**
   * Get application information
   */
  getApplicationInfo(): ApplicationInfo {
    return {
      name: this.configService.get<string>('APP_NAME', 'MonkAPI'),
      version: this.configService.get<string>('APP_VERSION', '1.0.0'),
      environment: this.configService.get<string>('NODE_ENV', 'development'),
      startTime: this.startTime,
      uptime: Math.floor((Date.now() - this.startTime.getTime()) / 1000),
      pid: process.pid,
    };
  }

  /**
   * Get process metrics
   */
  getProcessMetrics(): {
    memoryUsage: NodeJS.MemoryUsage;
    cpuUsage: NodeJS.CpuUsage;
    uptime: number;
    activeHandles: number;
    activeRequests: number;
  } {
    return {
      memoryUsage: process.memoryUsage(),
      cpuUsage: process.cpuUsage(),
      uptime: process.uptime(),
      activeHandles: (process as any)._getActiveHandles?.()?.length || 0,
      activeRequests: (process as any)._getActiveRequests?.()?.length || 0,
    };
  }

  /**
   * Get database health details
   */
  async getDatabaseHealth(): Promise<{
    status: string;
    connectionPool: { active: number; idle: number; waiting: number };
    replication: { isMaster: boolean; slaveCount?: number; replicationLag?: number };
  }> {
    try {
      await this.database.isHealthy('database');
      const connectionPool = await this.database.getConnectionPool();
      const replication = await this.database.getReplicationStatus();

      return {
        status: 'healthy',
        connectionPool,
        replication,
      };
    } catch (error) {
      return {
        status: 'unhealthy',
        connectionPool: { active: 0, idle: 0, waiting: 0 },
        replication: { isMaster: true },
      };
    }
  }

  /**
   * Get Redis health details
   */
  async getRedisHealth() {
    return this.redis.getDetailedInfo();
  }

  /**
   * Get storage health details
   */
  async getStorageHealth() {
    return this.storage.getStorageStats();
  }

  /**
   * Get all external services health
   */
  async getExternalServicesHealth() {
    return this.externalServices.checkAllServices();
  }

  /**
   * Get comprehensive health report
   */
  async getFullHealthReport(): Promise<{
    status: 'healthy' | 'degraded' | 'unhealthy';
    timestamp: string;
    application: ApplicationInfo;
    system: SystemInfo;
    process: ReturnType<typeof this.getProcessMetrics>;
    services: {
      database: Awaited<ReturnType<typeof this.getDatabaseHealth>>;
      redis: Awaited<ReturnType<typeof this.getRedisHealth>>;
      storage: Awaited<ReturnType<typeof this.getStorageHealth>>;
      external: Awaited<ReturnType<typeof this.getExternalServicesHealth>>;
    };
  }> {
    const [database, redis, storage, external] = await Promise.all([
      this.getDatabaseHealth(),
      this.getRedisHealth(),
      this.getStorageHealth(),
      this.getExternalServicesHealth(),
    ]);

    // Determine overall status
    let status: 'healthy' | 'degraded' | 'unhealthy' = 'healthy';
    if (database.status === 'unhealthy' || !redis.connected) {
      status = 'unhealthy';
    } else if (external.some((e) => e.status === 'down')) {
      status = 'degraded';
    }

    return {
      status,
      timestamp: new Date().toISOString(),
      application: this.getApplicationInfo(),
      system: this.getSystemInfo(),
      process: this.getProcessMetrics(),
      services: {
        database,
        redis,
        storage,
        external,
      },
    };
  }

  private formatBytes(bytes: number): string {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return `${parseFloat((bytes / Math.pow(k, i)).toFixed(2))} ${sizes[i]}`;
  }
}
