import { Controller, Get, UseGuards } from '@nestjs/common';
import { HealthCheck, HealthCheckResult } from '@nestjs/terminus';
import { HealthService } from './health.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { Public } from '../../common/decorators/public.decorator';

@Controller('health')
export class HealthController {
  constructor(private readonly healthService: HealthService) {}

  // ============================================================================
  // BASIC HEALTH CHECKS (public for load balancers)
  // ============================================================================

  /**
   * Basic health check
   * Used by load balancers and Kubernetes probes
   */
  @Get()
  @Public()
  @HealthCheck()
  async check(): Promise<HealthCheckResult> {
    return this.healthService.check();
  }

  /**
   * Readiness check
   * Is the application ready to receive traffic?
   */
  @Get('ready')
  @Public()
  @HealthCheck()
  async readiness(): Promise<HealthCheckResult> {
    return this.healthService.readiness();
  }

  /**
   * Liveness check
   * Is the application alive and not deadlocked?
   */
  @Get('live')
  @Public()
  @HealthCheck()
  async liveness(): Promise<HealthCheckResult> {
    return this.healthService.liveness();
  }

  /**
   * Ping endpoint
   * Simple ping/pong for basic connectivity check
   */
  @Get('ping')
  @Public()
  ping() {
    return {
      message: 'pong',
      timestamp: new Date().toISOString(),
    };
  }

  // ============================================================================
  // DETAILED HEALTH CHECKS (protected)
  // ============================================================================

  /**
   * Full health check
   * Checks all internal services
   */
  @Get('full')
  @UseGuards(JwtAuthGuard)
  @HealthCheck()
  async full(): Promise<HealthCheckResult> {
    return this.healthService.full();
  }

  /**
   * External services health
   * Checks connectivity to external APIs
   */
  @Get('external')
  @UseGuards(JwtAuthGuard)
  @HealthCheck()
  async externalServices(): Promise<HealthCheckResult> {
    return this.healthService.externalServicesHealth();
  }

  // ============================================================================
  // SYSTEM INFORMATION
  // ============================================================================

  /**
   * Get system information
   */
  @Get('system')
  @UseGuards(JwtAuthGuard)
  getSystemInfo() {
    return this.healthService.getSystemInfo();
  }

  /**
   * Get application information
   */
  @Get('app')
  @Public()
  getApplicationInfo() {
    return this.healthService.getApplicationInfo();
  }

  /**
   * Get process metrics
   */
  @Get('process')
  @UseGuards(JwtAuthGuard)
  getProcessMetrics() {
    return this.healthService.getProcessMetrics();
  }

  // ============================================================================
  // SERVICE-SPECIFIC HEALTH
  // ============================================================================

  /**
   * Database health details
   */
  @Get('database')
  @UseGuards(JwtAuthGuard)
  async getDatabaseHealth() {
    return this.healthService.getDatabaseHealth();
  }

  /**
   * Redis health details
   */
  @Get('redis')
  @UseGuards(JwtAuthGuard)
  async getRedisHealth() {
    return this.healthService.getRedisHealth();
  }

  /**
   * Storage health details
   */
  @Get('storage')
  @UseGuards(JwtAuthGuard)
  async getStorageHealth() {
    return this.healthService.getStorageHealth();
  }

  /**
   * External services health details
   */
  @Get('services')
  @UseGuards(JwtAuthGuard)
  async getExternalServicesHealth() {
    return this.healthService.getExternalServicesHealth();
  }

  // ============================================================================
  // COMPREHENSIVE REPORT
  // ============================================================================

  /**
   * Get full health report
   * Comprehensive health status of all services
   */
  @Get('report')
  @UseGuards(JwtAuthGuard)
  async getFullHealthReport() {
    return this.healthService.getFullHealthReport();
  }
}
