import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';
import { InjectQueue } from '@nestjs/bullmq';
import { Queue } from 'bullmq';

export enum ServiceCategory {
  PLUMBING = 'plumbing',
  ELECTRICAL = 'electrical',
  CLEANING = 'cleaning',
  PAINTING = 'painting',
  CARPENTRY = 'carpentry',
  AC_REPAIR = 'ac_repair',
  APPLIANCE_REPAIR = 'appliance_repair',
  PEST_CONTROL = 'pest_control',
  GARDENING = 'gardening',
  MOVING = 'moving',
  OTHER = 'other',
}

export enum BookingStatus {
  PENDING = 'pending',
  ACCEPTED = 'accepted',
  PROVIDER_ASSIGNED = 'provider_assigned',
  EN_ROUTE = 'en_route',
  ARRIVED = 'arrived',
  IN_PROGRESS = 'in_progress',
  COMPLETED = 'completed',
  CANCELLED = 'cancelled',
}

@Injectable()
export class HandymanService {
  private readonly logger = new Logger(HandymanService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
    @InjectQueue('booking') private bookingQueue: Queue,
  ) {}

  // ============================================================================
  // SERVICE CATEGORIES & SERVICES
  // ============================================================================

  /**
   * Get available service categories
   */
  async getServiceCategories(merchantId: number): Promise<any[]> {
    return this.prisma.handymanCategory.findMany({
      where: { merchant_id: merchantId, is_active: 1 },
      orderBy: { sort_order: 'asc' },
    });
  }

  /**
   * Get services by category
   */
  async getServicesByCategory(
    merchantId: number,
    categoryId: number,
  ): Promise<any[]> {
    return this.prisma.handymanService.findMany({
      where: {
        merchant_id: merchantId,
        category_id: categoryId,
        is_active: 1,
      },
      include: {
        category: { select: { name: true, icon: true } },
      },
      orderBy: { name: 'asc' },
    });
  }

  /**
   * Get all services
   */
  async getAllServices(
    merchantId: number,
    options?: { categoryId?: number; search?: string },
  ): Promise<any[]> {
    const where: any = {
      merchant_id: merchantId,
      is_active: 1,
    };

    if (options?.categoryId) {
      where.category_id = options.categoryId;
    }

    if (options?.search) {
      where.OR = [
        { name: { contains: options.search } },
        { description: { contains: options.search } },
      ];
    }

    return this.prisma.handymanService.findMany({
      where,
      include: {
        category: { select: { name: true, icon: true } },
      },
      orderBy: { name: 'asc' },
    });
  }

  /**
   * Get service details
   */
  async getServiceDetails(serviceId: number): Promise<any> {
    const service = await this.prisma.handymanService.findUnique({
      where: { id: serviceId },
      include: {
        category: true,
      },
    });

    if (!service) {
      throw new NotFoundException('Service non trouve');
    }

    return service;
  }

  // ============================================================================
  // BOOKING MANAGEMENT
  // ============================================================================

  /**
   * Create handyman booking
   */
  async createBooking(
    userId: number,
    merchantId: number,
    data: {
      serviceId: number;
      scheduledDate: Date;
      scheduledTime: string;
      address: string;
      latitude: number;
      longitude: number;
      description?: string;
      images?: string[];
    },
  ): Promise<any> {
    const service = await this.prisma.handymanService.findUnique({
      where: { id: data.serviceId },
    });

    if (!service || service.is_active !== 1) {
      throw new BadRequestException('Service non disponible');
    }

    // Calculate estimated price
    const estimatedPrice = await this.calculateEstimate(
      data.serviceId,
      data.description,
    );

    const booking = await this.prisma.handymanBooking.create({
      data: {
        user_id: userId,
        merchant_id: merchantId,
        service_id: data.serviceId,
        category_id: service.category_id,
        scheduled_date: data.scheduledDate,
        scheduled_time: data.scheduledTime,
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        description: data.description,
        images: data.images ? JSON.stringify(data.images) : null,
        estimated_price: estimatedPrice,
        status: BookingStatus.PENDING,
        booking_number: this.generateBookingNumber(),
        created_at: new Date(),
      },
    });

    // Add to queue for provider assignment
    await this.bookingQueue.add('assign-handyman-provider', {
      bookingId: booking.id,
      merchantId,
      latitude: data.latitude,
      longitude: data.longitude,
      categoryId: service.category_id,
    });

    this.logger.log(`Handyman booking created: #${booking.booking_number}`);

    return this.getBookingDetails(booking.id);
  }

  /**
   * Get booking details
   */
  async getBookingDetails(bookingId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findUnique({
      where: { id: bookingId },
      include: {
        service: true,
        category: true,
        provider: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            mobile: true,
            profile_image: true,
            rating: true,
          },
        },
        user: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            mobile: true,
          },
        },
      },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    return {
      ...booking,
      images: booking.images ? JSON.parse(booking.images as string) : [],
    };
  }

  /**
   * Get user bookings
   */
  async getUserBookings(
    userId: number,
    options?: { status?: BookingStatus; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { user_id: userId };
    if (options?.status) {
      where.status = options.status;
    }

    const [bookings, total] = await Promise.all([
      this.prisma.handymanBooking.findMany({
        where,
        include: {
          service: { select: { name: true, icon: true } },
          category: { select: { name: true } },
          provider: {
            select: {
              first_name: true,
              last_name: true,
              profile_image: true,
            },
          },
        },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.handymanBooking.count({ where }),
    ]);

    return { data: bookings, total };
  }

  /**
   * Cancel booking
   */
  async cancelBooking(
    bookingId: number,
    userId: number,
    reason?: string,
  ): Promise<void> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, user_id: userId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    const cancellableStatuses = [
      BookingStatus.PENDING,
      BookingStatus.ACCEPTED,
      BookingStatus.PROVIDER_ASSIGNED,
    ];

    if (!cancellableStatuses.includes(booking.status as BookingStatus)) {
      throw new BadRequestException('Cette reservation ne peut pas etre annulee');
    }

    await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.CANCELLED,
        cancellation_reason: reason,
        cancelled_at: new Date(),
        cancelled_by: 'user',
      },
    });

    // Notify provider if assigned
    if (booking.provider_id) {
      await this.notificationService.sendToUser(booking.provider_id, 'provider', {
        title: 'Reservation annulee',
        body: `La reservation #${booking.booking_number} a ete annulee par le client`,
        data: { type: 'handyman_booking_cancelled', booking_id: String(bookingId) },
      });
    }
  }

  // ============================================================================
  // PROVIDER MANAGEMENT
  // ============================================================================

  /**
   * Find available providers
   */
  async findAvailableProviders(
    merchantId: number,
    latitude: number,
    longitude: number,
    categoryId: number,
    radiusKm: number = 10,
  ): Promise<any[]> {
    // Get providers with matching category and within radius
    const providers = await this.prisma.$queryRaw<any[]>`
      SELECT
        p.*,
        (6371 * acos(cos(radians(${latitude})) * cos(radians(p.latitude)) *
        cos(radians(p.longitude) - radians(${longitude})) +
        sin(radians(${latitude})) * sin(radians(p.latitude)))) AS distance
      FROM handyman_providers p
      INNER JOIN handyman_provider_categories pc ON p.id = pc.provider_id
      WHERE p.merchant_id = ${merchantId}
        AND p.is_active = 1
        AND p.is_online = 1
        AND pc.category_id = ${categoryId}
      HAVING distance <= ${radiusKm}
      ORDER BY p.rating DESC, distance ASC
      LIMIT 10
    `;

    return providers;
  }

  /**
   * Assign provider to booking
   */
  async assignProvider(bookingId: number, providerId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findUnique({
      where: { id: bookingId },
      include: { user: true, service: true },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    const updated = await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        provider_id: providerId,
        status: BookingStatus.PROVIDER_ASSIGNED,
        assigned_at: new Date(),
      },
    });

    // Notify user
    const provider = await this.prisma.handymanProvider.findUnique({
      where: { id: providerId },
    });

    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Prestataire assigne',
      body: `${provider.first_name} va s'occuper de votre demande`,
      data: { type: 'handyman_provider_assigned', booking_id: String(bookingId) },
    });

    // Notify provider
    await this.notificationService.sendToUser(providerId, 'provider', {
      title: 'Nouvelle reservation',
      body: `Vous avez ete assigne a la reservation #${booking.booking_number}`,
      data: { type: 'handyman_booking_assigned', booking_id: String(bookingId) },
    });

    return updated;
  }

  /**
   * Provider accepts booking
   */
  async providerAcceptBooking(bookingId: number, providerId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, provider_id: providerId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    if (booking.status !== BookingStatus.PROVIDER_ASSIGNED) {
      throw new BadRequestException('Action non autorisee pour cette reservation');
    }

    const updated = await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.ACCEPTED,
        accepted_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Reservation confirmee',
      body: 'Votre prestataire a confirme la reservation',
      data: { type: 'handyman_booking_accepted', booking_id: String(bookingId) },
    });

    return updated;
  }

  /**
   * Provider starts journey to location
   */
  async providerEnRoute(bookingId: number, providerId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, provider_id: providerId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    const updated = await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.EN_ROUTE,
        en_route_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Prestataire en route',
      body: 'Votre prestataire est en chemin',
      data: { type: 'handyman_provider_en_route', booking_id: String(bookingId) },
    });

    return updated;
  }

  /**
   * Provider arrived
   */
  async providerArrived(bookingId: number, providerId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, provider_id: providerId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    const updated = await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.ARRIVED,
        arrived_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Prestataire arrive',
      body: 'Votre prestataire est arrive',
      data: { type: 'handyman_provider_arrived', booking_id: String(bookingId) },
    });

    return updated;
  }

  /**
   * Start service
   */
  async startService(bookingId: number, providerId: number): Promise<any> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, provider_id: providerId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    return this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.IN_PROGRESS,
        started_at: new Date(),
      },
    });
  }

  /**
   * Complete service
   */
  async completeService(
    bookingId: number,
    providerId: number,
    data: {
      finalPrice: number;
      workDuration?: number;
      completionNotes?: string;
      images?: string[];
    },
  ): Promise<any> {
    const booking = await this.prisma.handymanBooking.findFirst({
      where: { id: bookingId, provider_id: providerId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    if (booking.status !== BookingStatus.IN_PROGRESS) {
      throw new BadRequestException('Le service n\'est pas en cours');
    }

    const updated = await this.prisma.handymanBooking.update({
      where: { id: bookingId },
      data: {
        status: BookingStatus.COMPLETED,
        final_price: data.finalPrice,
        work_duration: data.workDuration,
        completion_notes: data.completionNotes,
        completion_images: data.images ? JSON.stringify(data.images) : null,
        completed_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Service termine',
      body: `Le service a ete complete. Montant: ${data.finalPrice} XOF`,
      data: { type: 'handyman_service_completed', booking_id: String(bookingId) },
    });

    return updated;
  }

  // ============================================================================
  // PROVIDER PROFILE
  // ============================================================================

  /**
   * Get provider bookings
   */
  async getProviderBookings(
    providerId: number,
    options?: { status?: BookingStatus; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { provider_id: providerId };
    if (options?.status) {
      where.status = options.status;
    }

    const [bookings, total] = await Promise.all([
      this.prisma.handymanBooking.findMany({
        where,
        include: {
          service: { select: { name: true } },
          user: {
            select: { first_name: true, last_name: true, mobile: true },
          },
        },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.handymanBooking.count({ where }),
    ]);

    return { data: bookings, total };
  }

  /**
   * Update provider location
   */
  async updateProviderLocation(
    providerId: number,
    latitude: number,
    longitude: number,
  ): Promise<void> {
    await this.prisma.handymanProvider.update({
      where: { id: providerId },
      data: {
        latitude,
        longitude,
        last_location_update: new Date(),
      },
    });
  }

  /**
   * Toggle provider online status
   */
  async toggleProviderOnlineStatus(
    providerId: number,
    isOnline: boolean,
  ): Promise<void> {
    await this.prisma.handymanProvider.update({
      where: { id: providerId },
      data: {
        is_online: isOnline ? 1 : 0,
        ...(isOnline
          ? { last_online_at: new Date() }
          : { last_offline_at: new Date() }),
      },
    });
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  /**
   * Calculate service estimate
   */
  private async calculateEstimate(
    serviceId: number,
    description?: string,
  ): Promise<number> {
    const service = await this.prisma.handymanService.findUnique({
      where: { id: serviceId },
    });

    if (!service) return 0;

    let basePrice = Number(service.base_price) || 5000;

    // Add complexity factor based on description length
    if (description && description.length > 200) {
      basePrice *= 1.2;
    }

    return Math.round(basePrice);
  }

  /**
   * Generate booking number
   */
  private generateBookingNumber(): string {
    const timestamp = Date.now().toString(36).toUpperCase();
    const random = Math.random().toString(36).substring(2, 6).toUpperCase();
    return `HM${timestamp}${random}`;
  }

  // ============================================================================
  // ADMIN FUNCTIONS
  // ============================================================================

  /**
   * Create service category
   */
  async createCategory(
    merchantId: number,
    data: {
      name: string;
      description?: string;
      icon?: string;
      sortOrder?: number;
    },
  ): Promise<any> {
    return this.prisma.handymanCategory.create({
      data: {
        merchant_id: merchantId,
        name: data.name,
        description: data.description,
        icon: data.icon,
        sort_order: data.sortOrder || 0,
        is_active: 1,
        created_at: new Date(),
      },
    });
  }

  /**
   * Create service
   */
  async createService(
    merchantId: number,
    data: {
      categoryId: number;
      name: string;
      description?: string;
      basePrice: number;
      icon?: string;
      estimatedDuration?: number;
    },
  ): Promise<any> {
    return this.prisma.handymanService.create({
      data: {
        merchant_id: merchantId,
        category_id: data.categoryId,
        name: data.name,
        description: data.description,
        base_price: data.basePrice,
        icon: data.icon,
        estimated_duration: data.estimatedDuration,
        is_active: 1,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get all providers (admin)
   */
  async getAllProviders(
    merchantId: number,
    options?: { isOnline?: boolean; categoryId?: number },
  ): Promise<any[]> {
    const where: any = { merchant_id: merchantId };

    if (options?.isOnline !== undefined) {
      where.is_online = options.isOnline ? 1 : 0;
    }

    if (options?.categoryId) {
      where.categories = { some: { category_id: options.categoryId } };
    }

    return this.prisma.handymanProvider.findMany({
      where,
      include: {
        categories: {
          include: { category: { select: { name: true } } },
        },
      },
      orderBy: { created_at: 'desc' },
    });
  }
}
