import { Injectable, Logger } from '@nestjs/common';
import { GeocodingProviderFactory } from './providers/provider.factory';
import {
  GeocodingResult,
  ReverseGeocodingResult,
  AutocompleteResult,
  PlaceDetailsResult,
  DirectionsResult,
  Coordinates,
} from './providers/geocoding.interface';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface GeocodingOptions {
  provider?: string;
  country?: string;
  cache?: boolean;
}

@Injectable()
export class GeocodingService {
  private readonly logger = new Logger(GeocodingService.name);
  private cache: Map<string, { data: any; expires: number }> = new Map();
  private readonly CACHE_TTL = 24 * 60 * 60 * 1000; // 24 hours

  constructor(
    private providerFactory: GeocodingProviderFactory,
    private prisma: PrismaService,
  ) {}

  /**
   * Geocode address to coordinates
   */
  async geocode(address: string, options?: GeocodingOptions): Promise<GeocodingResult> {
    const cacheKey = `geocode:${address}:${options?.country || ''}`;

    // Check cache
    if (options?.cache !== false) {
      const cached = this.getFromCache(cacheKey);
      if (cached) return cached;
    }

    const provider = this.providerFactory.getProvider(options?.provider);
    if (!provider) {
      return { success: false, error: 'No geocoding provider available' };
    }

    const result = await provider.geocode(address, { country: options?.country });

    // Cache successful results
    if (result.success && options?.cache !== false) {
      this.setCache(cacheKey, result);
    }

    return result;
  }

  /**
   * Reverse geocode coordinates to address
   */
  async reverseGeocode(
    lat: number,
    lng: number,
    options?: GeocodingOptions,
  ): Promise<ReverseGeocodingResult> {
    const cacheKey = `reverse:${lat.toFixed(6)}:${lng.toFixed(6)}`;

    // Check cache
    if (options?.cache !== false) {
      const cached = this.getFromCache(cacheKey);
      if (cached) return cached;
    }

    const provider = this.providerFactory.getProvider(options?.provider);
    if (!provider) {
      return { success: false, error: 'No geocoding provider available' };
    }

    const result = await provider.reverseGeocode(lat, lng);

    // Cache successful results
    if (result.success && options?.cache !== false) {
      this.setCache(cacheKey, result);
    }

    return result;
  }

  /**
   * Autocomplete address
   */
  async autocomplete(
    input: string,
    options?: {
      provider?: string;
      location?: Coordinates;
      radius?: number;
      types?: string[];
      country?: string;
    },
  ): Promise<AutocompleteResult> {
    const provider = this.providerFactory.getProvider(options?.provider);
    if (!provider || !provider.autocomplete) {
      return { success: false, error: 'Autocomplete not available' };
    }

    return provider.autocomplete(input, options);
  }

  /**
   * Get place details
   */
  async getPlaceDetails(
    placeId: string,
    provider?: string,
  ): Promise<PlaceDetailsResult> {
    const geocodingProvider = this.providerFactory.getProvider(provider);
    if (!geocodingProvider || !geocodingProvider.getPlaceDetails) {
      return { success: false, error: 'Place details not available' };
    }

    return geocodingProvider.getPlaceDetails(placeId);
  }

  /**
   * Get directions
   */
  async getDirections(
    origin: Coordinates,
    destination: Coordinates,
    options?: {
      provider?: string;
      waypoints?: Coordinates[];
      mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
      avoidTolls?: boolean;
      avoidHighways?: boolean;
      departureTime?: Date;
    },
  ): Promise<DirectionsResult> {
    const provider = this.providerFactory.getProvider(options?.provider);
    if (!provider || !provider.getDirections) {
      return { success: false, error: 'Directions not available' };
    }

    return provider.getDirections(origin, destination, options);
  }

  /**
   * Get ETA (Estimated Time of Arrival)
   */
  async getETA(
    origin: Coordinates,
    destination: Coordinates,
    options?: { provider?: string; mode?: 'driving' | 'walking' },
  ): Promise<{ success: boolean; duration?: number; distance?: number; error?: string }> {
    const directions = await this.getDirections(origin, destination, options);

    if (!directions.success || !directions.routes?.length) {
      return { success: false, error: directions.error || 'No routes found' };
    }

    const route = directions.routes[0];
    return {
      success: true,
      duration: route.durationInTraffic || route.duration,
      distance: route.distance,
    };
  }

  /**
   * Get nearby places
   */
  async getNearbyPlaces(
    location: Coordinates,
    type: string,
    radius: number = 1000,
    provider?: string,
  ): Promise<AutocompleteResult> {
    return this.autocomplete('', {
      provider,
      location,
      radius,
      types: [type],
    });
  }

  /**
   * Save favorite place
   */
  async saveFavoritePlace(
    userId: number,
    data: {
      name: string;
      address: string;
      lat: number;
      lng: number;
      type?: string;
      placeId?: string;
    },
  ): Promise<any> {
    return this.prisma.favoritePlace.create({
      data: {
        user_id: userId,
        name: data.name,
        address: data.address,
        lat: data.lat,
        lng: data.lng,
        type: data.type || 'other',
        place_id: data.placeId,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get user's favorite places
   */
  async getFavoritePlaces(userId: number): Promise<any[]> {
    return this.prisma.favoritePlace.findMany({
      where: { user_id: userId },
      orderBy: { created_at: 'desc' },
    });
  }

  /**
   * Delete favorite place
   */
  async deleteFavoritePlace(userId: number, placeId: number): Promise<boolean> {
    const result = await this.prisma.favoritePlace.deleteMany({
      where: { id: placeId, user_id: userId },
    });
    return result.count > 0;
  }

  /**
   * Get recent addresses
   */
  async getRecentAddresses(userId: number, limit: number = 10): Promise<any[]> {
    return this.prisma.booking.findMany({
      where: { user_id: userId },
      select: {
        pickup_address: true,
        pickup_lat: true,
        pickup_lng: true,
        dropoff_address: true,
        dropoff_lat: true,
        dropoff_lng: true,
        created_at: true,
      },
      orderBy: { created_at: 'desc' },
      take: limit,
    });
  }

  /**
   * Get available providers
   */
  getAvailableProviders() {
    return this.providerFactory.getProviderInfo();
  }

  // ============================================================================
  // CACHE METHODS
  // ============================================================================

  private getFromCache(key: string): any | null {
    const cached = this.cache.get(key);
    if (cached && cached.expires > Date.now()) {
      return cached.data;
    }
    this.cache.delete(key);
    return null;
  }

  private setCache(key: string, data: any): void {
    this.cache.set(key, {
      data,
      expires: Date.now() + this.CACHE_TTL,
    });

    // Cleanup old entries periodically
    if (this.cache.size > 10000) {
      const now = Date.now();
      for (const [k, v] of this.cache) {
        if (v.expires < now) this.cache.delete(k);
      }
    }
  }
}
