import { Injectable, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export type FleetStatus = 'active' | 'inactive' | 'suspended';

export interface Fleet {
  id: number;
  name: string;
  ownerId: number;
  ownerType: 'user' | 'corporate' | 'franchise';
  description?: string;
  logo?: string;
  status: FleetStatus;
  maxVehicles: number;
  maxDrivers: number;
  commissionRate: number;
  settings: FleetSettings;
  stats: FleetStats;
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface FleetSettings {
  autoApproveDrivers: boolean;
  autoApproveVehicles: boolean;
  requireInsurance: boolean;
  requireInspection: boolean;
  minDriverRating: number;
  maxDriverAge: number;
  allowedVehicleTypes: number[];
  operatingHours?: { start: string; end: string };
  operatingZones?: number[];
}

export interface FleetStats {
  totalDrivers: number;
  activeDrivers: number;
  totalVehicles: number;
  activeVehicles: number;
  totalBookings: number;
  completedBookings: number;
  totalEarnings: number;
  avgRating: number;
}

export interface FleetDriver {
  id: number;
  fleetId: number;
  driverId: number;
  role: 'driver' | 'supervisor' | 'manager';
  status: 'pending' | 'active' | 'inactive' | 'removed';
  joinedAt: Date;
  removedAt?: Date;
  commissionOverride?: number;
}

export interface FleetVehicle {
  id: number;
  fleetId: number;
  vehicleId: number;
  assignedDriverId?: number;
  status: 'available' | 'in_use' | 'maintenance' | 'removed';
  addedAt: Date;
  removedAt?: Date;
}

export interface CreateFleetDto {
  name: string;
  ownerId: number;
  ownerType: 'user' | 'corporate' | 'franchise';
  description?: string;
  logo?: string;
  maxVehicles?: number;
  maxDrivers?: number;
  commissionRate?: number;
  settings?: Partial<FleetSettings>;
}

export interface UpdateFleetDto extends Partial<CreateFleetDto> {
  status?: FleetStatus;
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class FleetService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // FLEET CRUD
  // ==========================================================================

  /**
   * Créer une flotte
   */
  async createFleet(merchantId: number, data: CreateFleetDto): Promise<Fleet> {
    const defaultSettings: FleetSettings = {
      autoApproveDrivers: false,
      autoApproveVehicles: false,
      requireInsurance: true,
      requireInspection: true,
      minDriverRating: 4.0,
      maxDriverAge: 65,
      allowedVehicleTypes: [],
    };

    const fleet = await this.prisma.fleet.create({
      data: {
        merchantId,
        name: data.name,
        ownerId: data.ownerId,
        ownerType: data.ownerType,
        description: data.description,
        logo: data.logo,
        status: 'active',
        maxVehicles: data.maxVehicles || 50,
        maxDrivers: data.maxDrivers || 50,
        commissionRate: data.commissionRate || 15,
        settings: JSON.stringify({ ...defaultSettings, ...data.settings }),
      },
    });

    return this.mapFleet(fleet);
  }

  /**
   * Obtenir une flotte
   */
  async getFleet(id: number, merchantId: number): Promise<Fleet | null> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id, merchantId },
    });

    if (!fleet) return null;

    const stats = await this.getFleetStats(id);
    return this.mapFleet(fleet, stats);
  }

  /**
   * Lister les flottes
   */
  async listFleets(
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      status?: FleetStatus;
      ownerId?: number;
      search?: string;
    } = {},
  ): Promise<{ fleets: Fleet[]; total: number; page: number; totalPages: number }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };
    if (options.status) where.status = options.status;
    if (options.ownerId) where.ownerId = options.ownerId;
    if (options.search) {
      where.OR = [
        { name: { contains: options.search } },
        { description: { contains: options.search } },
      ];
    }

    const [fleets, total] = await Promise.all([
      this.prisma.fleet.findMany({
        where,
        skip,
        take: limit,
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.fleet.count({ where }),
    ]);

    const fleetsWithStats = await Promise.all(
      fleets.map(async (f) => {
        const stats = await this.getFleetStats(f.id);
        return this.mapFleet(f, stats);
      }),
    );

    return {
      fleets: fleetsWithStats,
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Mettre à jour une flotte
   */
  async updateFleet(
    id: number,
    merchantId: number,
    data: UpdateFleetDto,
  ): Promise<Fleet | null> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id, merchantId },
    });

    if (!fleet) return null;

    const currentSettings = JSON.parse(fleet.settings || '{}');
    const newSettings = data.settings
      ? { ...currentSettings, ...data.settings }
      : currentSettings;

    const updated = await this.prisma.fleet.update({
      where: { id },
      data: {
        name: data.name,
        description: data.description,
        logo: data.logo,
        status: data.status,
        maxVehicles: data.maxVehicles,
        maxDrivers: data.maxDrivers,
        commissionRate: data.commissionRate,
        settings: JSON.stringify(newSettings),
      },
    });

    const stats = await this.getFleetStats(id);
    return this.mapFleet(updated, stats);
  }

  /**
   * Supprimer une flotte
   */
  async deleteFleet(id: number, merchantId: number): Promise<boolean> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id, merchantId },
    });

    if (!fleet) return false;

    // Vérifier si la flotte a des chauffeurs actifs
    const activeDrivers = await this.prisma.fleetDriver.count({
      where: { fleetId: id, status: 'active' },
    });

    if (activeDrivers > 0) {
      throw new BadRequestException(
        `Cette flotte a ${activeDrivers} chauffeur(s) actif(s). Retirez-les d'abord.`,
      );
    }

    await this.prisma.fleet.delete({ where: { id } });
    return true;
  }

  // ==========================================================================
  // FLEET DRIVERS
  // ==========================================================================

  /**
   * Ajouter un chauffeur à la flotte
   */
  async addDriver(
    fleetId: number,
    driverId: number,
    merchantId: number,
    role: 'driver' | 'supervisor' | 'manager' = 'driver',
  ): Promise<FleetDriver> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) {
      throw new NotFoundException('Flotte non trouvée');
    }

    // Vérifier le chauffeur
    const driver = await this.prisma.driver.findFirst({
      where: { id: driverId, merchantId },
    });

    if (!driver) {
      throw new NotFoundException('Chauffeur non trouvé');
    }

    // Vérifier si déjà dans une flotte
    const existing = await this.prisma.fleetDriver.findFirst({
      where: { driverId, status: 'active' },
    });

    if (existing) {
      throw new BadRequestException('Ce chauffeur appartient déjà à une flotte');
    }

    // Vérifier la limite
    const currentDrivers = await this.prisma.fleetDriver.count({
      where: { fleetId, status: 'active' },
    });

    if (currentDrivers >= fleet.maxDrivers) {
      throw new BadRequestException('Limite de chauffeurs atteinte pour cette flotte');
    }

    const settings = JSON.parse(fleet.settings || '{}');
    const status = settings.autoApproveDrivers ? 'active' : 'pending';

    const fleetDriver = await this.prisma.fleetDriver.create({
      data: {
        fleetId,
        driverId,
        role,
        status,
        joinedAt: new Date(),
      },
    });

    return this.mapFleetDriver(fleetDriver);
  }

  /**
   * Retirer un chauffeur de la flotte
   */
  async removeDriver(
    fleetId: number,
    driverId: number,
    merchantId: number,
  ): Promise<boolean> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return false;

    const fleetDriver = await this.prisma.fleetDriver.findFirst({
      where: { fleetId, driverId, status: { not: 'removed' } },
    });

    if (!fleetDriver) return false;

    await this.prisma.fleetDriver.update({
      where: { id: fleetDriver.id },
      data: { status: 'removed', removedAt: new Date() },
    });

    return true;
  }

  /**
   * Lister les chauffeurs d'une flotte
   */
  async getFleetDrivers(
    fleetId: number,
    merchantId: number,
    status?: 'pending' | 'active' | 'inactive',
  ): Promise<FleetDriver[]> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return [];

    const where: any = { fleetId, status: { not: 'removed' } };
    if (status) where.status = status;

    const drivers = await this.prisma.fleetDriver.findMany({
      where,
      include: {
        driver: {
          select: {
            id: true,
            firstName: true,
            lastName: true,
            phone: true,
            email: true,
            profilePicture: true,
            rating: true,
            status: true,
          },
        },
      },
      orderBy: { joinedAt: 'desc' },
    });

    return drivers.map(this.mapFleetDriver);
  }

  /**
   * Approuver un chauffeur
   */
  async approveDriver(
    fleetId: number,
    driverId: number,
    merchantId: number,
  ): Promise<FleetDriver | null> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return null;

    const fleetDriver = await this.prisma.fleetDriver.findFirst({
      where: { fleetId, driverId, status: 'pending' },
    });

    if (!fleetDriver) return null;

    const updated = await this.prisma.fleetDriver.update({
      where: { id: fleetDriver.id },
      data: { status: 'active' },
    });

    return this.mapFleetDriver(updated);
  }

  /**
   * Mettre à jour le rôle d'un chauffeur
   */
  async updateDriverRole(
    fleetId: number,
    driverId: number,
    merchantId: number,
    role: 'driver' | 'supervisor' | 'manager',
  ): Promise<FleetDriver | null> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return null;

    const fleetDriver = await this.prisma.fleetDriver.findFirst({
      where: { fleetId, driverId, status: 'active' },
    });

    if (!fleetDriver) return null;

    const updated = await this.prisma.fleetDriver.update({
      where: { id: fleetDriver.id },
      data: { role },
    });

    return this.mapFleetDriver(updated);
  }

  // ==========================================================================
  // FLEET VEHICLES
  // ==========================================================================

  /**
   * Ajouter un véhicule à la flotte
   */
  async addVehicle(
    fleetId: number,
    vehicleId: number,
    merchantId: number,
  ): Promise<FleetVehicle> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) {
      throw new NotFoundException('Flotte non trouvée');
    }

    // Vérifier le véhicule
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id: vehicleId, merchantId },
    });

    if (!vehicle) {
      throw new NotFoundException('Véhicule non trouvé');
    }

    // Vérifier si déjà dans une flotte
    const existing = await this.prisma.fleetVehicle.findFirst({
      where: { vehicleId, status: { not: 'removed' } },
    });

    if (existing) {
      throw new BadRequestException('Ce véhicule appartient déjà à une flotte');
    }

    // Vérifier la limite
    const currentVehicles = await this.prisma.fleetVehicle.count({
      where: { fleetId, status: { not: 'removed' } },
    });

    if (currentVehicles >= fleet.maxVehicles) {
      throw new BadRequestException('Limite de véhicules atteinte pour cette flotte');
    }

    const fleetVehicle = await this.prisma.fleetVehicle.create({
      data: {
        fleetId,
        vehicleId,
        status: 'available',
        addedAt: new Date(),
      },
    });

    return this.mapFleetVehicle(fleetVehicle);
  }

  /**
   * Retirer un véhicule de la flotte
   */
  async removeVehicle(
    fleetId: number,
    vehicleId: number,
    merchantId: number,
  ): Promise<boolean> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return false;

    const fleetVehicle = await this.prisma.fleetVehicle.findFirst({
      where: { fleetId, vehicleId, status: { not: 'removed' } },
    });

    if (!fleetVehicle) return false;

    await this.prisma.fleetVehicle.update({
      where: { id: fleetVehicle.id },
      data: { status: 'removed', removedAt: new Date() },
    });

    return true;
  }

  /**
   * Lister les véhicules d'une flotte
   */
  async getFleetVehicles(
    fleetId: number,
    merchantId: number,
    status?: 'available' | 'in_use' | 'maintenance',
  ): Promise<FleetVehicle[]> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) return [];

    const where: any = { fleetId, status: { not: 'removed' } };
    if (status) where.status = status;

    const vehicles = await this.prisma.fleetVehicle.findMany({
      where,
      include: {
        vehicle: {
          select: {
            id: true,
            make: true,
            model: true,
            year: true,
            plateNumber: true,
            color: true,
            status: true,
          },
        },
        assignedDriver: {
          select: {
            id: true,
            firstName: true,
            lastName: true,
          },
        },
      },
      orderBy: { addedAt: 'desc' },
    });

    return vehicles.map(this.mapFleetVehicle);
  }

  /**
   * Assigner un véhicule à un chauffeur
   */
  async assignVehicleToDriver(
    fleetId: number,
    vehicleId: number,
    driverId: number,
    merchantId: number,
  ): Promise<FleetVehicle | null> {
    // Vérifier que le chauffeur est dans la flotte
    const fleetDriver = await this.prisma.fleetDriver.findFirst({
      where: { fleetId, driverId, status: 'active' },
    });

    if (!fleetDriver) {
      throw new BadRequestException('Ce chauffeur n\'appartient pas à cette flotte');
    }

    // Vérifier le véhicule
    const fleetVehicle = await this.prisma.fleetVehicle.findFirst({
      where: { fleetId, vehicleId, status: 'available' },
    });

    if (!fleetVehicle) {
      throw new BadRequestException('Véhicule non disponible');
    }

    const updated = await this.prisma.fleetVehicle.update({
      where: { id: fleetVehicle.id },
      data: { assignedDriverId: driverId, status: 'in_use' },
    });

    return this.mapFleetVehicle(updated);
  }

  /**
   * Libérer un véhicule
   */
  async releaseVehicle(
    fleetId: number,
    vehicleId: number,
    merchantId: number,
  ): Promise<FleetVehicle | null> {
    const fleetVehicle = await this.prisma.fleetVehicle.findFirst({
      where: { fleetId, vehicleId, status: 'in_use' },
    });

    if (!fleetVehicle) return null;

    const updated = await this.prisma.fleetVehicle.update({
      where: { id: fleetVehicle.id },
      data: { assignedDriverId: null, status: 'available' },
    });

    return this.mapFleetVehicle(updated);
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Obtenir les statistiques d'une flotte
   */
  async getFleetStats(fleetId: number): Promise<FleetStats> {
    const [
      totalDrivers,
      activeDrivers,
      totalVehicles,
      activeVehicles,
      bookingStats,
    ] = await Promise.all([
      this.prisma.fleetDriver.count({
        where: { fleetId, status: { not: 'removed' } },
      }),
      this.prisma.fleetDriver.count({
        where: { fleetId, status: 'active' },
      }),
      this.prisma.fleetVehicle.count({
        where: { fleetId, status: { not: 'removed' } },
      }),
      this.prisma.fleetVehicle.count({
        where: { fleetId, status: { in: ['available', 'in_use'] } },
      }),
      this.prisma.$queryRaw<any[]>`
        SELECT
          COUNT(*) as totalBookings,
          SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completedBookings,
          SUM(CASE WHEN b.status = 'completed' THEN b.total_fare ELSE 0 END) as totalEarnings,
          AVG(CASE WHEN b.rating IS NOT NULL THEN b.rating ELSE NULL END) as avgRating
        FROM bookings b
        INNER JOIN fleet_drivers fd ON fd.driver_id = b.driver_id
        WHERE fd.fleet_id = ${fleetId}
          AND fd.status = 'active'
      `,
    ]);

    const stats = bookingStats[0] || {};

    return {
      totalDrivers,
      activeDrivers,
      totalVehicles,
      activeVehicles,
      totalBookings: parseInt(stats.totalBookings || '0'),
      completedBookings: parseInt(stats.completedBookings || '0'),
      totalEarnings: parseFloat(stats.totalEarnings || '0'),
      avgRating: parseFloat(stats.avgRating || '0'),
    };
  }

  /**
   * Rapport de performance de la flotte
   */
  async getFleetPerformance(
    fleetId: number,
    merchantId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<{
    daily: any[];
    byDriver: any[];
    summary: Record<string, any>;
  }> {
    const fleet = await this.prisma.fleet.findFirst({
      where: { id: fleetId, merchantId },
    });

    if (!fleet) {
      throw new NotFoundException('Flotte non trouvée');
    }

    // Données journalières
    const daily = await this.prisma.$queryRaw<any[]>`
      SELECT
        DATE(b.created_at) as date,
        COUNT(*) as bookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(b.total_fare) as revenue
      FROM bookings b
      INNER JOIN fleet_drivers fd ON fd.driver_id = b.driver_id
      WHERE fd.fleet_id = ${fleetId}
        AND fd.status = 'active'
        AND b.created_at >= ${startDate}
        AND b.created_at <= ${endDate}
      GROUP BY DATE(b.created_at)
      ORDER BY date ASC
    `;

    // Par chauffeur
    const byDriver = await this.prisma.$queryRaw<any[]>`
      SELECT
        d.id,
        d.first_name as firstName,
        d.last_name as lastName,
        COUNT(b.id) as bookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(b.total_fare) as earnings,
        AVG(b.rating) as avgRating
      FROM drivers d
      INNER JOIN fleet_drivers fd ON fd.driver_id = d.id
      LEFT JOIN bookings b ON b.driver_id = d.id
        AND b.created_at >= ${startDate}
        AND b.created_at <= ${endDate}
      WHERE fd.fleet_id = ${fleetId}
        AND fd.status = 'active'
      GROUP BY d.id, d.first_name, d.last_name
      ORDER BY earnings DESC
    `;

    // Résumé
    const totals = await this.prisma.$queryRaw<any[]>`
      SELECT
        COUNT(*) as totalBookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completedBookings,
        SUM(b.total_fare) as totalRevenue,
        AVG(b.rating) as avgRating
      FROM bookings b
      INNER JOIN fleet_drivers fd ON fd.driver_id = b.driver_id
      WHERE fd.fleet_id = ${fleetId}
        AND fd.status = 'active'
        AND b.created_at >= ${startDate}
        AND b.created_at <= ${endDate}
    `;

    const summary = totals[0] || {};

    return {
      daily,
      byDriver,
      summary: {
        totalBookings: parseInt(summary.totalBookings || '0'),
        completedBookings: parseInt(summary.completedBookings || '0'),
        totalRevenue: parseFloat(summary.totalRevenue || '0'),
        avgRating: parseFloat(summary.avgRating || '0'),
        completionRate: summary.totalBookings > 0
          ? (summary.completedBookings / summary.totalBookings) * 100
          : 0,
      },
    };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  private mapFleet(fleet: any, stats?: FleetStats): Fleet {
    return {
      id: fleet.id,
      name: fleet.name,
      ownerId: fleet.ownerId,
      ownerType: fleet.ownerType,
      description: fleet.description,
      logo: fleet.logo,
      status: fleet.status,
      maxVehicles: fleet.maxVehicles,
      maxDrivers: fleet.maxDrivers,
      commissionRate: parseFloat(fleet.commissionRate || '0'),
      settings: JSON.parse(fleet.settings || '{}'),
      stats: stats || {
        totalDrivers: 0,
        activeDrivers: 0,
        totalVehicles: 0,
        activeVehicles: 0,
        totalBookings: 0,
        completedBookings: 0,
        totalEarnings: 0,
        avgRating: 0,
      },
      merchantId: fleet.merchantId,
      createdAt: fleet.createdAt,
      updatedAt: fleet.updatedAt,
    };
  }

  private mapFleetDriver(fd: any): FleetDriver {
    return {
      id: fd.id,
      fleetId: fd.fleetId,
      driverId: fd.driverId,
      role: fd.role,
      status: fd.status,
      joinedAt: fd.joinedAt,
      removedAt: fd.removedAt,
      commissionOverride: fd.commissionOverride
        ? parseFloat(fd.commissionOverride)
        : undefined,
      ...(fd.driver && { driver: fd.driver }),
    };
  }

  private mapFleetVehicle(fv: any): FleetVehicle {
    return {
      id: fv.id,
      fleetId: fv.fleetId,
      vehicleId: fv.vehicleId,
      assignedDriverId: fv.assignedDriverId,
      status: fv.status,
      addedAt: fv.addedAt,
      removedAt: fv.removedAt,
      ...(fv.vehicle && { vehicle: fv.vehicle }),
      ...(fv.assignedDriver && { assignedDriver: fv.assignedDriver }),
    };
  }
}
