import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as nodemailer from 'nodemailer';
import { EmailTemplateService } from './template.service';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface EmailOptions {
  to: string | string[];
  subject: string;
  html?: string;
  text?: string;
  template?: string;
  data?: Record<string, any>;
  attachments?: Array<{
    filename: string;
    content?: Buffer | string;
    path?: string;
    contentType?: string;
  }>;
  cc?: string | string[];
  bcc?: string | string[];
  replyTo?: string;
  priority?: 'high' | 'normal' | 'low';
}

export interface EmailResult {
  success: boolean;
  messageId?: string;
  error?: string;
}

@Injectable()
export class EmailService {
  private readonly logger = new Logger(EmailService.name);
  private transporter: nodemailer.Transporter;
  private fromEmail: string;
  private fromName: string;

  constructor(
    private configService: ConfigService,
    private templateService: EmailTemplateService,
    private prisma: PrismaService,
  ) {
    this.fromEmail = this.configService.get<string>('MAIL_FROM_ADDRESS') || 'noreply@monkapi.com';
    this.fromName = this.configService.get<string>('MAIL_FROM_NAME') || 'MonkAPI';

    this.initializeTransporter();
  }

  private initializeTransporter() {
    const driver = this.configService.get<string>('MAIL_DRIVER') || 'smtp';

    if (driver === 'ses') {
      // AWS SES
      const AWS = require('@aws-sdk/client-ses');
      this.transporter = nodemailer.createTransport({
        SES: {
          ses: new AWS.SES({
            region: this.configService.get<string>('AWS_REGION'),
            credentials: {
              accessKeyId: this.configService.get<string>('AWS_ACCESS_KEY_ID'),
              secretAccessKey: this.configService.get<string>('AWS_SECRET_ACCESS_KEY'),
            },
          }),
          aws: AWS,
        },
      });
    } else if (driver === 'sendgrid') {
      // SendGrid
      this.transporter = nodemailer.createTransport({
        host: 'smtp.sendgrid.net',
        port: 587,
        auth: {
          user: 'apikey',
          pass: this.configService.get<string>('SENDGRID_API_KEY'),
        },
      });
    } else if (driver === 'mailgun') {
      // Mailgun
      this.transporter = nodemailer.createTransport({
        host: 'smtp.mailgun.org',
        port: 587,
        auth: {
          user: this.configService.get<string>('MAILGUN_USERNAME'),
          pass: this.configService.get<string>('MAILGUN_PASSWORD'),
        },
      });
    } else {
      // Default SMTP
      this.transporter = nodemailer.createTransport({
        host: this.configService.get<string>('MAIL_HOST') || 'smtp.mailtrap.io',
        port: this.configService.get<number>('MAIL_PORT') || 587,
        secure: this.configService.get<boolean>('MAIL_SECURE') || false,
        auth: {
          user: this.configService.get<string>('MAIL_USERNAME'),
          pass: this.configService.get<string>('MAIL_PASSWORD'),
        },
      });
    }

    this.logger.log(`Email transporter initialized with driver: ${driver}`);
  }

  /**
   * Send email
   */
  async send(options: EmailOptions): Promise<EmailResult> {
    try {
      let html = options.html;
      let text = options.text;

      // Render template if specified
      if (options.template && options.data) {
        html = await this.templateService.render(options.template, options.data);
        text = await this.templateService.renderText(options.template, options.data);
      }

      const mailOptions: nodemailer.SendMailOptions = {
        from: `${this.fromName} <${this.fromEmail}>`,
        to: Array.isArray(options.to) ? options.to.join(', ') : options.to,
        subject: options.subject,
        html,
        text,
        attachments: options.attachments,
        cc: options.cc,
        bcc: options.bcc,
        replyTo: options.replyTo,
        priority: options.priority,
      };

      const result = await this.transporter.sendMail(mailOptions);

      // Log email
      await this.logEmail({
        to: options.to,
        subject: options.subject,
        template: options.template,
        status: 'sent',
        messageId: result.messageId,
      });

      this.logger.log(`Email sent to ${options.to}: ${result.messageId}`);

      return {
        success: true,
        messageId: result.messageId,
      };
    } catch (error) {
      this.logger.error(`Failed to send email: ${error.message}`);

      // Log failed email
      await this.logEmail({
        to: options.to,
        subject: options.subject,
        template: options.template,
        status: 'failed',
        error: error.message,
      });

      return {
        success: false,
        error: error.message,
      };
    }
  }

  /**
   * Send bulk emails
   */
  async sendBulk(
    recipients: Array<{ email: string; data?: Record<string, any> }>,
    options: Omit<EmailOptions, 'to'>,
  ): Promise<{ success: number; failed: number; results: EmailResult[] }> {
    const results: EmailResult[] = [];
    let success = 0;
    let failed = 0;

    for (const recipient of recipients) {
      const result = await this.send({
        ...options,
        to: recipient.email,
        data: { ...options.data, ...recipient.data },
      });

      results.push(result);
      if (result.success) success++;
      else failed++;
    }

    return { success, failed, results };
  }

  /**
   * Log email to database
   */
  private async logEmail(data: {
    to: string | string[];
    subject: string;
    template?: string;
    status: string;
    messageId?: string;
    error?: string;
  }): Promise<void> {
    try {
      await this.prisma.emailLog.create({
        data: {
          recipient: Array.isArray(data.to) ? data.to.join(', ') : data.to,
          subject: data.subject,
          template: data.template,
          status: data.status,
          message_id: data.messageId,
          error: data.error,
          created_at: new Date(),
        },
      });
    } catch (error) {
      this.logger.error(`Failed to log email: ${error.message}`);
    }
  }

  // ============================================================================
  // PREDEFINED EMAIL METHODS
  // ============================================================================

  /**
   * Send welcome email
   */
  async sendWelcomeEmail(
    to: string,
    data: { name: string; userType: 'user' | 'driver' },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Bienvenue sur ${this.fromName}!`,
      template: 'welcome',
      data: {
        name: data.name,
        userType: data.userType,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send OTP email
   */
  async sendOtpEmail(
    to: string,
    data: { name: string; otp: string; expiresIn: number },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Votre code de verification ${this.fromName}`,
      template: 'otp',
      data: {
        name: data.name,
        otp: data.otp,
        expiresIn: data.expiresIn,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send password reset email
   */
  async sendPasswordResetEmail(
    to: string,
    data: { name: string; resetLink: string; expiresIn: number },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Reinitialisation de votre mot de passe ${this.fromName}`,
      template: 'password-reset',
      data: {
        name: data.name,
        resetLink: data.resetLink,
        expiresIn: data.expiresIn,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send booking confirmation email
   */
  async sendBookingConfirmationEmail(
    to: string,
    data: {
      name: string;
      bookingNumber: string;
      pickupAddress: string;
      dropoffAddress: string;
      scheduledTime?: Date;
      estimatedFare: number;
      currency: string;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Confirmation de votre course #${data.bookingNumber}`,
      template: 'booking-confirmation',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send booking completed email with invoice
   */
  async sendBookingCompletedEmail(
    to: string,
    data: {
      name: string;
      bookingNumber: string;
      pickupAddress: string;
      dropoffAddress: string;
      distance: number;
      duration: number;
      fare: number;
      currency: string;
      paymentMethod: string;
      driverName: string;
      vehicleInfo: string;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Facture de votre course #${data.bookingNumber}`,
      template: 'booking-completed',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send driver approved email
   */
  async sendDriverApprovedEmail(
    to: string,
    data: { name: string },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Felicitations! Votre compte chauffeur est approuve`,
      template: 'driver-approved',
      data: {
        name: data.name,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send driver rejected email
   */
  async sendDriverRejectedEmail(
    to: string,
    data: { name: string; reason: string },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Mise a jour de votre demande d'inscription chauffeur`,
      template: 'driver-rejected',
      data: {
        name: data.name,
        reason: data.reason,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send payment confirmation email
   */
  async sendPaymentConfirmationEmail(
    to: string,
    data: {
      name: string;
      amount: number;
      currency: string;
      reference: string;
      paymentMethod: string;
      description: string;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Confirmation de paiement - ${data.reference}`,
      template: 'payment-confirmation',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send withdrawal processed email
   */
  async sendWithdrawalProcessedEmail(
    to: string,
    data: {
      name: string;
      amount: number;
      currency: string;
      method: string;
      reference: string;
      status: 'completed' | 'failed';
      failureReason?: string;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Votre demande de retrait a ete ${data.status === 'completed' ? 'traitee' : 'rejetee'}`,
      template: 'withdrawal-processed',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send weekly earnings report to driver
   */
  async sendWeeklyEarningsEmail(
    to: string,
    data: {
      name: string;
      weekStart: Date;
      weekEnd: Date;
      totalTrips: number;
      totalEarnings: number;
      totalTips: number;
      currency: string;
      topDay: string;
      averageRating: number;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Votre resume hebdomadaire - ${data.totalTrips} courses`,
      template: 'weekly-earnings',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send referral bonus email
   */
  async sendReferralBonusEmail(
    to: string,
    data: {
      name: string;
      referredName: string;
      bonusAmount: number;
      currency: string;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Vous avez gagne ${data.bonusAmount} ${data.currency} grace a votre parrainage!`,
      template: 'referral-bonus',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send document expiry reminder
   */
  async sendDocumentExpiryEmail(
    to: string,
    data: {
      name: string;
      documentType: string;
      expiryDate: Date;
      daysRemaining: number;
    },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Rappel: Votre ${data.documentType} expire bientot`,
      template: 'document-expiry',
      data: {
        ...data,
        appName: this.fromName,
      },
    });
  }

  /**
   * Send corporate invoice email
   */
  async sendCorporateInvoiceEmail(
    to: string,
    data: {
      companyName: string;
      invoiceNumber: string;
      period: string;
      totalTrips: number;
      totalAmount: number;
      currency: string;
      dueDate: Date;
    },
    attachment?: { filename: string; content: Buffer },
  ): Promise<EmailResult> {
    return this.send({
      to,
      subject: `Facture ${data.invoiceNumber} - ${data.period}`,
      template: 'corporate-invoice',
      data: {
        ...data,
        appName: this.fromName,
      },
      attachments: attachment ? [attachment] : undefined,
    });
  }
}
