import { Injectable, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../shared/database/prisma.service';
import { UpdateDriverProfileDto } from './dto/update-profile.dto';
import { UpdateLocationDto } from './dto/update-location.dto';
import { UpdateStatusDto } from './dto/update-status.dto';

@Injectable()
export class DriverService {
  constructor(private prisma: PrismaService) {}

  async getProfile(driverId: number) {
    const driver = await this.prisma.driver.findUnique({
      where: { id: driverId },
      include: {
        vehicles: true,
      },
    });

    if (!driver) {
      throw new NotFoundException('Driver not found');
    }

    const { password, otp, otp_expires_at, ...profile } = driver;

    return {
      message: 'Driver profile',
      data: profile,
    };
  }

  async updateProfile(driverId: number, dto: UpdateDriverProfileDto) {
    const driver = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        first_name: dto.first_name,
        last_name: dto.last_name,
        email: dto.email,
        profile_image: dto.profile_image,
      },
    });

    const { password, otp, otp_expires_at, ...profile } = driver;

    return {
      message: 'Profile updated successfully',
      data: profile,
    };
  }

  async updateLocation(driverId: number, dto: UpdateLocationDto) {
    await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        latitude: dto.latitude,
        longitude: dto.longitude,
      },
    });

    // TODO: Broadcast location update via WebSocket
    // await this.websocketGateway.broadcastDriverLocation(driverId, dto);

    return {
      message: 'Location updated',
    };
  }

  async updateStatus(driverId: number, dto: UpdateStatusDto) {
    const driver = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_online: dto.is_online,
      },
    });

    return {
      message: dto.is_online === 1 ? 'You are now online' : 'You are now offline',
      data: {
        is_online: driver.is_online,
      },
    };
  }

  async getBookings(driverId: number, page: number, limit: number) {
    const skip = (page - 1) * limit;

    const [bookings, total] = await Promise.all([
      this.prisma.booking.findMany({
        where: { driver_id: driverId },
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
        include: {
          user: {
            select: {
              id: true,
              first_name: true,
              last_name: true,
              UserPhone: true,
              profile_image: true,
            },
          },
        },
      }),
      this.prisma.booking.count({ where: { driver_id: driverId } }),
    ]);

    return {
      message: 'Driver bookings',
      data: {
        bookings,
        pagination: {
          page,
          limit,
          total,
          total_pages: Math.ceil(total / limit),
        },
      },
    };
  }

  async getEarnings(driverId: number, startDate?: string, endDate?: string) {
    const where: any = {
      driver_id: driverId,
      booking_status: 'completed',
    };

    if (startDate) {
      where.completed_time = { gte: new Date(startDate) };
    }
    if (endDate) {
      where.completed_time = { ...where.completed_time, lte: new Date(endDate) };
    }

    const earnings = await this.prisma.booking.aggregate({
      where,
      _sum: {
        final_amount: true,
      },
      _count: true,
    });

    const todayEarnings = await this.prisma.booking.aggregate({
      where: {
        driver_id: driverId,
        booking_status: 'completed',
        completed_time: {
          gte: new Date(new Date().setHours(0, 0, 0, 0)),
        },
      },
      _sum: {
        final_amount: true,
      },
      _count: true,
    });

    return {
      message: 'Driver earnings',
      data: {
        total_earnings: earnings._sum.final_amount || 0,
        total_rides: earnings._count || 0,
        today_earnings: todayEarnings._sum.final_amount || 0,
        today_rides: todayEarnings._count || 0,
      },
    };
  }

  async getWallet(driverId: number, page: number, limit: number) {
    const skip = (page - 1) * limit;

    const balanceResult = await this.prisma.driverWalletTransaction.aggregate({
      where: { driver_id: driverId },
      _sum: {
        amount: true,
      },
    });

    const balance = balanceResult._sum.amount || 0;

    const [transactions, total] = await Promise.all([
      this.prisma.driverWalletTransaction.findMany({
        where: { driver_id: driverId },
        orderBy: { created_at: 'desc' },
        skip,
        take: limit,
      }),
      this.prisma.driverWalletTransaction.count({ where: { driver_id: driverId } }),
    ]);

    return {
      message: 'Driver wallet',
      data: {
        balance,
        transactions,
        pagination: {
          page,
          limit,
          total,
          total_pages: Math.ceil(total / limit),
        },
      },
    };
  }

  async getVehicles(driverId: number) {
    const vehicles = await this.prisma.driverVehicle.findMany({
      where: { driver_id: driverId },
      orderBy: { is_default: 'desc' },
    });

    return {
      message: 'Driver vehicles',
      data: vehicles,
    };
  }
}
